import React, { useEffect, useState } from "react";
import { Map as YMap, Placemark, withYMaps, ZoomControl } from "react-yandex-maps";

import "./Map.scss";

import skeleton from "./skeleton.png";

export const Map = withYMaps(({ ymaps, points }) => {
    const [placemarks, setPlacemarks] = useState([]);

    if (!ymaps) {
        return null;
    }

    const bounds = getMapBounds(points);
    const iconLayout = getIconLayout(ymaps);
    const hintLayout = getHintLayout(ymaps);

    useEffect(() => {
        const addresses = Promise.all(
            points.map(e => ymaps
                .geocode(e.position, { results: 1 })
                .then(e => e.geoObjects.get(0).properties.get("name")),
            ),
        );

        addresses.then(addresses => {
            setPlacemarks(
                points.map((e, i) => ({
                    geometry: e.position,
                    options: {
                        iconLayout,
                        iconShape: {
                            type: "Circle",
                            coordinates: [0, -e.size / 46 * 33], // size / svgWidth * svgHeight
                            radius: e.size / 2,
                        },
                        hintLayout,
                        hintPane: "hint",
                    },
                    properties: {
                        color: e.color,
                        size: e.size,
                        image: e.imageHref,
                        address: addresses[i],
                    },
                })),
            );
        });
    }, [ymaps, points]);

    return (
        <YMap
            className="Map"
            state={{
                bounds,
                behaviors: [
                    "drag",
                    "dblClickZoom",
                    "multiTouch",
                    "rightMouseButtonMagnifier",
                    "rightMouseButtonMagnifier",
                ],
            }}
            modules={[
                "geoObject.addon.hint",
                "shape.Rectangle",
                "geometry.pixel.Rectangle",
            ]}
            options={{
                yandexMapDisablePoiInteractivity: true,
            }}
        >
            {placemarks.map((placemark, i) => <Placemark key={i} {...placemark} />)}
            <ZoomControl/>
        </YMap>
    );
}, true, ["templateLayoutFactory", "geocode"]);

export function MapSkeleton() {
    return (
        <img className="Map Map_skeleton" alt="" src={skeleton}/>
    );
}

function getIconLayout(ymaps) {
    return ymaps.templateLayoutFactory.createClass(`
        <div class="Map-Point">
            <svg width="{{ properties.size }}" viewBox="0 0 46 59" fill="none" xmlns="http://www.w3.org/2000/svg">
                <path d="M28.3423 45.3763C25.2659 46.3058 23.6553 48.3546 23.5107 51.5229C23.4985 51.7898 23.2785 52
                    23.0112 52L22.9887 51.9999C22.7215 51.9999 22.5015 51.7897 22.4893 51.5228C22.3446 48.3546 20.7341
                    46.3058 17.6577 45.3763C7.53142 42.9677 0 33.8634 0 23C0 10.2975 10.2975 0 23 0C35.7025 0 46
                    10.2975 46 23C46 33.8634 38.4686 42.9677 28.3423 45.3763Z" fill="{{ properties.color }}"/>
                <circle cx="23" cy="56" r="2" fill="white" stroke="{{ properties.color }}" stroke-width="2"/>
                <image xlink:href="{{ properties.image }}" height="26" width="26" x="10" y="10"/>
            </svg>
        </div>
    `);
}

function getHintLayout(ymaps) {
    return ymaps.templateLayoutFactory.createClass(
        `<div class='Map-Hint'>{{ properties.address }}</div>`,
        {
            getShape: function () {
                const el = this.getElement();
                let result = null;
                if (el) {
                    const firstChild = el.firstChild;
                    const margin = 8;

                    result = new ymaps.shape.Rectangle(
                        new ymaps.geometry.pixel.Rectangle([
                            [0, 0],
                            [firstChild.offsetWidth + margin, firstChild.offsetHeight + margin],
                        ]),
                    );
                }
                return result;
            },
        },
    );
}

function getMapBounds(points) {
    const xs = points.map(e => e.position[1]);
    const ys = points.map(e => e.position[0]);

    if (points.length === 1) {
        const delta = 0.01;

        return [
            [ys[0] - delta, xs[0] - delta], // leftBottom
            [ys[0] + delta, xs[0] + delta], // rightTop
        ];
    }

    const minX = Math.min(...xs);
    const maxX = Math.max(...xs);
    const minY = Math.min(...ys);
    const maxY = Math.max(...ys);

    const deltaX = maxX - minX;
    const deltaY = maxY - minY;
    const extraSpace = 0.2;

    return [
        [minY - deltaY * extraSpace, minX - deltaX * extraSpace], // leftBottom
        [maxY + deltaY * extraSpace, maxX + deltaX * extraSpace],  // rightTop
    ];
}
