import React, { useEffect, useMemo, useState } from "react";
import { useSelector } from "react-redux";
import { getPublicGraph, getPublicGraphLoading } from "../../store/selectors";
import { Graph, GraphSkeleton } from "../../components/Graph/Graph";
import { Section } from "../../components/Section/Section";
import { getServiceIcon } from "../../icons/services";
import { getAppIcon } from "../../icons/apps";

import "./GraphSection.scss";

import noData from "./no-data.svg";

const IMAGE_SIZE_XS = 12;
const IMAGE_SIZE_S = 36;
const IMAGE_SIZE_M = 56;
const IMAGE_SIZE_L = 80;

const NODE_MAPPING = {
    email: {
        imageSize: IMAGE_SIZE_M,
        imageHref: "mail",
    },
    yandexuid: {
        imageSize: IMAGE_SIZE_XS,
        imageHref: "yandexuid",
    },
    idfa: {
        imageSize: IMAGE_SIZE_L,
        imageHref: "ios",
    },
    gaid: {
        imageSize: IMAGE_SIZE_L,
        imageHref: "android",
    },
    oaid: {
        imageSize: IMAGE_SIZE_L,
        imageHref: "android",
    },
    login: {
        imageSize: IMAGE_SIZE_M,
        imageHref: "key",
    },
    puid: {
        imageSize: IMAGE_SIZE_M,
        imageHref: "key",
    },
    instagram_login: {
        imageSize: IMAGE_SIZE_M,
        imageHref: "instagram"
    },
    instagram_id: {
        imageSize: IMAGE_SIZE_M,
        imageHref: "instagram"
    },
    fb_id: {
        imageSize: IMAGE_SIZE_M,
        imageHref: "facebook"
    },
    ok_id: {
        imageSize: IMAGE_SIZE_M,
        imageHref: "ok"
    },
    vk_id: {
        imageSize: IMAGE_SIZE_M,
        imageHref: "vk"
    },
    vk_name: {
        imageSize: IMAGE_SIZE_M,
        imageHref: "vk"
    },
    kp_id: {
        imageSize: IMAGE_SIZE_M,
        imageHref: "kinopoisk"
    },
}

function getNodeMapping(item) {
    if (item.idType === 'uuid') {
        // App
        return { imageHref: item.icon, imageSize: IMAGE_SIZE_S };
    }

    return NODE_MAPPING[item.icon] ?? { imageHref: "default", imageSize: IMAGE_SIZE_XS };
}

function getImage(item) {
    const disabled = !item.isActive;

    if (item.idType === 'uuid') {
        return getAppIcon(item.imageHref, disabled)
            .catch(() => getAppIcon("default", disabled));
    }

    return getServiceIcon(item.imageHref, disabled)
        .catch(() => getServiceIcon("default", disabled));
}

export function GraphSection() {
    const loading = useSelector(getPublicGraphLoading);
    const graph = useSelector(getPublicGraph);

    const edges = useMemo(() => graph.edges?.map(e => ({
        fromId: `${e.id1Type}-${e.id1}`,
        toId: `${e.id2Type}-${e.id2}`,
    })) ?? [], [graph]);

    const graphNodes = useMemo(() => graph.idsInfo?.map(e => ({
        id: `${e.idType}-${e.id}`,
        idType: e.idType,
        text: e.title,
        popup: (e.popup?.header || e.popup?.text) ? {
            header: e.popup?.header,
            text: e.popup?.text,
        } : null,
        isActive: e.isActive,
        ...getNodeMapping(e),
    })) ?? [], [graph]);

    const [nodes, setNodes] = useState(graphNodes);

    useEffect(() => {
        Promise.all(graphNodes.map(e => getImage(e)))
            .then(icons => {
                graphNodes.forEach((node, i) => {
                    node.imageHref = icons[i];
                });

                setNodes([...graphNodes]);
            });
    }, [graphNodes]);

    if (loading) {
        return (
            <Section
                id="graph"
                header="Связанные ID"
            >
                <GraphSkeleton/>
            </Section>
        );
    }

    const description = nodes.length === 0 ? "К сожалению, мы не смогли связать ваши аккаунты" : undefined;

    return (
        <Section
            id="graph"
            header="Связанные ID"
            description={description}
        >
            {
                nodes.length !== 0
                    ? <Graph nodes={nodes} edges={edges}/>
                    : <img alt="Нет данных" src={noData}/>
            }
        </Section>
    );
}
