import { createAction } from "redux-actions";
import { addErrorNotification, dropAllNotifications } from "../../utils/notifications/notificationsState";
import {
    getMyRoles,
    getPublicCryptaId,
    getPublicGraph,
    getPublicProfile,
    getPublicTransactions,
} from "../utils/apiClient";

export const requestMyRoles = createAction("my roles requested");
export const receiveMyRoles = createAction("my roles received");

export function fetchMyRoles() {
    return function (dispatch) {
        dispatch(dropAllNotifications());
        dispatch(requestMyRoles());

        getMyRoles()
            .then((response) => {
                let roles = [];

                if (response) {
                    roles = response.roles;
                }

                dispatch(receiveMyRoles(roles));
            })
            .catch((error) => {
                console.error("Error loading my roles:", error);

                dispatch(receiveMyRoles([]));

                if (error.status && error.status !== 404) {
                    dispatch(addErrorNotification(error, "Error loading my roles"));
                }
            });
    };
}

export const requestPublicProfile = createAction("public profile requested");
export const receivePublicProfile = createAction("public profile received");

export function fetchPublicProfile(cryptaId) {
    return function (dispatch) {
        dispatch(dropAllNotifications());
        dispatch(requestPublicProfile());

        getPublicProfile(cryptaId)
            .then((response) => {
                let profile = {};

                if (response) {
                    profile = response;
                }

                dispatch(receivePublicProfile(profile));
            })
            .catch((error) => {
                console.error("Error loading public profile:", error);

                dispatch(receivePublicProfile({}));

                if (error.status && error.status !== 404) {
                    dispatch(addErrorNotification(error, "Error loading public profile"));
                }
            });
    };
}

export const requestPublicTransactions = createAction("public transactions requested");
export const receivePublicTransactions = createAction("public transactions received");

export function fetchPublicTransactions(cryptaId) {
    return function (dispatch) {
        dispatch(dropAllNotifications());
        dispatch(requestPublicTransactions());

        getPublicTransactions(cryptaId)
            .then((response) => {
                let transactions = [];

                if (response) {
                    transactions = response;
                }

                dispatch(receivePublicTransactions(transactions));
            })
            .catch((error) => {
                console.error("Error loading public transactions:", error);

                dispatch(receivePublicTransactions([]));

                if (error.status && error.status !== 404) {
                    dispatch(addErrorNotification(error, "Error loading public transactions"));
                }
            });
    };
}


export const requestPublicCryptaId = createAction("public crypta id requested");
export const receivePublicCryptaId = createAction("public crypta id received");

export function fetchPublicCryptaId(id) {
    return function (dispatch) {
        dispatch(dropAllNotifications());
        dispatch(requestPublicCryptaId());

        getPublicCryptaId(id)
            .then((response) => {
                let cryptaId = null;

                if (response && response.length > 0) {
                    cryptaId = response[0].cryptaId;
                }

                dispatch(receivePublicCryptaId(cryptaId));

                if (cryptaId) {
                    dispatch(fetchPublicProfile(cryptaId));
                    dispatch(fetchPublicTransactions(cryptaId));
                    dispatch(fetchPublicGraph(cryptaId));
                }
            })
            .catch((error) => {
                console.error("Error loading public crypta id:", error);

                dispatch(receivePublicCryptaId(null));

                if (error.status && error.status !== 404) {
                    dispatch(addErrorNotification(error, "Error loading public crypta id"));
                }
            });
    };
}

export const requestPublicGraph = createAction("public graph requested");
export const receivePublicGraph = createAction("public graph received");

export function fetchPublicGraph(cryptaId) {
    return function (dispatch) {
        dispatch(dropAllNotifications());
        dispatch(requestPublicGraph());

        getPublicGraph(cryptaId)
            .then((response) => {
                let graph = {};

                if (response) {
                    graph = response;
                }

                dispatch(receivePublicGraph(graph));
            })
            .catch((error) => {
                console.error("Error loading public graph:", error);

                dispatch(receivePublicGraph({}));

                if (error.status && error.status !== 404) {
                    dispatch(addErrorNotification(error, "Error loading public graph"));
                }
            });
    };
}
