import Swagger from "swagger-client";
import { isUndefined } from "lodash";
import { getEnv } from "./env";

const HTTP_CODE_OLD_SESSION_ID = 419;
const HTTP_CODE_NOT_AUTHENTICATED = 401;

export default function createSwagger() {
    return getEnv()
        .then((env) => {
            return new Swagger({
                url: env.CLIENT_ENV_API,
                usePromise: true,
                requestInterceptor: (request) => {
                    request.credentials = "include";
                    return request;
                },
                responseInterceptor: (data) => {
                    if (data.status === HTTP_CODE_OLD_SESSION_ID) {
                        if (
                            !isUndefined(data.obj) &&
                            !isEmpty(data.obj.authenticationUrl)
                        ) {
                            fetch(data.obj.authenticationUrl, {
                                credentials: "same-origin",
                            }).then((response) => {
                                if (response.ok) {
                                    console.info(
                                        "You have been reauthenticated at yandex-team.ru"
                                    );
                                }
                            });
                        }
                    } else if (data.status === HTTP_CODE_NOT_AUTHENTICATED) {
                        if (
                            !isUndefined(data.obj) &&
                            !isEmpty(data.obj.authenticationUrl)
                        ) {
                            window.location.assign(data.obj.authenticationUrl);
                        }
                    }
                    return data;
                },
                success: function () {
                    console.log("Successfully connected to Crypta API");
                },
                error: function () {
                    alert("Failed to connect to API");
                },
            });
        })
        .then((api) => {
            return api.apis;
        });
}

export const API = createSwagger();

function getGrabUrl() {
    return getEnv().then((env) => {
        return env.CLIENT_ENV_GRAB_URL;
    });
}

let notAuthorized = new Error("User not authorized");
notAuthorized.status = 403;

let noContent = new Error("No content");
noContent.status = 204;

let notFound = new Error("Not found");
notFound.status = 404;

export { notAuthorized, noContent, notFound };

export function getUserProfile(uidType, uid) {
    return API.then((api) => {
        return api.bs
            .getUserProfile({ uidType: uidType, uid: uid })
            .then((response) => {
                return response;
            });
    });
}

export function getSearchTextWithMatching(uidType, uid) {
    return API.then((api) => {
        return api.bs
            .getUserSearchTextWithMatching({ uidType: uidType, uidValue: uid })
            .then((response) => {
                return response;
            });
    });
}

export function getEnhancedProfile(uidType, uid, matching) {
    return API.then((api) => {
        return api.bs
            .getEnhancedUserProfile({
                uidType: uidType,
                uidValue: uid,
                matching: matching,
            })
            .then((response) => {
                return response;
            });
    });
}

export function getHomeLocation(uid) {
    return API.then((api) => {
        return api.portal
            .getHome({ yandexuid: uid })
            .then((response) => response.obj);
    });
}

export function getWorkLocation(uid) {
    return API.then((api) => {
        return api.portal
            .getWork({ yandexuid: uid })
            .then((response) => response.obj);
    });
}

export function getSelectTypes() {
    return API.then((api) => {
        return api.bs.getSelectTypes().then((response) => response.obj);
    });
}

export function getMeProfile(uidType, uidValue) {
    return API.then((api) => {
        return api.portal
            .getProfile({ uidType: uidType, uid: uidValue })
            .then((response) => response.obj);
    });
}

export function getIdentifiers(idType, idValue) {
    return API.then((api) => {
        return api.portal
            .getIdentifiers({ idType: idType, idValue: idValue })
            .then((response) => {
                return response.obj;
            });
    });
}

export function getBigbProfileSettings() {
    return API.then((api) => {
        return api.portal.getBigbIdTypes().then((response) => {
            return response.obj;
        });
    });
}

export function updateDebugCookie() {
    return API.then((api) => {
        return api.bs.updateDebugCookie().then((json) => {
            return json;
        });
    });
}

export function getAdJson(
    pageId,
    queryParams,
    userId,
    idType,
    userAgent,
    yabsExpSid,
    debugMode
) {
    return API.then((api) => {
        return api.bs
            .getAdJson({
                pageId: pageId,
                queryParams: queryParams,
                userId: idType + "=" + userId,
                userAgent: userAgent,
                yabsExpSid: yabsExpSid,
                debugMode: debugMode,
            })
            .then((response) => {
                return response;
            });
    });
}

export function getPageIds(siteUrl) {
    return API.then((api) => {
        return api.ext.getPageIdsByUrl({ url: siteUrl }).then((response) => {
            return response;
        });
    });
}

export function getGrabLink() {
    return getGrabUrl().then((url) => {
        return API.then((api) => {
            return api.portal.newGrabId().then((json) => {
                const grabId = JSON.parse(json.data);
                return {
                    url: [url, grabId].join("/"),
                    grabId: grabId,
                };
            });
        });
    });
}

export function getGrabResult(grabId) {
    return API.then((api) => {
        return api.portal.getGrab({ grab_id: grabId }).then((json) => {
            return json;
        });
    });
}

export function getTopAffinitySites(yandexuid) {
    return API.then((api) => {
        return api.ext
            .getTopAffinitySites({ yandexuid: yandexuid })
            .then((json) => {
                return json;
            });
    });
}

export function getTopSites() {
    return API.then((api) => {
        return api.ext.getTopSites().then((json) => {
            return json;
        });
    });
}

export function getBmCategories(categories) {
    return API.then((api) => {
        return api.ext
            .getBMCategoriesByIds({ ids: categories })
            .then((json) => {
                return json;
            });
    });
}

export function getTargetDomains(domainIds, uid) {
    return API.then((api) => {
        return api.ext
            .getUserTargetDomainsByIds({ ids: domainIds, yandexuid: uid })
            .then((json) => {
                return json;
            });
    });
}

export function getExperiments(uid) {
    return API.then((api) => {
        return api.experiments
            .getExperimentsParametersFlatten({ id: uid })
            .then((response) => {
                return response;
            });
    });
}

export function getExperimentsDescription(uid) {
    return API.then((api) => {
        return api.experiments
            .getActiveExperiments({ id: uid })
            .then((response) => {
                return response;
            });
    });
}

export function getTransactions(uid) {
    return API.then((api) => {
        return api.tx
            .getAllViaBB({ uid: uid, uidType: "BIGB_UID" })
            .then((response) => {
                return response;
            });
    });
}

export const DEFAULT_MATCHING_TYPE = "v2_prod_yt";
export const DEFAULT_MATCHING_SCOPE = "crypta_id";
export const DEFAULT_GRAPH_DEPTH = 3;

export function getGraphById(
    id,
    idType,
    matchingType = DEFAULT_MATCHING_TYPE,
    matchingScope = DEFAULT_MATCHING_SCOPE,
    depth = DEFAULT_GRAPH_DEPTH
) {
    return API.then((api) => {
        return api.graph
            .getById({
                id: id,
                idType: idType,
                matchingType: matchingType,
                matchingScope: matchingScope,
                depth: depth,
            })
            .then((json) => {
                return json;
            });
    });
}

export function getIdInfo(id, idType) {
    return API.then((api) => {
        return api.graph
            .getIdInfo({
                id: id,
                idType: idType,
            })
            .then((json) => {
                return json;
            });
    });
}

export function exportGraph(
    id,
    idType,
    matchingType = DEFAULT_MATCHING_TYPE,
    matchingScope = DEFAULT_MATCHING_SCOPE,
    depth = DEFAULT_GRAPH_DEPTH
) {
    return API.then((api) => {
        return api.graph
            .exportGraph({
                id: id,
                idType: idType,
                matchingType: matchingType,
                matchingScope: matchingScope,
                depth: depth,
            })
            .then((response) => {
                return response.data;
            });
    });
}

export function getGraphsDiff(
    id1,
    idType1,
    matchingType1,
    matchingScope1,
    depth1,
    id2,
    idType2,
    matchingType2,
    matchingScope2,
    depth2
) {
    return API.then((api) => {
        return api.graph
            .getGraphsDiff({
                id1: id1,
                idType1: idType1,
                matchingType1: matchingType1,
                matchingScope1: matchingScope1,
                depth1: depth1,
                id2: id2,
                idType2: idType2,
                matchingType2: matchingType2,
                matchingScope2: matchingScope2,
                depth2: depth2,
            })
            .then((response) => {
                return response;
            });
    });
}

export function exportGraphsDiff(
    id1,
    idType1,
    matchingType1,
    matchingScope1,
    depth1,
    id2,
    idType2,
    matchingType2,
    matchingScope2,
    depth2
) {
    return API.then((api) => {
        return api.graph
            .exportGraphsDiff({
                id1: id1,
                idType1: idType1,
                matchingType1: matchingType1,
                matchingScope1: matchingScope1,
                depth1: depth1,
                id2: id2,
                idType2: idType2,
                matchingType2: matchingType2,
                matchingScope2: matchingScope2,
                depth2: depth2,
            })
            .then((response) => {
                return response.data;
            });
    });
}

function toMatchTypeConfig(matchType) {
    return API.then((api) => {
        return api.graph.getMatchTypeConfig({ matchingType: matchType });
    })
        .then((response) => {
            return response.obj;
        })
        .catch((error) => {
            console.error("Got error getting match type config:", error);
        });
}

export function getGraphSettings() {
    return API.then((api) => {
        return api.graph.getSupportedMatchTypes().then((response) => {
            let matchTypes = response.obj;
            return Promise.all(
                matchTypes.map((matchType) => toMatchTypeConfig(matchType))
            );
        });
    });
}

export function evalGeo(
    yandexuid,
    latitude,
    longitude,
    state,
    location,
    timestamp
) {
    return API.then((api) => {
        return api.me
            .storeGeoEvaluationToYt({
                yandexuid: yandexuid,
                latitude: latitude,
                longitude: longitude,
                state: state,
                location: location,
                timestamp: timestamp,
            })
            .then((json) => {
                return json;
            });
    });
}

export function getCommonProfile(commonId) {
    return API.then((api) => {
        return api.bs
            .getCommonProfile({ common_id: commonId })
            .then((response) => {
                return response.obj;
            });
    });
}

export function getUserAgents() {
    return API.then((api) => api.ext.getTopUserAgents())
        .then((response) => {
            return response.obj;
        })
        .catch((error) => {
            console.error("Can't fetch userAgents:", error);
        });
}

export function getEntityCounters(uid, uidType) {
    return API.then((api) =>
        api.bs.getEntityCounters({
            uidValue: uid,
            uidType: uidType,
        })
    )
        .then((response) => {
            return response;
        })
        .catch((error) => {
            console.error("Can't fetch entity counters:", error);
        });
}
