import Swagger from "swagger-client";

export function getEnv() {
    return fetch("/env.json").then((response) => response.json());
}

function createSwagger() {
    return getEnv()
        .then((env) => {
            return new Swagger({
                url: env.CLIENT_ENV_API,
                usePromise: true,
                requestInterceptor: (request) => {
                    request.credentials = "include";
                    return request;
                },
                success: function () {
                    console.log("Successfully connected to Crypta API");
                },
                error: function () {
                    alert("Failed to connect to API");
                },
            });
        })
        .then((api) => {
            return api;
        });
}

export const API = createSwagger();

function createResponse(type, value) {
    return Promise.resolve({
        obj: {
            Responses: [
                {
                    Value: {
                        [type]: value,
                    },
                },
            ],
        },
    });
}

export function getSearchResult(query) {
    const urlParams = new URLSearchParams(window.location.search);
    const nocache = urlParams.has("nocache");
    const lang = urlParams.get("lang");

    const clientSide = {
        ["^(yandexuid)$"]: (query) =>
            createResponse(
                "Url",
                `https://redirect.crypta.yandex.ru/${window.location.host}/`
            ),
        ["^(useragent|user-agent|user agent|ua)$"]: (query) =>
            createResponse("UserAgent", `"${navigator.userAgent}"`),
        ["^banner [0-9]+$"]: (query) => createResponse("Banner", query),
    };

    for (const re in clientSide) {
        if (new RegExp(re).test(query)) {
            return clientSide[re](query);
        }
    }

    return API.then((api) => api.apis.search.search).then((method) =>
        method({ query: query, nocache: nocache, lang: lang })
    );
}

export function getSearchExamples() {
    return API.then((api) => api.apis.search.examples).then((method) =>
        method()
    );
}

export function getLtpResult(options) {
    return API.then((api) => api.apis.ltp.getLtpProfile).then((method) =>
        method(options)
    );
}

export function getUserHistory() {
    return API.then((api) => api.apis.ltp.getUserHistory).then((method) =>
        method()
    );
}

export function getLtpProgress({ Id, IdType, FromDate, ToDate }) {
    return API.then((api) => api.apis.ltp.getLtpProgress).then((method) =>
        method({ Id, IdType, FromDate, ToDate })
    );
}
