import React, {Component} from "react";
import _ from "lodash";
import Copyable from "@lib/components/copyable";

import "./ExpStats.scss";

const META_KEYS = ["exp", "exp_description", "rsya", "weight", "child", "special_grep"];

const IMPORTANT_WORDS = [
    "capped_real_cost",
    "cost",
    "ctr",
    "cpm",
    "cpc",
    "pr_",
    "time",
    "clicks",
    "shows",
];

function getIndexOfOrLast(list, x) {
    const idx = list.indexOf(x);
    if (idx == -1) {
        return list.length;
    } else {
        return idx;
    }
}

function getIndexIfContains(list, x) {
    for (var i = 0; i < list.length; i++) {
        if (x.includes(list[i])) {
            return i;
        }
    }
    return list.length;
}

function sortedKeys(rows, headerKeys) {
    return _(Object.keys(rows[0].Values))
        .sortBy((key) => [
            getIndexOfOrLast(headerKeys, key),
            getIndexOfOrLast(META_KEYS, key),
            key.endsWith("_d") ? "0" : "1",
            getIndexIfContains(IMPORTANT_WORDS, key),
            key,
        ])
        .value();
}

function RowHeaderCell(props) {
    return <th nowrap="true" className={props.className}>
        <Copyable value={props.value}/>
    </th>
}

class ColumnMultiHeaderCell extends Component {
    constructor(props) {
        super(props);
        this.state = {
            height: 0
        }
    }

    componentDidMount() {
        const height = this.domElement.clientHeight;
        this.setState({height});
    }

    render() {
        return <th
            nowrap="true"
            className={this.props.className}
            ref={(domElement) => {
                this.domElement = domElement
            }}
            // calculate proper top margin based on row height
            style={{'top': this.state.height * this.props.rowIndex}}
        >
            <Copyable value={this.props.value}/>
        </th>
    }
}

function ValueCell(props) {
    let value = props.value;
    var classes = ["value"];

    if (value === undefined) {
        return <td className={classes.join(" ")}/>;
    }

    let absDelta = Math.abs(value.Delta);

    if (absDelta >= 2.0) {
        classes.push("double-sigma");
    } else if (absDelta >= 1.0) {
        classes.push("single-sigma");
    }
    if (value.Delta < 0.0) {
        classes.push("positive");
    } else {
        classes.push("negative");
    }
    return (
        <td className={classes.join(" ")}>
            <Copyable value={props.value.Repr}/>
            {absDelta >= 1 && (
                <span className="sigma">
                    {absDelta.toFixed(1) + "σ"}
                </span>
            )}
        </td>
    );
}

function Row(props) {
    return <tr className={props.index % 2 == 0 ? "even" : "odd"}>
        {props.children}
    </tr>
}

export class ExpStats extends Component {
    render() {
        // in fact, we transpose original rows and columns
        const columns = this.props.data.Stats.Rows;
        if (columns.length === 0) {
            return null;
        }
        const headerKeys = this.props.data.Stats.Keys;
        const rowKeys = sortedKeys(columns, headerKeys);

        const rows = rowKeys.map((rowKey, i) => (
            <Row index={i} key={"key-" + i} header={headerKeys.includes(rowKey)}>
                <RowHeaderCell
                    className="row-key"
                    key={"key-" + i}
                    value={rowKey}
                />
                {columns.map((column, j) => {
                        if (headerKeys.includes(rowKey)) {
                            return <ColumnMultiHeaderCell
                                className="column-key"
                                key={"col-" + j}
                                value={column.Values[rowKey].Repr}
                                rowIndex={i}
                            />
                        } else {
                            return <ValueCell
                                key={"col-" + j}
                                value={column.Values[rowKey]}
                            />
                        }
                    }
                )}
            </Row>
        ));

        return (
            <div className="exp-stats">
                <div className="exp-stats-inner">
                    <table>
                        <thead>
                        {rows.filter((row) => row.props.header)}
                        </thead>
                        <tbody>
                        {rows.filter((row) => !row.props.header)}
                        </tbody>
                    </table>
                </div>
            </div>
        );
    }
}

export { sortedKeys };
