import React, { useCallback, useEffect, useRef, useState } from 'react';
import PropTypes from 'prop-types';
import { useEnterPress, useInput, useDebounce } from 'hooks';
import { InputTag } from './InputTag';
import { SuggestPopup } from './SuggestPopup';
import { getSuggestData } from './utils';
import './InputWithTags.scss';

export function InputWithTags({
    tags,
    onTagsChange,
    onValueChange,
    placeholder,
    maxTags,
    maxTagLength,
    maxSuggestCount,
    validator,
    normalizer,
    suggest,
    children,
}) {
    const [showSuggest, setShowSuggest] = useState(false);
    const [suggestData, setSuggestData] = useState([]);
    const inputRef = useRef(null);
    const { value, onChange, setValue } = useInput();

    const readOnly = maxTags && tags.length >= maxTags;
    const hasValue = Boolean(value);

    useEffect(() => {
        onValueChange?.(value);
    }, [value, onValueChange]);

    useDebounce(() => {
        if (!suggest) {
            return;
        }

        const newSuggestData = getSuggestData(
            suggest,
            value,
            maxSuggestCount,
        );
        setSuggestData(newSuggestData);

        const hasData = newSuggestData.length > 0;
        setShowSuggest(hasData);
    }, 200, [value, maxSuggestCount, suggest]);

    const addTagToList = useCallback((newTag) => {
        if (!newTag) {
            return;
        }

        if (validator && !validator(newTag)) {
            return;
        }

        if (normalizer) {
            newTag = normalizer(newTag);
        }

        setValue('');
        onTagsChange([...tags, newTag]);
    }, [tags, validator, normalizer, onTagsChange]);

    const handleEnterPress = useEnterPress(() => {
        addTagToList(value);
    }, [addTagToList, value]);

    const onKeyDown = useCallback((e) => {
        handleEnterPress(e);

        if (e.keyCode === 8 && !hasValue) {
            const newTags = tags.slice(0, tags.length - 1);
            onTagsChange(newTags);
        }
    }, [tags, hasValue, handleEnterPress, onTagsChange]);

    const onBlur = useCallback(() => {
        addTagToList(value);
    }, [addTagToList, value]);

    const onSuggestClose = useCallback(() => {
        setShowSuggest(false);
    }, []);

    const onSuggestItemSelect = useCallback((item) => {
        addTagToList(item);
        setShowSuggest(false);
        inputRef.current.focus();
    }, [addTagToList]);

    return (
        <>
            <div className="InputWithTags">
                {children}
                {tags.map((tag, index) => {
                    const tagName = tag.length > maxTagLength
                        ? `${tag.slice(0, maxTagLength - 3)}...`
                        : tag;

                    return (
                        <InputTag
                            key={`${tagName}${index}`}
                            name={tagName}
                            onClose={() => {
                                onTagsChange(tags.filter(
                                    (_, i) => i !== index,
                                ));
                            }}
                        />
                    );
                })}
                <input
                    type="text"
                    className="InputWithTags-Control"
                    placeholder={placeholder}
                    value={value}
                    readOnly={readOnly}
                    ref={inputRef}
                    onChange={onChange}
                    onKeyDown={onKeyDown}
                    onBlur={onBlur}
                />
                <div className="InputWithTags-Box" />
            </div>
            <SuggestPopup
                items={suggestData}
                visible={showSuggest}
                inputRef={inputRef}
                onClose={onSuggestClose}
                onSelect={onSuggestItemSelect}
            />
        </>
    );
}

InputWithTags.propTypes = {
    tags: PropTypes.arrayOf(PropTypes.string).isRequired,
    onTagsChange: PropTypes.func.isRequired,
    onValueChange: PropTypes.func,
    placeholder: PropTypes.string,
    maxTags: PropTypes.number,
    maxTagLength: PropTypes.number,
    suggest: PropTypes.arrayOf(PropTypes.string),
    validator: PropTypes.func,
    normalizer: PropTypes.func,
    maxSuggestCount: PropTypes.number,
};

InputWithTags.defaultProps = {
    maxTagLength: 100,
    maxSuggestCount: 5,
};
