import React, { useCallback, useEffect } from "react";
import { useSelector, useDispatch } from "react-redux";
import FontAwesome from "react-fontawesome";
import copyToClipboard from "copy-to-clipboard";
import { Button } from "@yandex-lego/components/Button/desktop/bundle";
import { ButtonGroup } from "@yandex-lego/components/ButtonGroup/desktop/bundle";
import { Icon } from "@yandex-lego/components/Icon/desktop/bundle";
import {
    toggleLtpFilter,
    loadLtpHistory,
    setLtpResultId,
    loadLtpResult,
    loadLtpProgress,
} from "store/actions/ltp";
import { getLtp, isLtpLoading, isLtpFilterOpen } from "store/selectors/ltp";
import { useDebounceValue } from "hooks";
import { LtpViewerHistory } from "./LtpViewerHistory";
import { LtpViewerForm } from "./LtpViewerForm";
import { LtpViewerLoader } from "./LtpViewerLoader";
import { LtpViewerProgressBar } from "./LtpViewerProgressBar";
import { LtpViewerTable } from "./LtpViewerTable";
import { DEBOUNCE_DELAY } from "./const";
import "./LtpViewer.scss";

export function LtpViewer({ result }) {
    const dispatch = useDispatch();

    const {
        error,
        allResultIdList,
        resultId,
        data: { total },
        progress,
    } = useSelector(getLtp);
    const isFilterOpen = useSelector(isLtpFilterOpen);
    const isLoading = useDebounceValue(
        useSelector(isLtpLoading),
        DEBOUNCE_DELAY
    );
    const hasData = Boolean(total);
    const hasContent = !error && !isLoading;

    const firstResultId = allResultIdList[0]?.resultId ?? null;
    const firstId = allResultIdList[0]?.id ?? null;
    const firstIdType = allResultIdList[0]?.idType ?? null;
    const firstFromDate = allResultIdList[0]?.fromDate ?? null;
    const firstToDate = allResultIdList[0]?.toDate ?? null;

    const resultIdToLoad = (result && result.ResultID) ?? firstResultId;
    const fromDate = (result && result.FromDate) ?? firstFromDate;
    const toDate = (result && result.ToDate) ?? firstToDate;
    const idToLoad = (result && result.Id) ?? firstId;
    const idTypeToLoad = (result && result.IdType) ?? firstIdType;

    const loadTable = useCallback(
        (ResultID, Id, IdType, FromDate, ToDate) => {
            dispatch(
                setLtpResultId({
                    resultId: ResultID,
                    fromDate: FromDate,
                    toDate: ToDate,
                    id: Id,
                    idType: IdType,
                })
            );
            dispatch(loadLtpProgress(Id, IdType, FromDate, ToDate));
        },
        [dispatch]
    );

    const toggleFilter = useCallback(() => {
        dispatch(toggleLtpFilter());
    }, [dispatch]);

    const copyShareLink = useCallback(() => {
        const { origin } = window.location;
        const shareUrl = `${origin}?query="history"%20${resultId}`;

        fetch(`https://nda.ya.ru/--?url=${shareUrl}`)
            .then((response) => response.text())
            .then((url) => copyToClipboard(url))
            .catch((_) => copyToClipboard(shareUrl));
    }, [resultId]);

    useEffect(() => {
        dispatch(loadLtpHistory());
    }, [dispatch]);

    useEffect(() => {
        if (!resultId) {
            dispatch(
                setLtpResultId({
                    resultId: resultIdToLoad,
                    fromDate,
                    toDate,
                    id: idToLoad,
                    idType: idTypeToLoad,
                })
            );
        }
    }, [
        dispatch,
        resultId,
        resultIdToLoad,
        fromDate,
        toDate,
        idToLoad,
        idTypeToLoad,
    ]);

    useEffect(() => {
        dispatch(loadLtpProgress(idToLoad, idTypeToLoad, fromDate, toDate));
    }, [dispatch, idToLoad, idTypeToLoad, fromDate, toDate]);

    const arrowIcon = useCallback(
        (className) => (
            <Icon
                className={className}
                glyph="type-arrow"
                direction={isFilterOpen ? "top" : "bottom"}
            />
        ),
        [isFilterOpen]
    );

    return (
        <div className="LtpViewer">
            {hasContent && (
                <div className="LtpViewer-Left">
                    <div className="LtpViewer-HistoryWrapper">
                        <LtpViewerHistory onLoad={loadTable} />
                    </div>
                </div>
            )}

            {hasContent && (
                <div className="LtpViewer-Right">
                    <div className="LtpViewer-ControlPanel">
                        <ButtonGroup gap="xl">
                            {hasData && (
                                <Button
                                    size="s"
                                    view="pseudo"
                                    iconRight={arrowIcon}
                                    onClick={toggleFilter}
                                >
                                    Фильтрация
                                </Button>
                            )}
                            <Button
                                size="s"
                                view="pseudo"
                                onClick={copyShareLink}
                            >
                                <FontAwesome name="share" />
                            </Button>
                        </ButtonGroup>
                        <LtpViewerProgressBar />
                    </div>
                    {isFilterOpen && <LtpViewerForm />}
                    <LtpViewerTable className="LtpViewer-Table" />
                </div>
            )}

            {error && !isLoading && (
                <span className="LtpViewer-Error">{error.message}</span>
            )}
            {isLoading && <LtpViewerLoader />}
        </div>
    );
}
