import React, { useMemo, useEffect } from "react";
import { useSelector, useDispatch } from "react-redux";
import { useTable, usePagination } from "react-table";
import { loadLtpResult } from "store/actions/ltp";
import { getLtp, ltpFilterForm } from "store/selectors/ltp";
import { useDebounce, useDebounceMemo } from "hooks";
import { DEBOUNCE_DELAY } from "./const";

import "./LtpViewerTable.scss";

const COLUMNS = [
    "EventTime",
    "SourceType",
    "Id",
    "IdType",
    "Description",
    "AdditionalDescription",
];

const sourceTypeIndex = COLUMNS.findIndex((column) => column === "SourceType");

// https://www.color-hex.com/color-palette/55783
const sourceTypeColorMapper = {
    UserSessionQueries: "#fff33b",
    UserSessionClickedDocs: "#fff33b",

    LtpRsyaShows: "#fdc70c",
    LtpSearchShows: "#fdc70c",
    LtpLocalAds: "#fdc70c",

    LtpRsyaClicks: "#f3903f",
    LtpSearchClicks: "#f3903f",

    LtpEcom: "#ed683c",
    CryptaTxLog: "#ed683c",
};

export function LtpViewerTable({ ...rest }) {
    const dispatch = useDispatch();

    const { isTableLoading, resultId, data, id, idType, fromDate, toDate } =
        useSelector(getLtp);
    const { ready, total, items = [] } = data;
    const hasData = Boolean(total);

    const filterForm = useSelector(ltpFilterForm);
    const pageSize = filterForm.Limit;
    const pageCount = total ? Math.ceil(total / pageSize) : 0;

    const { isLoading, isReady } = useDebounceMemo(
        () => ({
            isLoading: resultId ? isTableLoading : true,
            isReady: ready,
        }),
        [isTableLoading, ready, resultId],
        DEBOUNCE_DELAY
    );

    const columns = useMemo(() => {
        return COLUMNS.map((name) => ({
            Header: name,
            accessor: name,
            Cell:
                name === "EventTime"
                    ? (props) => (
                          <span>
                              {new Date(props.value * 1000).toISOString()}
                          </span>
                      )
                    : name === "Description"
                    ? (props) => <pre>{props.value}</pre>
                    : (props) => props.value,
        }));
    }, []);

    const {
        getTableProps,
        getTableBodyProps,
        prepareRow,
        headerGroups,
        page,
        canPreviousPage,
        canNextPage,
        gotoPage,
        nextPage,
        previousPage,
        setPageSize,
        state: { pageIndex },
    } = useTable(
        {
            columns,
            data: decode(items),
            initialState: {
                pageIndex: 0,
                pageSize,
            },
            manualPagination: true,
            pageCount,
        },
        usePagination
    );

    useEffect(() => {
        gotoPage(0);
    }, [resultId, filterForm]);

    useEffect(() => {
        setPageSize(pageSize);
    }, [pageSize]);

    useDebounce(
        () => {
            if (resultId) {
                dispatch(
                    loadLtpResult({
                        ResultID: resultId,
                        FromDate: fromDate,
                        ToDate: toDate,
                        Id: id,
                        IdType: idType,
                        Offset: pageIndex * pageSize,
                        ...filterForm,
                    })
                );
            }
        },
        200,
        [dispatch, resultId, pageSize, pageIndex, filterForm, fromDate, toDate]
    );

    const getRowColor = (row) => {
        const { value: sourceType } = row.cells[sourceTypeIndex];
        const backgroundColor = sourceTypeColorMapper[sourceType];

        if (backgroundColor) {
            return { style: { backgroundColor } };
        }
    };

    return (
        <div className="LtpViewerTable">
            {isLoading && <span>data loading...</span>}
            {!isLoading && !isReady && <span>async request not ready yet</span>}
            {!isLoading && isReady && !hasData && <span>data is empty</span>}
            {!isLoading && isReady && hasData && (
                <>
                    <div className="LtpViewerTable-Table-Container">
                        <table
                            {...rest}
                            {...getTableProps()}
                            className="LtpViewerTable-Table"
                        >
                            <thead>
                                {headerGroups.map((headerGroup) => (
                                    <tr {...headerGroup.getHeaderGroupProps()}>
                                        {headerGroup.headers.map((column) => (
                                            <th {...column.getHeaderProps()}>
                                                {column.render("Header")}
                                            </th>
                                        ))}
                                    </tr>
                                ))}
                            </thead>
                            <tbody {...getTableBodyProps()}>
                                {page.map((row) => {
                                    prepareRow(row);
                                    return (
                                        <tr
                                            {...row.getRowProps(
                                                getRowColor(row)
                                            )}
                                        >
                                            {row.cells.map((cell) => {
                                                return (
                                                    <td
                                                        {...cell.getCellProps()}
                                                    >
                                                        {cell.render("Cell")}
                                                    </td>
                                                );
                                            })}
                                        </tr>
                                    );
                                })}
                            </tbody>
                        </table>
                    </div>
                    <div className="LtpViewerTable-Pagination">
                        <button
                            onClick={() => gotoPage(0)}
                            disabled={!canPreviousPage}
                        >
                            {"<<"}
                        </button>
                        <button
                            onClick={() => previousPage()}
                            disabled={!canPreviousPage}
                        >
                            {"<"}
                        </button>
                        <button
                            onClick={() => nextPage()}
                            disabled={!canNextPage}
                        >
                            {">"}
                        </button>
                        <button
                            onClick={() => gotoPage(pageCount - 1)}
                            disabled={!canNextPage}
                        >
                            {">>"}
                        </button>
                        <span>
                            Страница{" "}
                            <strong>
                                {pageIndex + 1} из {pageCount}
                            </strong>
                        </span>
                    </div>
                </>
            )}
        </div>
    );
}

function decode(items) {
    return items.map((item) => {
        return {
            ...item,
            ...safeDecode(item),
        };
    });
}

function safeDecode(item) {
    try {
        return {
            Description: decodeURIComponent(item.Description),
            AdditionalDescription: decodeURIComponent(
                item.AdditionalDescription
            ),
        };
    } catch {
        return {};
    }
}
