import React from "react";
import FontAwesome from "react-fontawesome";
import { connect, useSelector } from "react-redux";
import { MD5 } from "object-hash";
import { isEmpty } from "lodash";

import Loading from "@lib/components/loading";
import Body from "@lib/components/body";
import Json from "@lib/components/json";
import Footer from "@lib/components/footer";
import { Banner } from "@lib/components/pcode";
import { formatDate } from "@lib/components/time";

import {
    CountLink,
    SimpleResultBox,
    Socdem,
    ExpStatsWikiButton,
    ExpStats,
    StatMx,
    UserDataStats,
    Geo,
    Failable,
    AdReasons,
    SearchHeader,
    HistoricalSearchRequest,
    UrlResult,
    UserHistory,
    KeywordDescription,
    KeywordsDigest,
    Repin,
    LtpViewer,
} from "components";

import { getVersionState, getApiVersionState } from "store/selectors/api";
import { getQuery } from "store/selectors/query";

import "./MainSearch.scss";

@connect((state) => ({
    version: getVersionState(state),
    apiVersion: getApiVersionState(state),
}))
class ConnectedFooter extends React.Component {
    render() {
        const { apiVersion, version } = this.props;
        return <Footer apiVersion={apiVersion} version={version} />;
    }
}

function MissedRoles(props) {
    const roles = props.roles.Roles;
    const text = (
        <React.Fragment>
            Lorem ipsum dolor sit amet, consectetur adipiscing elit. Sed laoreet
            vestibulum erat sit amet pretium. Proin facilisis orci mi, sit amet
            imperdiet nulla fermentum et. Integer tincidunt turpis magna, id
            porttitor lectus euismod eget. Proin sollicitudin eros volutpat,
            semper turpis lobortis, interdum neque. Nulla iaculis tellus dolor,
            eu porta ante venenatis id. Morbi eu tortor lorem. Integer vel
            auctor elit. Fusce convallis porta erat, vel congue tellus congue
            in. Suspendisse id augue ut orci interdum consequat. Sed a tempor
            quam. In varius sem eget tortor tincidunt viverra. In non ante in
            lorem efficitur pharetra. Praesent et mi ante.
            <br />
            <br />
            Duis congue eros at felis mollis, quis efficitur leo posuere.
            Quisque at ornare velit. Sed vestibulum nisl non arcu pulvinar
            scelerisque. Nam a vehicula neque. Vivamus efficitur nunc libero, ut
            commodo ipsum lacinia quis. Sed lacus velit, vehicula at ipsum at,
            dapibus luctus est. Morbi sit amet nulla id neque lacinia efficitur.
            Praesent ut convallis ex.
        </React.Fragment>
    );

    return (
        <div className="missed-roles">
            <div className="wrapper">
                <div className="text">{text}</div>
                <div className="overlay">
                    <div className="block">
                        <div className="icon">
                            <FontAwesome name="eye-slash" />
                        </div>
                        <div className="label">
                            Нет {roles.length === 1 ? "роли" : "ролей"}:{" "}
                            {roles.join(", ")}
                        </div>
                    </div>
                </div>
            </div>
        </div>
    );
}

function BannerResult(props) {
    const { data } = props;
    const re = new RegExp("banner ([0-9]+)");
    const match = re.exec(data);
    const id = match[1];
    return <Banner id={id} />;
}

function SingleResult({ result, type }) {
    const renderers = {
        Socdem: () => <Socdem profile={result} />,
        Geo: () => <Geo regular={result.Regular} actual={result.Actual} />,
        Keyword: () => (
            <KeywordDescription
                description={result.Description}
                dataType={result.DataTypes}
            />
        ),
        Keywords: () => <KeywordsDigest keywords={result.Values} />,
        CountLink: () => <CountLink>{result}</CountLink>,
        Insights: () => <UserDataStats stats={result} />,
        MissedRoles: () => <MissedRoles roles={result} />,
        UserDataStats: () => <UserDataStats stats={result} />,
        ExpStats: () => <ExpStats data={result} />,
        Url: () => <UrlResult data={result} />,
        HistoricalSearchRequest: () => (
            <HistoricalSearchRequest data={result} />
        ),
        UserHistory: () => <UserHistory data={result} />,
        AdReasons: () => <AdReasons data={result} />,
        Banner: () => <BannerResult data={result} />,
        StatMx: () => <StatMx data={result} />,
        LtpProfileResponse: () => <LtpViewer result={result} />,
        LtpProfileResultResponse: () => <LtpViewer result={result} />,
    };

    if (type in renderers) {
        return renderers[type]();
    }

    return <Json data={result} scrollable />;
}

function AdditionalButtons(props) {
    const { result, type } = props;
    const renderers = {
        ExpStats: () => [<ExpStatsWikiButton key="copy-wiki" data={result} />],
    };

    if (type in renderers) {
        return renderers[type]();
    }

    return null;
}

function SearchResults({ values }) {
    const query = useSelector(getQuery);

    return values.map((singleResult) => {
        const responseType = Object.keys(singleResult.Value)[0];
        const result = singleResult.Value[responseType];
        const source = singleResult.Source;
        const meta = singleResult.Meta;
        const json = JSON.stringify(singleResult, null, 2);
        let heading = responseType;

        if (responseType === "Json" || responseType === "Text") {
            heading = source;
        }

        if (responseType === "HistoricalSearchRequest") {
            heading = formatDate(result.Ts);
        }

        if (query.startsWith("history")) {
            return (
                <SimpleResultBox heading="LtpProfileResponse">
                    <LtpViewer result={result} />
                </SimpleResultBox>
            );
        }

        if (query.startsWith("repin")) {
            return (
                <SimpleResultBox heading="Портрет Аудитории">
                    <Repin result={result} type={responseType} />
                </SimpleResultBox>
            );
        }

        return (
            <SimpleResultBox
                key={MD5(singleResult)}
                heading={heading}
                cached={meta?.Cached}
                ts={meta?.Timestamp}
                url={meta?.DetailsUrl}
                toCopy={json}
                buttons={
                    <AdditionalButtons type={responseType} result={result} />
                }
            >
                <SingleResult
                    type={responseType}
                    source={source}
                    result={result}
                />
            </SimpleResultBox>
        );
    });
}

const SearchResult = connect((state) => ({
    searchResult: state.searchResult.payload,
}))((props) => {
    const { searchResult } = props;
    if (isEmpty(searchResult)) {
        return <div />;
    }
    return (
        <Body>
            <div className="search-result-container">
                <SearchResults values={searchResult} />
            </div>
        </Body>
    );
});

export const MainSearch = connect((state) => ({
    loading: state.searchResult.loading,
    failed: state.searchResult.error !== null,
    message: state.searchResult.error,
}))((props) => {
    const { loading, failed, message } = props;

    return (
        <div className="crypta-search-box-container">
            <SearchHeader />
            <Failable failed={failed} errorMessage={message}>
                <Loading loaded={!loading}>
                    <SearchResult />
                </Loading>
            </Failable>
            <ConnectedFooter />
        </div>
    );
});
