import React, { useEffect, useRef } from 'react';
import { useDispatch, useSelector } from 'react-redux';

import { Form, FormSpy } from 'react-final-form';
import arrayMutators from 'final-form-arrays';

import { Button } from '@yandex-lego/components/Button/desktop/bundle';
import { ButtonGroup } from '@yandex-lego/components/ButtonGroup/desktop/bundle';

import { UserDataStats } from 'components';

import { getQuery } from 'store/selectors/query';
import { setQuery } from 'store/actions/query';
import { doSearch } from 'store/actions/search';

import { FormCheckbox } from './FormCheckbox';
import { FormMultiRadioInput } from './FormMultiRadioInput';
import { FormRadiobox } from './FormRadiobox';
import { FormRadioButton } from './FormRadioButton';
import { RegionsInput } from './RegionsInput';

import { Type } from './const';
import { getInitialValues, formStateToQuery, extractHostname } from './utils';

import './Repin.scss';

export function Repin({ result, type }) {
    const dispatch = useDispatch();
    const query = useSelector(getQuery);
    const initialValues = getInitialValues(query);

    const hasResp = type !== 'Text';
    const hasData = hasApiResponse(result);

    const scrollAnchorRef = useRef(null);

    useEffect(() => {
        let tId;

        if (hasResp || hasData) {
            tId = setTimeout(() => {
                scrollAnchorRef.current.scrollIntoView({ behavior: 'smooth' });
            }, 400);
        }

        return () => clearTimeout(tId);
    }, [scrollAnchorRef, result])

    const dispatchQuery = (query) => {
        dispatch(setQuery(query));
        dispatch(doSearch());
    };

    return (
        <div className="Repin">
            <Form
                initialValues={initialValues}
                mutators={{ ...arrayMutators }}
                onSubmit={(formState) => {
                    const query = formStateToQuery(formState);
                    dispatchQuery(query);
                }}
                subscription={{ submitting: true }}
                render={({ handleSubmit }) => (
                    <div className="Repin-Form">
                        <FormRadioButton name={Type.Gender} />
                        <FormCheckbox name={Type.Ages} />
                        <FormCheckbox name={Type.Incomes} />
                        <FormCheckbox name={Type.Countries} />
                        <RegionsInput />
                        <FormCheckbox name={Type.Devices} />
                        <div className="Repin-Divider" />
                        <FormMultiRadioInput name={Type.Hosts} normalizer={extractHostname} />
                        <FormMultiRadioInput name={Type.Phrases} />
                        <FormMultiRadioInput name={Type.Apps} />
                        <FormRadiobox text="Через" name={Type.Aggregate} />
                        <ButtonGroup gap="xl">
                            <FormSpy
                                subscription={{ modified: true, pristine: true }}
                                render={({ modified, pristine }) => {
                                    const disabled = !Object.values(modified).some((x) => x);

                                    return (
                                        <Button
                                            size="s"
                                            view="action"
                                            disabled={disabled || pristine}
                                            onClick={handleSubmit}
                                        >
                                            Поиск
                                        </Button>
                                    )
                                }}
                            />
                            <Button
                                size="s"
                                view="pseudo"
                                onClick={() => dispatchQuery('repin')}
                            >
                                Сбросить
                            </Button>
                        </ButtonGroup>
                    </div >
                )}
            />
            <div ref={scrollAnchorRef}></div>
            {hasData ? (
                <UserDataStats stats={result} />
            ) : (
                hasResp && (
                    <div className="Repin-Empty">
                        По данному запросу ничего не найдено
                    </div>
                )
            )}
        </div >
    );
}

function hasApiResponse(result) {
    try {
        const affinities = Object.values(result.Stats.Affinities);
        const counts = Object.values(result.Stats.Counts);

        const hasAffinities = affinities.some((item) => item.length > 0);
        const hasCounts = counts.some((item) => item.length > 0);

        return hasAffinities || hasCounts;
    } catch {
        return false;
    }
}