import { restoreRegions, mapRegionsToApiFormat } from './regions';
import { Type, DEFAULT_MULTI_RADIO_INPUT, CONTROLS } from '../const';

function queryToJSON(query) {
    try {
        const prefix = /repin\s/;
        const json = query.replace(prefix, '');
        return JSON.parse(json);
    } catch {
        return {};
    }
}

export function getInitialValues(query) {
    const json = queryToJSON(query);
    const initialValues = {};

    const addInitialValueToRadioButton = (type) => {
        const item = CONTROLS[type].options
            .find(({ value }) => value === json[type]);

        initialValues[type] = item?.value ?? '';
    };

    const addInitialValueToCheckbox = (type) => {
        const keys = [...CONTROLS[type].data.keys()];
        initialValues[type] = [...keys].fill(false);

        json[type] && json[type].forEach((value) => {
            const index = keys.indexOf(value);

            if (index > -1) {
                initialValues[type][index] = true;
            }
        });
    };

    addInitialValueToRadioButton(Type.Gender);
    addInitialValueToCheckbox(Type.Ages);
    addInitialValueToCheckbox(Type.Incomes);
    addInitialValueToCheckbox(Type.Countries);
    addInitialValueToCheckbox(Type.Devices);

    const { Kernel: {
        AggregateByOr = 'or',
        Hosts = [],
        Phrases = [],
        Apps = [],
    } = {} } = json;

    initialValues.Aggregate = AggregateByOr ? 'or' : 'and';

    const apiToForm = ({ Negative, Items }) => ({
        radio: Negative ? 'not' : 'or',
        input: serializeTags(Items),
    });

    initialValues.Hosts = Hosts.map(apiToForm);
    initialValues.Phrases = Phrases.map(apiToForm);
    initialValues.Apps = Apps.map(apiToForm);

    [Type.Hosts, Type.Phrases, Type.Apps].forEach((type) => {
        if (initialValues[type].length === 0) {
            initialValues[type].push({ ...DEFAULT_MULTI_RADIO_INPUT });
        }
    });

    initialValues.Regions = serializeTags(restoreRegions(json));

    return initialValues;
}

export function formStateToQuery(formState) {
    const radioButtonToJson = (type) => {
        if (formState[type]) {
            json[type] = formState[type];
        }
    }

    const checkboxToJson = (type) => {
        const array = [...CONTROLS[type].data.keys()]
            .filter((_, i) => formState[type][i]);

        if (array.length > 0) {
            json[type] = array;
        }
    }

    const tagInputToJson = (type) => {
        const array = formState[type].filter(({ input }) => Boolean(input));
        if (array.length > 0) {
            json.Kernel[type] = array.map(({ radio, input }) => ({
                Negative: radio === 'not',
                Items: deserializeTags(input)
            }))
        }
    }

    const json = {};

    json.Kernel = {
        AggregateByOr: formState.Aggregate === 'or',
    }

    radioButtonToJson(Type.Gender);
    checkboxToJson(Type.Ages);
    checkboxToJson(Type.Incomes);
    checkboxToJson(Type.Countries);
    checkboxToJson(Type.Devices);
    tagInputToJson(Type.Hosts);
    tagInputToJson(Type.Phrases);
    tagInputToJson(Type.Apps);

    const regionNames = deserializeTags(formState[Type.Regions]);

    if (regionNames.length > 0) {
        json[Type.Regions] = mapRegionsToApiFormat(regionNames);
    }

    return `repin ${JSON.stringify(json)}`;
}

export const extractHostname = (url) => {
    let hostname;

    if (url.indexOf("//") > -1) {
        hostname = url.split('/')[2];
    } else {
        hostname = url.split('/')[0];
    }

    hostname = hostname.split(':')[0];
    hostname = hostname.split('?')[0];

    return hostname;
}

export const serializeTags = (tags) => tags.join(',');

export const deserializeTags = (value) => value.split(',').filter(Boolean);

export * from './regions';
