import React, { useCallback, useEffect, useState } from 'react';
import { connect } from "react-redux";
import _ from "lodash";
import { useTranslation } from "react-i18next";
import FontAwesome from "react-fontawesome";

import { useEnterPress } from 'hooks';
import { getQuery } from "store/selectors/query";
import { setQuery } from "store/actions/query";
import { doSearch, fetchExamples } from "store/actions/search";
import copyToClipboard from "@lib/utils/clipboard";
import Header from "@lib/components/header";
import User from "@lib/components/user";
import { Icon } from "@lib/components/lego";
import SimpleLogo from "@lib/images/simple_logo.svg";
import { push } from "connected-react-router";

import "./SearchHeader.scss";

function selectExample(examples) {
    return _.sample(_.get(examples, "Examples", [])) || "";
}

function shouldPaste(text) {
    // TODO fetch that from API
    const adsdk = RegExp("https?://adsdk.yandex.ru\/.*");
    const an = RegExp("https?://(yabs|an)\.yandex\.ru/count/.*");
    return adsdk.test(text) || an.test(text);
}

function readClipboard() {
    try {
        return navigator.clipboard.readText();
    } catch {
        return Promise.resolve("");
    }
}

function SearchTextInput(props) {
    const onKeyDown = useEnterPress(props.onSearch);
    return (
        <input
            autoFocus
            type="text"
            className="crypta-search-input"
            spellCheck="false"
            autoComplete="off"
            autoCorrect="off"
            autoCapitalize="off"
            placeholder={props.example}
            value={props.query}
            onChange={props.onChange}
            onKeyDown={onKeyDown}
        />
    );
}

function ClearButton(props) {
    return (
        <span className="clear" onClick={props.onClick}>
            <Icon glyph="type-cross-websearch" />
        </span>
    );
}

function LinkButton(props) {
    const onClick = () => {
        const currentUrl = window.location.href;
        fetch("https://nda.ya.ru/--?url=" + currentUrl)
            .then((response) => response.text())
            .then((url) => copyToClipboard(url))
            .catch((error) => copyToClipboard(currentUrl));
    };

    return (
        <span className="link" onClick={onClick}>
            <FontAwesome name="link" />
        </span>
    );
}

const HelpButton = connect(
    null,
    (dispatch) => ({
        askForHelp: () => {
            dispatch(push({ search: "?query=/help" }));
            dispatch(doSearch("help"));
        }
    })
)((props) => {
    return (
        <span className="help" onClick={props.askForHelp}>
            <FontAwesome name="info-circle" />
        </span>
    );
});

const HistoryButton = connect(
    null,
    (dispatch) => ({
        showHistory: () => {
            dispatch(push({ search: "?query=/history" }));
            dispatch(doSearch());
        }
    })
)((props) => {
    return (
        <span className="history" onClick={props.showHistory}>
            <FontAwesome name="history" />
        </span>
    );
});

const SearchInput = connect(
    (state) => ({
        query: getQuery(state),
        examples: state.searchExamples.payload,
    }),
    (dispatch) => ({
        setQuery: (query) => dispatch(setQuery(query)),
        fetchExamples: () => dispatch(fetchExamples()),
        doSearch: () => dispatch(doSearch())
    })
)((props) => {
    const { query, examples, setQuery, fetchExamples, doSearch } = props;
    const [searchText, setSearchText] = useState("");
    const clearInput = () => {
        setQuery("");
        setSearchText("");
    };
    const delayedSetQuery = useCallback(_.debounce(setQuery, 500), []);
    const changeInput = (event) => {
        event.preventDefault();
        setSearchText(event.target.value);
        delayedSetQuery(event.target.value);
    };

    // когда внешний код меняет query через setQuery,
    // то выставляем в SearchTextInput актуальное значение query
    useEffect(() => {
        setSearchText(query);
    }, [query]);

    useEffect(() => { fetchExamples() }, []);
    useEffect(() => { doSearch() }, []);
    useEffect(() => {
        readClipboard().then(text => {
                if (!query && shouldPaste(text)) {
                    setQuery(text);
                    doSearch();
                }
        }, () => {});
    }, []);

    const onSearch = useCallback(() => {
        setQuery(searchText);
        doSearch();
    }, [setQuery, doSearch, searchText]);

    return (
        <div className="crypta-input-box-container">
            <img className="simpleLogo" src={ SimpleLogo } alt="Crypta" />
            <ClearButton onClick={clearInput} />
            <LinkButton />
            <HelpButton />
            <HistoryButton />
            <SearchTextInput
                query={searchText}
                onChange={changeInput}
                onSearch={onSearch}
                example={selectExample(examples)}
            />
        </div>
    );
});



const SearchButton = connect(
    (state) => ({
        query: getQuery(state),
    }),
    (dispatch, ownProps) => ({
        doSearch: () => dispatch(doSearch(ownProps.query)),
    })
)((props) => {
    const { t } = useTranslation("common");
    return (
        <div className="crypta-search-button" onClick={ props.doSearch } >
            {t("search.button")}
        </div>
    );
});

function SearchBox(props) {
    return (
        <div className="crypta-search-box">
            <SearchInput />
        </div>
    );
}

function UserPic(props) {
    return (
        <div className="crypta-user">
            <User fromCookie />
        </div>
    );
}

export function SearchHeader(props) {
    return (
        <Header>
            <div className="crypta-search-header">
                <SearchBox />
            </div>
        </Header>
    );
}
