import React, { Component } from "react";
import { useTranslation } from "react-i18next";

import DataTable from 'react-data-table-component';
import { Banner } from "@lib/components/pcode";
import copyToClipboard from "@lib/utils/clipboard";
import { TextInput } from "@lib/components/lego";

import "./StatMx.scss";

function BcOrAb(bc, ab) {
    if (ab !== 1.0) {
        return ab;
    }

    return bc;
}

function BannerCell(props) {
    const { t } = useTranslation('common');

    const { row } = props;

    if (!row.BannerTitle && !row.BannerText) {
        if (row.StatValue == 0) {
            return <span className='Title'>{ t('statmx.empty-banner') }</span>
        } else {
            return <span className='Title'>{ t('statmx.banner-without-title-and-text') }</span>
        }
    }

    const url = row.BannerUrl;
    return <a href={ url } className='BannerCell'>
        { row.BannerTitle && <div className='Title'>{ row.BannerTitle }</div> }
        { row.BannerText && <div className='Text'>{ row.BannerText }</div> }
    </a>;
}

function ColumnHeader(name, description) {
    return <span className="ColumnHeader" title={ description }>{ name }<sup>?</sup></span>;
}

function CellWithDescription(value, description) {
    if (!value) {
        return value;
    }

    return <span className='CellWithDescription' title={ description }>{ value }</span>;
}

function Id(value) {
    const onClick = () => copyToClipboard(value);
    return <span className='Id' onClick={ onClick }>{ value }</span>
}

const FilterComponent = ({ filterText, onChange}) => (
    <TextInput
        view="default"
        size="s"
        hasClear
        poll="yes"
        value={filterText}
        onChange={onChange}
    />
);

function filterItems(data, productTypeFilterText, selectTypeFilterText, contextTypeFilterText) {
    if (!productTypeFilterText && !selectTypeFilterText && !contextTypeFilterText) {
        return data;
    }

    return data.filter(
        item => {
            return (!productTypeFilterText || String(item.ProductType) == productTypeFilterText) &&
                   (!selectTypeFilterText || String(item.SelectType) == selectTypeFilterText) &&
                   (!contextTypeFilterText || String(item.ContextType) == contextTypeFilterText);
        },
    );
}

export function StatMx(props) {
    const { t } = useTranslation('common');

    const data = JSON.parse(props.data);

    const [productTypeFilterText, setProductTypeFilterText] = React.useState('');
    const [selectTypeFilterText, setSelectTypeFilterText] = React.useState('');
    const [contextTypeFilterText, setContextTypeFilterText] = React.useState('');

    const filteredItems = filterItems(data, productTypeFilterText, selectTypeFilterText, contextTypeFilterText);

    const statValueColumnId = 'column-stat-value';

    const columns = [
        {
            name: 'Banner',
            cell: row => <BannerCell row={ row }/>,
            minWidth: '400px',
        },
        {
            name: ColumnHeader('BannerID', t('statmx.description.banner-id')),
            cell: row => Id(row.BannerID),
            minWidth: '200px',
        },
        {
            id: statValueColumnId,
            name: ColumnHeader('StatValue', t('statmx.description.stat-value')),
            selector: row => row.StatValue,
            sortable: true,
            minWidth: '120px',
        },
        {
            name: ColumnHeader('SelectType', t('statmx.description.select-type')),
            selector: row => row.SelectType,
            cell: row => CellWithDescription(row.SelectType, row.SelectTypeAsText),
            sortable: true,
            minWidth: '120px',
        },
        {
            name: ColumnHeader('ProductType', t('statmx.description.product-type')),
            selector: row => row.ProductType,
            cell: row => CellWithDescription(row.ProductType, row.ProductTypeAsText),
            sortable: true,
            minWidth: '120px',
        },
        {
            name: ColumnHeader('ContextType', t('statmx.description.context-type')),
            selector: row => row.ContextType,
            cell: row => CellWithDescription(row.ContextType, row.ContextTypeAsText),
            sortable: true,
            minWidth: '120px',
        },
        {
            name: ColumnHeader('Cost', t('statmx.description.cost')),
            selector: row => row.Cost,
            sortable: true,
            minWidth: '100px',
        },
        {
            name: ColumnHeader('Ctr', t('statmx.description.ctr')),
            selector: row => row.Ctr,
            sortable: true,
            minWidth: '100px',
        },
        {
            name: ColumnHeader('SourceCost', t('statmx.description.source-cost')),
            selector: row => row.Cost / BcOrAb(row.ABConversionCostCoef, row.BidCorrection),
            sortable: true,
            minWidth: '120px',
        },
        {
            name: ColumnHeader('APC', t('statmx.description.apc')),
            selector: row => BcOrAb(row.ABConversionCostCoef, row.BidCorrection),
            sortable: true,
            minWidth: '100px',
        },
        {
            name: ColumnHeader('APS', t('statmx.description.aps')),
            selector: row => row.Ctr * BcOrAb(row.ABConversionCostCoef, row.BidCorrection),
            sortable: true,
            minWidth: '100px',
        },
        {
            name: ColumnHeader('OrderID', t('statmx.description.order-id')),
            cell: row => Id(row.OrderID),
            minWidth: '110px',
        },
        {
            name: ColumnHeader('TargetDomainID', t('statmx.description.target-domain-id')),
            cell: row => Id(row.TargetDomainID),
            minWidth: '130px',
        },
        {
            name: 'FilterReason',
            selector: row => row.FilterReason,
            cell: row => CellWithDescription(row.FilterReason, row.FilterReasonAsText),
            sortable: true,
            minWidth: '120px',
        },
    ];

    const subHeaderComponent = (
            <table className="Filter">
                <tbody>
                    <tr>
                    <td>ProductType</td>
                    <td className="Input"><FilterComponent onChange={e => setProductTypeFilterText(e.target.value)} filterText={productTypeFilterText} /></td>
                    <td className="Empty"></td>
                    <td>SelectType</td>
                    <td className="Input"><FilterComponent onChange={e => setSelectTypeFilterText(e.target.value)} filterText={selectTypeFilterText} /></td>
                    <td className="Empty"></td>
                    <td>ContextType</td>
                    <td className="Input"><FilterComponent onChange={e => setContextTypeFilterText(e.target.value)} filterText={contextTypeFilterText} /></td>
                    </tr>
                </tbody>
            </table>
    );

    return <div><DataTable
        columns={columns}
        data={filteredItems}
        striped
        defaultSortFieldId={statValueColumnId}
        defaultSortAsc={false}
        highlightOnHover
        responsive={false}
        fixedHeader
        fixedHeaderScrollHeight={'65vh'}
        subHeader
        subHeaderAlign={'left'}
        subHeaderComponent={subHeaderComponent}
    /></div>;
}
