import React from "react";

import _ from "lodash";
import { HalfDonutChart } from "./HalfDonutChart";
import { PredictedLabel } from "./PredictedLabel";

import "./DemographicChart.scss";

function argmax(obj) {
    return Object.keys(obj).reduce(function (a, b) {
        return obj[a] > obj[b] ? a : b;
    });
}

export class DemographicChart extends React.Component {
    constructor(props) {
        super(props);
        this.setDefaultPrediction = this.setDefaultPrediction.bind(this);
        this.handleDisplayPrediction = this.handleDisplayPrediction.bind(this);

        this.state = { isExact: false };
    }

    componentDidMount() {
        this.setDefaultPrediction();
    }

    setDefaultPrediction() {
        const exact = this.props.exact;
        const isExact = !exact.startsWith("unknown");
        const predictions = _.omit(this.props.values, "timestamp");

        let label;

        if (isExact) {
            label = exact;
        } else {
            label = argmax(predictions);
        }

        let prediction = this.props.labels[label];
        let p = Math.round(_.get(predictions, label) * 100).toFixed(0);

        this.setState({
            predictionName: prediction,
            predictionValue: p,
            isExact: isExact,
        });
    }

    handleDisplayPrediction(prediction) {
        this.setState({
            predictionName: prediction.name,
            predictionValue: Math.round(prediction.count * 100).toFixed(0),
            isExact: false,
        });
    }

    render() {
        const { labels, values, rt_exact } = this.props;

        let segmentIsEmpty = Object.values(values).every(
            (value) => value === 0
        );

        let data = [];
        for (let label in labels) {
            if (labels.hasOwnProperty(label)) {
                data.push({ name: labels[label], count: values[label] });
            }
        }

        return (
            <div className="demographics-chart">
                <HalfDonutChart
                    data={data}
                    width={240}
                    handleMouseOver={this.handleDisplayPrediction}
                    handleMouseLeave={this.setDefaultPrediction}
                />

                <div className="demographics-chart-content-formatting" />

                {!segmentIsEmpty ? (
                    <div
                        title={
                            this.state.isExact
                                ? "Prediction of extended demographic model. May not match the main ones."
                                : "Exact segment is missing; usual segment shown"
                        }
                    >
                        <div className="prediction">
                            {this.state.predictionName}
                        </div>
                        <div className="prediction-score">
                            {this.state.predictionValue}% sure{" "}
                            {this.state.isExact ? "*" : ""}
                        </div>
                        <div className="predicted-time">
                            <PredictedLabel time={values.timestamp} />
                        </div>
                    </div>
                ) : (
                    <div className="empty-prediction">Not predicted yet</div>
                )}

                {!rt_exact.startsWith("unknown") && (
                    <div className="prediction-rt">
                        <span className="prediction-rt-label">Realtime:</span>{" "}
                        <span>{labels[rt_exact]}</span>
                    </div>
                )}
            </div>
        );
    }
}
