import { useState, useEffect, useRef } from 'react';

export const useDebounce = (callback, delay, deps) => {
    const callbackRef = useRef(callback);
    const timeoutRef = useRef(null);

    useEffect(() => {
        callbackRef.current = callback;
    }, [callback]);

    useEffect(() => {
        if (timeoutRef.current !== null) {
            clearTimeout(timeoutRef.current);
        }

        timeoutRef.current = setTimeout(() => {
            callbackRef.current();
            timeoutRef.current = null;
        }, delay);
    }, [...deps, delay]);
};

export const useDebounceValue = (value, delay) => {
    const [debouncedValue, setDebouncedValue] = useState(value);

    useDebounce(() => {
        setDebouncedValue(value);
    }, delay, [value]);

    return debouncedValue;
};

export const useDebounceMemo = (callback, deps, delay) => {
    const [debouncedValue, setDebouncedValue] = useState(callback);

    useDebounce(() => {
        setDebouncedValue(callback());
    }, delay, deps);

    return debouncedValue;
};
