import React, { useEffect, useCallback } from "react";
import ReactDOM from "react-dom";
import { useSelector, useDispatch } from "react-redux";
import { Route, Redirect, Switch, withRouter } from "react-router-dom";

import { createStore, applyMiddleware, compose } from "redux";
import { ConnectedRouter, routerMiddleware } from "connected-react-router";
import { Provider } from "react-redux";
import { createBrowserHistory } from "history";
import thunk from "redux-thunk";

import { configureLego } from "@lib/components/lego";
import ErrorBoundary, { initializeErrorCounter } from "@lib/components/errorboundary";
import { initializeMetrika, provideLoginUserId } from "@lib/utils/metrika";

import createRootReducer from "store/reducers";
import { updateNow } from "store/actions/search";

import { MainSearch, Failable } from "components";

import "i18n/config";
import "styles/index.scss";

function createApplicationStore(history) {
    const middleware = [
        applyMiddleware(thunk),
        applyMiddleware(routerMiddleware(history)),
    ];
    return compose(...middleware)(createStore)(
        createRootReducer(history),
        window.__REDUX_DEVTOOLS_EXTENSION__ && window.__REDUX_DEVTOOLS_EXTENSION__()
    );
}

function DefaultRoute() {
    return <Redirect to={{ pathname: "/" }} />;
}

function Layout(props) {
    const failed = useSelector(state => state.base.failed);
    const message = useSelector(state => state.base.message);

    return (
        <Failable failed={failed} errorMessage={message} >
            {props.children}
        </Failable>
    );
}

function Application(props) {
    const dispatch = useDispatch();
    const updateTime = useCallback(
        () => dispatch(updateNow(Date.now())),
        [dispatch]
    );

    useEffect(() => {
        initializeErrorCounter();
    }, []);
    useEffect(() => {
        initializeMetrika(55654957);
        setTimeout(provideLoginUserId, 5000);
    }, []);
    useEffect(() => {
        setInterval(updateTime, 10000);
    }, []);

    return <Layout>
        <Switch>
            <Route exact path="/" component={MainSearch} />
            <Route path="*" render={DefaultRoute} />
        </Switch>
    </Layout>;
}

const ApplicationWithRouter = withRouter(Application);

function render() {
    configureLego();
    const history = createBrowserHistory();
    const store = createApplicationStore(history);

    ReactDOM.render(
        <React.StrictMode>
            <ErrorBoundary>
                <Provider store={store}>
                    <ConnectedRouter history={history}>
                        <ApplicationWithRouter />
                    </ConnectedRouter>
                </Provider>
            </ErrorBoundary>
        </React.StrictMode>,
        document.getElementById("root")
    );
}

render();
