import { isEmpty } from "lodash";
import { createAction } from "redux-actions";

import { truncateLongerThan } from "@lib/utils/safety";
import { reachGoal } from "@lib/utils/metrika";

import { getQuery } from "store/selectors/query";
import { resetLtp } from "store/actions/ltp";
import { getSearchResult, getSearchExamples } from "clients";

export const gotApi = createAction("got api");
export const updateNow = createAction("update time");

export const requestSearchResults = createAction("search results requested");
export const receiveSearchResults = createAction("search results received");
export const failedSearchResults = createAction("search results failed");

export const requestSearchExamples = createAction("search examples requested");
export const receiveSearchExamples = createAction("search examples received");
export const failedSearchExamples = createAction("search examples failed");

export const requestSingleIdInfo = createAction("single id info requested");
export const receiveSingleIdInfo = createAction("single id info received");
export const failSingleIdInfo = createAction("single id info failed");

export function doSearch() {
    return function (dispatch, getState) {
        let query = getQuery(getState());

        dispatch(resetLtp());

        if (isEmpty(query)) {
            return dispatch(receiveSearchResults([]));
        }

        reachGoal("did_search");
        dispatch(requestSearchResults());

        getSearchResult(query)
            .then((result) => {
                let responses = result.obj.Responses;
                dispatch(receiveSearchResults(responses));

                if (isEmpty(responses)) {
                    reachGoal("nothing_found");
                    throw new Error(
                        "Nothing found for " + truncateLongerThan(query, 10)
                    );
                }
            })
            .catch((error) => {
                console.error("Error getting search result:", error);
                return dispatch(failedSearchResults(error));
            });
    };
}

export function fetchExamples() {
    return function (dispatch, getState) {
        dispatch(requestSearchExamples());
        getSearchExamples()
            .then((result) => {
                return dispatch(receiveSearchExamples(result.obj));
            })
            .catch((error) => {
                return dispatch(failedSearchExamples(error));
            });
    };
}

export function searchForSpecialId(query) {
    return function (dispatch) {
        dispatch(requestSingleIdInfo());

        getSearchResult(query)
            .then((result) => {
                let responses = result.obj.Responses;
                dispatch(receiveSingleIdInfo(responses));

                if (isEmpty(responses)) {
                    throw new Error("Nothing found for " + query);
                }
            })
            .catch((error) => {
                console.error("Error getting search result:", error);
                dispatch(failSingleIdInfo(error.message));
            });
    };
}
