const path = require('path');
const { EnvironmentPlugin } = require('webpack');
const MiniCssExtractPlugin = require('mini-css-extract-plugin');
const HtmlWebpackPlugin = require('html-webpack-plugin');

require('dotenv/config');

const ROOT_DIR = process.cwd();
const BUILD_DIR = path.join(ROOT_DIR, 'build');
const PUBLIC_DIR = path.join(ROOT_DIR, 'public');

module.exports = {
    name: 'client',
    target: 'web',
    entry: {
        app: path.join(ROOT_DIR, 'src/index.js'),
    },
    output: {
        path: BUILD_DIR,
        clean: true,
        publicPath: '/',
        filename: 'js/[name].[contenthash:8].bundle.js',
        chunkFilename: 'js/[id].[chunkhash:8].chunk.js',
        assetModuleFilename: 'assets/[name].[contenthash:8].[ext][query]',
    },
    module: {
        rules: [
            {
                test: /\.jsx?$/,
                exclude: /node_modules/,
                use: [
                    {
                        loader: 'babel-loader',
                        options: {
                            cacheDirectory: true,
                        },
                    },
                ],
            },
            {
                test: /\.s?css$/i,
                use: [MiniCssExtractPlugin.loader, 'css-loader', 'sass-loader'],
            },
            {
                test: /\.(png|svg|jpg|jpeg|webp|gif)$/i,
                type: 'asset/resource',
            },
            {
                test: /\.(woff|woff2|eot|ttf|otf)$/i,
                type: 'asset/resource',
            },
        ],
    },
    resolve: {
        modules: [
            path.join(ROOT_DIR, 'node_modules'),
        ],
        extensions: ['.js', '.jsx', '.json'],
        alias: {
            styles: path.join(ROOT_DIR, './src/styles'),
            '@lib': path.join(ROOT_DIR, '../lib'),
        },
    },
    optimization: {
        moduleIds: 'deterministic',
        runtimeChunk: 'single',
        splitChunks: {
            cacheGroups: {
                vendor: {
                    test: /[\\/]node_modules[\\/]/,
                    name: 'vendors',
                    chunks: 'all',
                },
            },
        },
    },
    plugins: [
        process.env.NODE_ENV === 'production'
            ? new MiniCssExtractPlugin({
                filename: 'css/[name].[contenthash:8].css',
                chunkFilename: 'css/[id].[chunkhash:8].css',
            })
            : new MiniCssExtractPlugin({
                filename: 'css/[name].css',
                chunkFilename: 'css/[id].css',
            }),
        new EnvironmentPlugin(['NODE_ENV']),
        new HtmlWebpackPlugin({
            template: path.join(PUBLIC_DIR, 'index.html'),
            favicon: path.join(PUBLIC_DIR, 'favicon.ico'),
        }),
    ],
};
