import collections
import threading
import urllib

import cachetools
from library.python import resource
from library.python.protobuf.json import proto2json
import telegram
import telegram.ext

from crypta.lib.python import (
    templater,
    time_utils,
)
from crypta.lib.python.lb_pusher import logbroker
from crypta.lib.python.per_thread_session_pool import PerThreadSessionPool
from crypta.why_this_ad.services.tg_bot.lib.access_pb2 import TAccess
from crypta.why_this_ad.services.tg_bot.lib.ad_complaint_pb2 import TAdComplaint
from crypta.why_this_ad.services.tg_bot.lib.bs_info_client import BsInfoClient
from crypta.why_this_ad.services.tg_bot.lib.search_client import SearchClient
from crypta.why_this_ad.services.tg_bot.lib.staff_client import StaffClient


PLACE_ID_TO_STRING = {
    "542": "Перформанс",
    "1542": "Охватный продукт",
}


def run(config, logger):
    bot = TgBot(config, logger)
    bot.run()


class TgBot:
    class _Decorators:
        @staticmethod
        def with_tg_login_check(f):
            def wrapper(self, update, context):
                tg_login = update.message.from_user["username"]
                staff_login = self.staff_client.get_staff_login(tg_login)

                if staff_login is None:
                    response = Text.yandex_only
                    update.message.reply_text(response, reply_markup=telegram.ReplyKeyboardRemove())
                    self._write_access_log(tg_login, staff_login, update.message.text, response)
                    return State.end
                else:
                    handler_result = f(self, update, context, tg_login, staff_login)
                    if isinstance(handler_result, TextResult):
                        update.message.reply_text(handler_result.response, reply_markup=handler_result.reply_markup, parse_mode=handler_result.parse_mode)
                        self._write_access_log(tg_login, staff_login, update.message.text, handler_result.response)
                        return handler_result.next_state
                    elif isinstance(handler_result, VideoResult):
                        file_id = self.video_file_ids.get(handler_result.path)
                        video = file_id if file_id is not None else open(handler_result.path, "rb")
                        message = update.message.reply_video(video, reply_markup=handler_result.reply_markup)

                        if file_id is None:
                            self.video_file_ids[handler_result.path] = message.video.file_id

                        self._write_access_log(tg_login, staff_login, update.message.text, "{}:{}".format(handler_result.path, file_id))
                        return handler_result.next_state
                    else:
                        self.logger.error("Unknown handler result '%s'", handler_result)
                        response = Text.something_wrong
                        update.message.reply_text(response=response, reply_markup=telegram.ReplyKeyboardRemove())
                        self._write_access_log(tg_login, staff_login, update.message.text, response)
                        return State.end

            return wrapper

    def __init__(self, config, logger):
        self.config = config
        self.logger = logger

        self.data = LockDict()
        self.video_file_ids = LockDict()

        self.pq_client_lock = threading.Lock()
        self.pq_client = None
        self.access_log_pq_writer = None
        self.ad_complaint_log_pq_writer = None

        self.session_pool = PerThreadSessionPool()
        self.bs_info_client = BsInfoClient(config.BsInfoApi, self.session_pool, logger)
        self.search_client = SearchClient(config.SearchApi, self.session_pool, logger)
        self.staff_client = StaffClient(config.StaffApi, self.session_pool, logger)

    def run(self):
        self.logger.info("Create pq client and writers...")
        self.pq_client = logbroker.PQClient(
            self.config.Logbroker.Url,
            self.config.Logbroker.Port,
            tvm_id=self.config.Tvm.SelfClientId,
            tvm_secret=self.config.Tvm.Secret,
            logger=self.logger,
        )
        self.access_log_pq_writer = self.pq_client.get_writer(self.config.AccessLogTopic)
        self.ad_complaint_log_pq_writer = self.pq_client.get_writer(self.config.AdComplaintLogTopic)

        wait_link_handler = telegram.ext.MessageHandler(
            telegram.ext.Filters.regex("^https://(an\\.yandex\\.ru|yandex\\.ru/an)/count/.*$"),
            self._handle_link,
        )
        search_link_handler = telegram.ext.MessageHandler(
            telegram.ext.Filters.regex("^(.*://yabs\\.yandex\\.ru/count/.*)$"),
            self._handle_search_link,
        )
        help_handler = telegram.ext.CommandHandler("help", self._handle_help, run_async=True)
        fallback_handler = telegram.ext.MessageHandler(telegram.ext.Filters.all, self._handle_fallback, run_async=True)

        self.logger.info("Run tg updater...")
        conv_handler = telegram.ext.ConversationHandler(
            entry_points=[
                telegram.ext.CommandHandler('start', self._handle_start, run_async=True),
                help_handler,
                wait_link_handler,
            ],
            states={
                State.wait_link: [
                    wait_link_handler,
                ],
                State.wait_feedback: [
                    telegram.ext.MessageHandler(Feedback.regex, self._handle_feedback, run_async=True),
                    wait_link_handler,
                ],
            },
            fallbacks=[
                help_handler,
                search_link_handler,
                fallback_handler,
            ],
        )

        updater = telegram.ext.Updater(token=self.config.Telegram.Token, use_context=True, workers=self.config.Telegram.Workers)
        updater.dispatcher.add_handler(conv_handler)
        updater.dispatcher.add_handler(search_link_handler)
        updater.dispatcher.add_handler(fallback_handler)
        updater.dispatcher.add_error_handler(self._handle_error)

        with self.pq_client:
            with self.access_log_pq_writer, self.ad_complaint_log_pq_writer:
                updater.start_polling()
                updater.idle()

    @_Decorators.with_tg_login_check
    def _handle_start(self, update, context, tg_login, staff_login):
        return TextResult(response=Text.greeting, reply_markup=telegram.ReplyKeyboardRemove(), next_state=State.wait_link)

    @_Decorators.with_tg_login_check
    def _handle_search_link(self, update, context, tg_login, staff_login):
        return TextResult(response=Text.can_not_handle_search_link, reply_markup=telegram.ReplyKeyboardRemove(), next_state=State.end)

    @_Decorators.with_tg_login_check
    def _handle_link(self, update, context, tg_login, staff_login):
        count_link_params = self._get_count_link_params(update.message.text)

        if not count_link_params:
            return TextResult(response=Text.something_wrong_with_link, reply_markup=telegram.ReplyKeyboardRemove(), next_state=State.end)

        hit_log_id = count_link_params[Consts.hit_log_id]
        candidates = self.bs_info_client.get_candidates(hit_log_id)

        response = templater.render_template(
            Text.ad_reasons_template,
            vars={
                "banner_info": self._get_banner_info(count_link_params),
                "candidates": candidates,
                "more_candidates_params": urllib.parse.urlencode({"query": "hitlogid {}".format(hit_log_id)}),
            },
            strict=True,
        )

        self.data[tg_login] = count_link_params

        return TextResult(response=response, reply_markup=Feedback.keyboard, next_state=State.wait_feedback, parse_mode=telegram.ParseMode.HTML)

    @_Decorators.with_tg_login_check
    def _handle_feedback(self, update, context, tg_login, staff_login):
        reason_id = Feedback.feedback_to_reason_id(update.message.text)

        count_link_params = self.data.pop(tg_login)

        if reason_id is not None:
            self._write_ad_complaint_log(tg_login, staff_login, reason_id, count_link_params)

        return TextResult(response=Text.thankyou, reply_markup=telegram.ReplyKeyboardRemove(), next_state=State.end)

    @_Decorators.with_tg_login_check
    def _handle_help(self, update, context, tg_login, staff_login):
        return VideoResult(path=self.config.CountLinkTutorialPath, reply_markup=telegram.ReplyKeyboardRemove(), next_state=State.end)

    @_Decorators.with_tg_login_check
    def _handle_fallback(self, update, context, tg_login, staff_login):
        return TextResult(response=Text.unknown, reply_markup=telegram.ReplyKeyboardRemove(), next_state=State.end)

    def _handle_error(self, update, context):
        self.logger.error(context.error)

    def _write_access_log(self, tg_login, staff_login, request, response):
        proto = TAccess(
            Timestamp=time_utils.get_current_time(),
            TelegramLogin=tg_login,
            StaffLogin=staff_login,
            Request=request,
            Response=response,
        )
        s = proto2json.proto2json(proto)
        self.logger.info(s)

        with self.pq_client_lock:
            self.access_log_pq_writer.write(s)

    def _write_ad_complaint_log(self, tg_login, staff_login, reason_id, count_link_params):
        proto = TAdComplaint(
            Timestamp=time_utils.get_current_time(),
            TelegramLogin=tg_login,
            StaffLogin=staff_login,
            ReasonID=reason_id,
            **{
                k: int(count_link_params[k])
                for k in {"BannerID", "UniqID", "BMCategoryID", "Duid", "PageID", Consts.hit_log_id, Consts.selectype}
            },
        )
        s = proto2json.proto2json(proto)
        self.logger.info(s)

        with self.pq_client_lock:
            self.ad_complaint_log_pq_writer.write(s)

    def _get_count_link_params(self, count_link):
        try:
            responses = self.search_client.search(query=count_link, matcher="CountLink").Responses
            count_link_params = None

            for response in responses:
                which_one = response.Value.WhichOneof("Responses")
                if "CountLink" == which_one:
                    count_link_params = {param.Key: param.Value for param in response.Value.CountLink.Parameters}

            return count_link_params

        except Exception:
            self.logger.exception("")
            return None, None

    @cachetools.cached(cache=cachetools.TTLCache(maxsize=4096, ttl=86400))
    def _get_selecttype_description_unsafe(self, selecttype):
        for response in self.search_client.search(query="ST {}".format(selecttype), matcher="SelectType").Responses:
            if "SelectType" == response.Value.WhichOneof("Responses"):
                return response.Value.SelectType.Description

    def _get_selecttype_description(self, selecttype):
        try:
            return self._get_selecttype_description_unsafe(selecttype)
        except:
            self.logger.exception("")

        return None

    def _get_banner_info(self, count_link_params):
        select_type = count_link_params["SelectType"]

        return [
            ("BannerID", count_link_params["BannerID"]),
            ("SelectType", "{} ({})".format(select_type, self._get_selecttype_description(select_type))),
            ("PlaceID", _place_id_description(count_link_params["PlaceID"])),
            ("CryptaID", _crypta_id_description(count_link_params["CryptaIDv2"])),
            ("Гендер", _gender_description(count_link_params["Gender"])),
            ("Возраст", _age_description(count_link_params["Age"])),
            ("Доход", _income_description(count_link_params["Income"])),
        ]


class Feedback:
    ok = "Хорошая реклама"
    already_hide = "Я уже скрывал этот баннер"
    not_interesting = "Не интересная тема"
    not_actual = "Не актуально, куплено"

    _feedback_to_reason_id = collections.OrderedDict([
        (ok, None),
        (already_hide, 22),
        (not_interesting, 3),
        (not_actual, 14),
    ])

    regex = telegram.ext.Filters.regex("^({})$".format("|".join(_feedback_to_reason_id.keys())))

    keyboard = telegram.ReplyKeyboardMarkup([
        [k] for k in _feedback_to_reason_id.keys()
    ], one_time_keyboard=True)

    @classmethod
    def feedback_to_reason_id(cls, feedback):
        return cls._feedback_to_reason_id[feedback]


TextResult = collections.namedtuple("TextResult", ["response", "reply_markup", "next_state", "parse_mode"], defaults=(None,))
VideoResult = collections.namedtuple("VideoResult", ["path", "reply_markup", "next_state"])


class LockDict:
    def __init__(self):
        self._data = {}
        self._lock = threading.Lock()

    def __setitem__(self, key, value):
        with self._lock:
            self._data[key] = value

    def pop(self, key):
        with self._lock:
            return self._data.pop(key)

    def get(self, key, default=None):
        with self._lock:
            return self._data.get(key, default)


class Consts:
    selectype = "SelectType"
    hit_log_id = "HitLogID"


class State:
    end = telegram.ext.ConversationHandler.END
    (
        wait_link,
        wait_feedback,
    ) = range(2)


class Text:
    greeting = """Привет!
Меня зовут adreasons_bot.
Я подскажу, почему тебе показали какой-то баннер, а потом мы вместе пожалуемся на неинтересную или повторяющуюся рекламу.
Если у тебя есть ссылка на баннер, то кидай её в чат.
Набери /help чтобы узнать, как достать ссылку на баннер.
"""
    yandex_only = "Извини, но этот бот только для сотрудников Яндекса. Добавь свой telegram логин на стафф и мы сможем пообщаться."
    something_wrong = "Что-то пошло не так."
    something_wrong_with_link = "Что-то пошло не так или ссылка невалидная."
    thankyou = "Спасибо!"
    unknown = "Не могу понять, что от меня нужно."
    can_not_handle_search_link = "Это ссылка с поиска, а я умею работать с РСЯ."
    ad_reasons_template = resource.find("/ad_reasons.template").decode("utf-8")


def _crypta_id_description(crypta_id):
    if crypta_id == "0":
        return "не определён"

    return '<a href="https://crypta.yandex-team.ru/me?{}">{}</a>'.format(
        urllib.parse.urlencode({"uidType": "crypta_id", "uid": crypta_id}),
        crypta_id,
    )


def _place_id_description(place_id):
    s = PLACE_ID_TO_STRING.get(place_id)
    if s:
        return "{} ({})".format(place_id, s)

    return place_id


def _gender_description(gender):
    return "не определён" if gender == "-1" else "определён"


def _age_description(age):
    return "не определён" if age == "-1" else "определён"


def _income_description(income):
    return "не определён" if income == "-1" else "определён"
