import click
import logging
from datetime import date, datetime
from typing import Tuple
from customer_service.ml.lib.data.utils import (
    get_secret_uuid,
    get_yql_client,
    execute_query
)

logging.basicConfig(level=logging.INFO)
logger = logging.getLogger(__name__)


@click.command()
@click.option('--input_data', default='//home/samsara/prod_private/ticket', help='Path to input dataset in YT')
@click.option('--output_data', required=True, help='Path to output dataset in YT')
@click.option('--queues', '-q', required=True, multiple=True, type=int,
              help='List of queues to select data from tickets')
@click.option('--start_dt', type=click.DateTime(formats=["%Y-%m-%d"]), default=str(date(2021, 3, 1)),
              help='Start date to select data from tickets')
@click.option('--yql_query_file', default='chats/data/yql/extract.sql', help='Path to query template')
@click.option('--yql_alias', default='customer-service-yql', help='Alias for YQL in yandex vault')
def main(input_data: str,
         output_data: str,
         queues: Tuple[int],
         start_dt: datetime,
         yql_query_file: str,
         yql_alias: str):

    params = {
        'input_path': input_data,
        'output_path': output_data,
        'queues': list(queues),
        'start_date': start_dt.date()
    }

    _, token = get_secret_uuid(yql_alias, 'YQL_TOKEN')
    client = get_yql_client(token)
    logger.info(f'Extracting raw data from {input_data} to {output_data}')
    execute_query(client, query_file=yql_query_file, templating=params)


if __name__ == '__main__':
    main()
