import itertools
from flask import Flask, jsonify, request
from datacloud.dev_utils.time.utils import now_str
from datacloud.dev_utils.status_db.db import StatusDB
from datacloud.dev_utils.status_db.task import Task, Status


app = Flask(__name__)
MAX_ITEMS = 100


@app.route('/ping')
def ping():
    return 'ping\n'


@app.route('/api/v1.0/tasks', methods=['GET'])
def get_tasks():
    status = request.args.get('status', 'READY')
    db = StatusDB()
    tasks = list(itertools.islice(
        db.get_all_tasks_with_status_raw(status), MAX_ITEMS))
    return jsonify({'tasks': tasks})


@app.route('/api/v1.0/tasks/get_task', methods=['POST'])
def get_task():
    usage = '{"program": "program_name", "key": "program_key"}'
    data = request.json
    program = data.get('program')
    if not program:
        return jsonify({'error': 'empty program', 'usage': usage}), 406
    key = data.get('key')
    if not key:
        return jsonify({'error': 'empty key', 'usage': usage}), 406
    db = StatusDB()
    task = db.get_task_raw(program, key)
    if task:
        return jsonify({'task': task})
    return jsonify({})


@app.route('/api/v1.0/tasks/get_tasks_ready', methods=['POST'])
def get_tasks_ready():
    usage = '{"program": "program_name"}'
    data = request.json
    program = data.get('program')
    if not program:
        return jsonify({'error': 'empty program', 'usage': usage}), 406
    db = StatusDB()
    tasks = list(itertools.islice(
        db.get_tasks_with_status_raw(program, 'READY'), MAX_ITEMS))
    if tasks:
        return jsonify({'tasks': tasks})
    return jsonify({})


@app.route('/api/v1.0/tasks/get_all_ready', methods=['POST'])
def get_all_ready():
    db = StatusDB()
    tasks = list(itertools.islice(
        db.get_all_with_status_raw('READY'), MAX_ITEMS))
    if tasks:
        return jsonify({'tasks': tasks})
    return jsonify({})


@app.route('/api/v1.0/tasks/add_task', methods=['POST'])
def add_task():
    STATUSES = ('READY', 'DONE', 'SKIPPED')
    usage = '{"program": "program_name", "key": "key", "data": {}, "status": "READY"}'
    json_data = request.json
    program = json_data.get('program')
    if not program:
        return jsonify({'error': 'empty program', 'usage': usage}), 406
    key = json_data.get('key')
    if not key:
        return jsonify({'error': 'empty key', 'usage': usage}), 406
    status = json_data.get('status', 'READY')
    if not status or status not in STATUSES:
        return jsonify({'error': 'wrong status, use: READY/DONE/SKIPPED', 'usage': usage}), 406
    task_data = json_data.get('data', {})
    if not isinstance(task_data, dict):
        return jsonify({'error': 'wrong data, use dict', 'usage': usage}), 406

    db = StatusDB()
    current_time = now_str()
    task = Task(
        program=program,
        key=key,
        status=Status(status),
        data=task_data,
        create_time=current_time,
        update_time=current_time,
    )
    db.add_task(task)
    return jsonify({'message': 'task added'})


def get_app():
    return app
