# -*- coding: utf-8 -*-
from datacloud.config.yt import PRODUCTION_ROOT, CRYPTA_DB_LAST_FOLDER
from yt.wrapper import ypath_join, TablePath
from datacloud.dev_utils.yt import features


class CSEBuildConfig(object):
    CREDIT_SCORING_EVENTS = ypath_join(PRODUCTION_ROOT, 'partners_data/credit_scoring_events')
    CRYPTA_DB_FOLDER = CRYPTA_DB_LAST_FOLDER
    DEFAULT_PROD_ROOT = PRODUCTION_ROOT

    COMPRESSION_PARAMS = {
        'compression_codec': 'brotli_3',
        'optimize_for': 'scan'
    }

    def __init__(
            self,
            date=None,
            is_retro=False,
            tag='CSE-FEATURES',
            use_cloud_nodes=False,
            ext_id_key=None,
            root=None,
            data_dir=None,
            ready_dir=None,
            raw_table=None,
            glued_table=None,
            partners=None,
            target_checks=None,
            id_value_to_ext_id=None):

        self.is_retro = is_retro
        self.tag = tag
        self.use_cloud_nodes = use_cloud_nodes

        assert not self.is_retro or root, 'root should be provided in retro'
        self.root = root or self.DEFAULT_PROD_ROOT

        if self.is_retro and id_value_to_ext_id is None:
            self.id_value_to_ext_id = ypath_join(self.root, 'get_yuids', 'input')
        else:
            self.id_value_to_ext_id = id_value_to_ext_id or ypath_join(self.CRYPTA_DB_FOLDER, 'id_value_to_cid')

        assert self.is_retro or not raw_table, 'raw_table shouldn\'t be provided in prod'
        if self.is_retro:
            self.raw_table = raw_table or ypath_join(self.root, 'raw_data', 'table')

        assert self.is_retro or not glued_table, 'glued_table shouldn\'t be provided in prod'
        if self.is_retro:
            self.all_ext_id_table = glued_table or ypath_join(self.root, 'raw_data', 'glued')
        else:
            self.all_ext_id_table = ypath_join(self.CRYPTA_DB_FOLDER, 'cid_to_all')

        assert not self.is_retro or partners, 'partners should be provided in retro'
        assert self.is_retro or not partners, 'partners shouldn\'t be provided in prod'
        if self.is_retro:
            self.partners = partners

        assert self.is_retro or date, 'date should be provided in prod'
        self.date = date or 'features_event'

        if ext_id_key is not None:
            self.ext_id_key = ext_id_key
        elif is_retro:
            self.ext_id_key = 'external_id'
        else:
            self.ext_id_key = 'cid'

        if self.is_retro:
            self.data_dir = data_dir or ypath_join(self.root, '/datacloud/aggregates/credit_scoring_events')
            assert ready_dir is None, 'ready_dir shouldn\'t be provided in retro'
        else:
            self.data_dir = data_dir or ypath_join(self.root, '/datacloud/aggregates/credit_scoring_events/weekly')
            self.ready_dir = ypath_join(self.root, '/datacloud/aggregates/credit_scoring_events/ready')

        self.target_checks = target_checks or [
            lambda targets: len(targets) > 0,
            lambda targets: any(t == 1 for t in targets)
        ]

        if self.is_retro:
            aggregates_schema = [
                {'name': self.ext_id_key, 'type': 'string'},
                {'name': 'target', 'type': 'int64'},
                {'name': 'retro_date', 'type': 'string'},
                {'name': 'partner', 'type': 'string'},
            ]
        else:
            aggregates_schema = [
                {'name': self.ext_id_key, 'type': 'string'},
                {'name': 'target', 'type': 'int64'},
            ]

        self.aggregates_table = TablePath(
            ypath_join(self.data_dir, 'aggregate'),
            attributes=dict(
                schema=aggregates_schema,
                **self.COMPRESSION_PARAMS
            )
        )

        if self.is_retro:
            self.aggregates_table2 = TablePath(
                ypath_join(self.data_dir, 'aggregate2'),
                attributes=dict(
                    schema=aggregates_schema,
                    **self.COMPRESSION_PARAMS
                )
            )

        features_schema = [
            {'name': self.ext_id_key, 'type': 'string'},
            {'name': 'features', 'type': 'string'},
        ]
        self.features_table = TablePath(
            ypath_join(self.data_dir, self.date),
            attributes=dict(
                schema=features_schema,
                **self.COMPRESSION_PARAMS
            )
        )

        if not self.is_retro:
            self.ready_table = TablePath(
                ypath_join(self.ready_dir, 'features'),
                attributes=dict(
                    schema=features_schema,
                    **self.COMPRESSION_PARAMS
                )
            )

    @property
    def cloud_nodes_spec(self):
        return features.cloud_nodes_spec(self.use_cloud_nodes)
