# -*- coding: utf-8 -*-
import yt.wrapper as yt_wrapper
from datetime import datetime, timedelta
from datacloud.dev_utils.yt import yt_utils
from datacloud.dev_utils.time.utils import assert_date_str
from datacloud.dev_utils.time.patterns import FMT_DATE_HM, FMT_DATE
from datacloud.config.yt import YT_PROXY
from datacloud.dev_utils.time.utils import now_str
from datacloud.dev_utils.status_db.task import Task, Status
from datacloud.dev_utils.logging.logger import get_basic_logger

logger = get_basic_logger(__name__)

__all__ = [
    'detect_ready',
    'save_matches'
]


TAG = 'SAVE_ID_VALUE_MATCHES'
DEFAULT_ID_MATCHES_PATH = '//home/x-products/production/crypta_v2/id_value_matches'
DEFAULT_ID_MATCHES_SCHEMA = [
    {'name': 'cid', 'type': 'string'},
    {'name': 'id_value', 'type': 'string'},
    {'name': 'id_type', 'type': 'string'},
    {'name': 'date_str', 'type': 'string'}
]

input_tables = [
    '//home/crypta/production/state/graph/v2/matching/by_id/crypta_id/email_md5',
    '//home/crypta/production/state/graph/v2/matching/by_id/crypta_id/phone_md5'
]


class IdMatchSaver:
    def __init__(self, date_str):
        self.date_str = date_str

    def __call__(self, rec):
        yield {'cid': rec['id'], 'id_value': rec['target_id'], 'id_type': rec['target_id_type'], 'date_str': self.date_str}


def detect_ready(date_time):
    yt_wrapper.config.set_proxy(YT_PROXY)
    date_time = datetime.strptime(date_time, FMT_DATE_HM)
    days = 3
    for min_diff in xrange(days):
        time_to_check = date_time - timedelta(days=min_diff)
        logger.info(' Check day: {}'.format(time_to_check))
        if time_to_check.weekday() != 0:  # Run only on Monday
            logger.info(' Skip {}, not Monday'.format(time_to_check))
            continue
        date_str = time_to_check.strftime(FMT_DATE)
        yield date_str, {'date_str': date_str}
        break


def save_matches(task):
    date_str = task.data['date_str']
    assert_date_str(date_str)
    yt_client = yt_utils.get_yt_client()

    if not yt_client.exists(DEFAULT_ID_MATCHES_PATH):
        yt_utils.create_folders([DEFAULT_ID_MATCHES_PATH], yt_client)

    output_table = yt_wrapper.TablePath(
        yt_wrapper.ypath_join(DEFAULT_ID_MATCHES_PATH, date_str),
        attributes={
            'schema': DEFAULT_ID_MATCHES_SCHEMA,
            'compression_codec': 'brotli_6',
            'erasure_codec': 'lrc_12_2_2',
            'optimize_for': 'scan',
        }
    )

    with yt_client.Transaction():
        yt_client.run_map(
            IdMatchSaver(date_str),
            input_tables,
            output_table,
            spec={'title': '[{}] Save phone/mail to cid id value matchs'.format(TAG)}
        )
        yt_client.run_sort(
            output_table,
            sort_by='id_value',
            spec={'title': '[{}] Sort id_value matches table'.format(TAG)}
        )

    current_time = now_str()
    return [
        task.make_done(),
        Task(
            'blend_crypta_matches', date_str, Status.READY,
            {'date_str': date_str},
            current_time, current_time
        )
    ]
