# -*- coding: utf-8 -*-
from datetime import datetime, timedelta
from yt.wrapper import ypath_join

from datacloud.dev_utils.logging.logger import get_basic_logger
from datacloud.dev_utils.yt.yt_utils import get_yt_client
from datacloud.dev_utils.time.patterns import FMT_DATE_HM, FMT_DATE

from datacloud.features.time_hist.time_hist_grep import run_grep, get_log_tables
from datacloud.features.time_hist.build_config import TimeHistBuildConfig
from datacloud.features.time_hist.time_hist_features import build_time_hist_vectors

from datacloud.dev_utils.crypta import crypta_snapshot
from datacloud.dev_utils.yql import yql_helpers
from datacloud.dev_utils.time.utils import assert_date_str
from datacloud.config.yt import GREP_ROOT


logger = get_basic_logger()


def detect_ready_time_hist_grep(date_time):
    yt_client = get_yt_client()
    date_time = datetime.strptime(date_time, FMT_DATE_HM)
    date_str = date_time.strftime(FMT_DATE)

    result_table = ypath_join(GREP_ROOT, 'region_log', date_str)
    if yt_client.exists(result_table):
        logger.info(' Already calculated %s', result_table)
        return

    for table in get_log_tables(date_str):
        if not yt_client.exists(table):
            logger.info('Log table {} not exists, try later'.format(table))
            return

    yield date_str, {'date_str': date_str}


def run_time_hist_grep(task):
    yt_client = get_yt_client()

    date_str = task.data['date_str']

    assert_date_str(date_str)

    logger.info(' Start grep user retion for %s', date_str)
    run_grep(date_str, yt_client=yt_client)
    logger.info(' Finish grep user retion for %s', date_str)
    return [task.make_done()]


def detect_ready_time_hist_features(date_time, days=None):
    yt_client = get_yt_client()
    days = days or 5
    date_time = datetime.strptime(date_time, FMT_DATE_HM)
    for min_diff in xrange(days):
        time_to_check = date_time - timedelta(days=min_diff)
        logger.info(' Check day: {}'.format(time_to_check))
        date_str = time_to_check.strftime(FMT_DATE)
        snapshot = crypta_snapshot.get_snapshot(yt_client, date_str)
        if snapshot is not None:
            yield date_str, {'date_str': date_str}
            break


def run_time_hist_features(task):
    yt_client = get_yt_client()
    yql_client = yql_helpers.create_yql_client(yt_client, token=yql_helpers.get_yql_token())
    date_str = task.data['date_str']
    assert_date_str(date_str)

    config = TimeHistBuildConfig(is_retro=False, snapshot_date=date_str)
    logger.info(' Start calc time hist for %s', date_str)
    build_time_hist_vectors(yt_client=yt_client, yql_client=yql_client, build_config=config, logger=logger)
    logger.info(' Finish calc time hist for %s', date_str)
    return [task.make_done()]
