#!/usr/bin/env python
# -*- coding: utf-8 -*-
import re
import json
import argparse
from startrek_client import Startrek


ENG2ORIGINALS = {
    'surname': [
        u'Фамилия',
    ],
    'name': [
        u'Имя',
    ],
    'email': [
        u'Почта',
    ],
    'email_for_notifications': [
        u'Укажите адрес электронной почты, на который вы хотите получать уведомления',
    ],
    'phone': [
        u'Телефон',
    ],
    'company_name': [
        u'Название компании',
    ],
    'login': [
        u'Логин клиента',
    ],
    'inn': [
        u'ИНН',
    ],
    'user_destination': [
        u'Куда вы хотите приводить пользователей после клика по объявлению?',
    ],
    'user_purchase_destination': [
        u'Укажите ссылку на страницу завершения покупки/заказа',
    ],
    'user_destination_categories': [
        u'Выберите несколько позиций, страниц сайта или категорий, на которые вам важнее всего привести покупателей с рекламы.',
    ],
    'user_important_actions': [
        u'Какие действия на вашем сайте вы считаете важными?',
    ],
    'site': [
        u'Адрес сайта',
    ],
    'target_audience': [
        u'Целевая аудитория',
    ],
    'product_description': [
        u'Расскажите о своих преимуществах: качестве товара, скидках, особенностях. Чем вы отличаетесь от конкурентов? Что готовы предложить клиентам? Напишите об этом как можно подробнее — как минимум 10 предложений.',
        u'Пожалуйста, расскажите о ваших конкурентных преимуществах',
        u'Расскажите о ваших конкурентных преимуществах. Напишите около 10 фраз о вашем предложении, возможно об уникальных особенностях, скидках, качестве.',
        u'Расскажите о своих преимуществах: качестве товара, скидках, особенностях. Чем вы отличаетесь от конкурентов? Что готовы предложить клиентам?',
        u'Напишите около 10 фраз о вашем предложении. Расскажите о ваших своих конкурентных преимуществах. Чем вы отличаетесь от конкурентов? Что готовы предложить клиентам?',
    ],
    'product_description_link': [
        u'Приложите файл с описанием 1-2 услуг, если вам так будет удобнее',
    ],
    'country': [
        u'В какой стране будут показываться объявления?',
    ],
    'cities': [
        u'Укажите регион показов объявлений',
    ],
    'budget': [
        u'Планируемый бюджет на рекламу в месяц',
    ],
    'currency': [
        u'Выберите валюту',
        u'Укажите валюту',
        u'сurrency',
        u'Валюта',
    ],
    'legal_information': [
        u'Юридическая информация',
        u'Юридическая информация о вашей компании',
    ],
    'fio': [
        u'ФИО',
    ],
    'address': [
        u'Адрес (город, улица, дом, офис)',
    ],
    'phone_company': [
        u'Телефон вашей компании',
        u'Телефон компании',
    ],
    'contact_messanger': [
        u'Контакты в мессенджерах',
    ],
    'contact_link': [
        u'Укажите ссылку на страницу "Контакты"',
    ],
    'work_time': [
        u'Время работы (дни, часы)',
    ],
    'marketing_message_from_yandex_aggrement': [
        u'Согласен на получение рекламных и иных маркетинговых сообщений от ООО "Яндекс" на условиях Политики  конфиденциальности, размещённой по адресу: <a href="https://yandex.ru/legal/confidential/"  target="_blank">https://yandex.ru/legal/confidential/</a>',
        u'Соглашаюсь получать рекламные и иные маркетинговые сообщения от ООО «Яндекс» на условиях Политики конфиденциальности, размещённой по адресу:  <a href="https://yandex.ru/legal/confidential/"  target="_blank">https://yandex.ru/legal/confidential/</a>',
    ],
    'contract_aggrement': [
        u'Я подтверждаю свое согласие на заключение договора на оказание услуг на <a href="https://yandex.ru/legal/terms_yandex_direct_set_up_service/"  target="_blank"> Условиях</a>.',
        u'Я подтверждаю свое согласие на заключение договора на оказание услуг на условиях  <a href="https://yandex.ru/legal/terms_yandex_direct_set_up_service/"> Условий</a>.',
        u'Я подтверждаю свое согласие на заключение договора на оказание услуг на условиях <a href="https://yandex.ru/legal/oferta_yandex_direct_set_up_service/"> Оферты</a> и <a href="https://yandex.ru/legal/terms_yandex_direct_set_up_service/"> Условий</a>.',
        u'Я подтверждаю свое согласие на заключение договора на оказание услуг на условиях <a href="https://yandex.ru/legal/terms_yandex_direct_set_up_service/" target="_blank"> Условий</a>.',
        u'Я подтверждаю свое согласие на заключение договора на оказание услуг на условиях  <a href="https://yandex.ru/legal/terms_yandex_direct_set_up_service/"  target="_blank"> Условий</a>.',
    ],
    'offer_aggrement': [
        u'Я принимаю условия <a href="https://yandex.ru/legal/oferta_direct/?lang=ru"  target="_blank">Оферты на оказание услуг «Яндекс.Директ»</a> и <a href="https://yandex.com/legal/direct_dpa_ch/?lang=ru"  target="_blank">Yandex.Direct DPA</a>',
    ],
    'metrics_aggrement': [
        u'Я подтверждаю свое согласие с <a href="https://yandex.ru/legal/metrica_termsofuse/?lang=ru">Условиями использования сервиса Яндекс.Метрика</a> и разрешаю внесение изменений в счетчик Яндекс.Метрики согласно указанным мною сведениям.',
        u'Я подтверждаю свое согласие с <a href="https://yandex.ru/legal/metrica_termsofuse/?lang=ru"  target="_blank">Условиями использования сервиса Яндекс.Метрика</a> и разрешаю внесение изменений в счетчик Яндекс.Метрики согласно указанным мною сведениям.',
    ],
    'gdpr_aggrement': [
        u'Я даю свое согласие ООО «ЯНДЕКС» (ОГРН: 1027700229193) на обработку моих персональных данных и указанных сведений в целях использования сервисов Яндекс.Директ и Яндекс.Метрика и сопутствующих функциональных возможностей, улучшения их работы, осуществления информационной и рекламной рассылки сервисов посредством электронной почты или по телефону в целях уточнения и/или подтверждения указанных данных, на условиях, определённых в Политике конфиденциальности, размещённой по адресу: (https://yandex.ru/legal/confidential/)',
        u'Я даю свое согласие ООО «ЯНДЕКС» (ОГРН: 1027700229193) на обработку моих персональных данных и указанных сведений в целях использования сервисов Яндекс.Директ и Яндекс.Метрика и сопутствующих функциональных возможностей, улучшения их работы, осуществления информационной и рекламной рассылки сервисов посредством электронной почты или по телефону в целях уточнения и/или подтверждения указанных данных, на условиях, определённых в Политике конфиденциальности, размещённой по адресу: <a href="https://yandex.ru/legal/confidential/"  target="_blank">https://yandex.ru/legal/confidential/</a>',
    ],
    'yang_info': [
        u'Текстовые поля отправляемые в ЯНГ:'
    ],
    '10_links': [
        u'Ваша Яндекс.Коллекция из 10 карточек (ссылка):',
    ],
    'has_metriks': [
        u'На вашем сайте установлен счетчик Яндекс.Метрики?',
        u'На вашем сайте установлен счётчик Яндекс.Метрики?',
    ],
    '10_links_2': [
        u'Настроить отдельную рекламную кампанию на основе Яндекс.Коллекции из 10 карточек',
        u'Настроить отдельную рекламную кампанию на основе Яндекс.Коллекции из 10 картинок',
    ],
    'want_metriks': [
        u'Если вы хотите, чтобы мы настроили для вас счетчик, то отметьте следующий пункт:',
    ],
    'actions_for_metriks': [
        u'Выберите из списка, что вы хотите отслеживать с помощью Яндекс.Метрики',
    ],
    'want_turbo': [
        u'Я хочу, чтобы мне настроили Турбо-страницу',
    ],
    'CounterID': [
        u'Укажите номер счетчика Метрики',
        u'Укажите номер счётчика',
    ],
    'company_logo': [
        u'Горизонтальный логотип вашей компании',
        u'Логотип вашей компании',
    ],
    'how_check_company_effectiveness': [
        u'Как будете оценивать эффективность подготовленной кампании?',
        u'Как вы будете оценивать эффективность подготовленной кампании?',
    ],
    'goods_description': [
        u'Опишите словами 1-2 позиций, страниц сайта или категорий о которых вам важнее всего рассказать в рекламе.',
    ],
    'goods_description_link': [
        u'Приложите файл с описание 1-2 товаров или услуг, если вам так будет удобнее',
        u'Приложите файл с описанием этих товаров или услуг, если вам так будет удобнее.',
    ],
    'goods_images': [
        u'Изображения ваших товаров или услуг',
    ],
    'shoping_cart_link': [
        u'Укажите ссылку на страницу "Корзина" или форму "Оформление заказа"',
    ],
}


ORIGINAL2ENG = {org: eng for eng, orgs in ENG2ORIGINALS.items() for org in orgs}
DESC_PATTERN = re.compile(r'\n[*]{2}([^*]+)[*]{2}\n%%\n(.+?)\n%%', flags=re.DOTALL)


def issue2dict(issue):
    return {
        'ticket': issue.key,
        'tags': issue.tags,
        'created': issue.createdAt,
        'description': issue.description,
        'client_id': issue.uid,
        'summary': issue.summary,
    }


def parse_desc(desc):
    return re.findall(DESC_PATTERN, desc)


def issue_dict2row(issue_dict):
    row = {}
    row['ticket'] = issue_dict['ticket']
    row['tags'] = issue_dict['tags']
    row['keywords_count'] = (
        30 if 'standart' in issue_dict['tags'] else
        10 if '10phrase' in issue_dict['tags'] else
        100 if '100phrase' in issue_dict['tags'] else
        None
    )
    assert row['keywords_count']
    row['created'] = issue_dict['created']
    row['client_id'] = issue_dict['client_id']
    row['summary'] = issue_dict['summary']
    row['other'] = {}

    kv_pairs = parse_desc(issue_dict['description'])

    all_keys = {k for k, _ in kv_pairs}
    for eng, orgs in ENG2ORIGINALS.items():
        orgs_keys = [org for org in orgs if org in all_keys]
        assert len(orgs_keys) <= 1, r'Has keys:\n%s\nwith the same eng_name: %s' % (orgs_keys, eng)

    for k, v in kv_pairs:
        if k in ORIGINAL2ENG:
            row[ORIGINAL2ENG[k]] = v
        else:
            row['other'][k] = v
    return row


def main():

    parser = argparse.ArgumentParser()
    parser.add_argument('--input-tasks', type=str)
    parser.add_argument('--output-tasks-data', type=str)
    parser.add_argument('--startrek-token-path', type=str)
    args = parser.parse_args()

    with open(args.startrek_token_path, 'r') as f:
        startrek_token = f.read().strip()

    startrek_client = Startrek(useragent='python', token=startrek_token)

    with open(args.input_tasks, 'r') as stream:
        input_tasks = json.load(stream)

    issues = [
        issue_dict2row(issue2dict(startrek_client.issues[row['ticket']]))
        for row in input_tasks
    ]

    with open(args.output_tasks_data, 'w') as output:
        json.dump(issues, output)


if __name__ == '__main__':
    main()
