# -*- coding: utf-8 -*-
import logging
from flask import Blueprint

from datacloud.score_api.server.errors import ApiException, handle_api_error

from .views import ApiView
from .views.score import ScoreValueView

from datacloud.score_api.server.blueprints.views.ydb_score import ScoreValueView as YdbScoreValueView
from datacloud.score_api.server.blueprints.views.ydb_stability import StabilityView
from datacloud.score_api.server.blueprints.views.geo_score import GeoScoreValueView
from datacloud.score_api.server.blueprints.views.ydb_reject import RejectRequestView


__all__ = [
    'get_score_api',
    'get_ydb_score_api',
    'get_ydb_stability_api',
    'get_geo_score_api',
    'get_ydb_reject_api'
]


logger = logging.getLogger(__name__)


def get_score_api(scores_storage, crypta_storage, cookie_sync_api, depricated_crypta_storage, depricated_scores):
    score_api = Blueprint('score_api', __name__)
    score_api.register_error_handler(ApiException, handle_api_error)
    url(score_api, '/accounts/<partner_id>/scores', ScoreValueView,
        scores_storage=scores_storage,
        crypta_storage=crypta_storage,
        cookie_sync_api=cookie_sync_api,
        depricated_crypta_storage=depricated_crypta_storage,
        depricated_scores=depricated_scores)
    return score_api


def get_ydb_score_api(ydb_manager, connection_params, path_config, score_path_storage, cookie_sync_api):
    score_api = Blueprint('score_api', __name__)
    score_api.register_error_handler(ApiException, handle_api_error)
    url(score_api, '/accounts/<partner_id>/scores', YdbScoreValueView,
        ydb_manager=ydb_manager,
        connection_params=connection_params,
        ydb_root_dir=path_config.root_dir,
        score_path_storage=score_path_storage,
        cookie_sync_api=cookie_sync_api)
    return score_api


def get_geo_score_api(ydb_manager, connection_params, path_config, score_path_storage, geo_storage, cookie_sync_api):
    score_api = Blueprint('geo_score_api', __name__)
    score_api.register_error_handler(ApiException, handle_api_error)
    url(score_api, '/accounts/<partner_id>/geoscores', GeoScoreValueView,
        ydb_manager=ydb_manager,
        connection_params=connection_params,
        ydb_root_dir=path_config.root_dir,
        score_path_storage=score_path_storage,
        geo_storage=geo_storage,
        cookie_sync_api=cookie_sync_api)
    return score_api


def get_ydb_stability_api(ydb_manager, connection_params, path_config, stability_storage):
    stability_api = Blueprint('stability_api', __name__)
    stability_api.register_error_handler(ApiException, handle_api_error)

    url(stability_api, '/accounts/<partner_id>/stability', StabilityView,
        ydb_manager=ydb_manager,
        connection_params=connection_params,
        ydb_root_dir=path_config.root_dir,
        stability_storage=stability_storage)

    # TODO: Remove later, when partners will remove call to regular
    url(stability_api, '/accounts/<partner_id>/stability/regular', StabilityView,
        name='freaking-hack-to-support-outdated-api',
        ydb_manager=ydb_manager,
        connection_params=connection_params,
        ydb_root_dir=path_config.root_dir,
        stability_storage=stability_storage)

    return stability_api


def get_ydb_reject_api(ydb_manager, connection_params, path_config, reject_storage):
    reject_api = Blueprint('reject_api', __name__)
    reject_api.register_error_handler(ApiException, handle_api_error)
    url(reject_api, '/accounts/<partner_id>/reject', RejectRequestView,
        ydb_manager=ydb_manager,
        connection_params=connection_params,
        ydb_root_dir=path_config.root_dir,
        reject_storage=reject_storage)
    return reject_api


def url(app, url, view, name=None, methods=None, **kwargs):
    """ Registers view in the Flask app """
    if issubclass(view, ApiView):
        methods = methods or view.methods
        name = name or view.__name__
        view = view.as_view(str(name), **kwargs)
    app.add_url_rule(url, view_func=view, methods=methods)
