USE chyt.hahn/taxi-delivery-rt;


WITH
formatDateTime(DATEADD(hour, -1, toStartOfFiveMinute(NOW('UTC'), 'UTC')), '%Y-%m-%d %R:%S') as HOUR_AGO,
formatDateTime(DATEADD(hour, -3, toStartOfFiveMinute(NOW('UTC'), 'UTC')),'%Y-%m-%d %R:%S') as THREE_HOURS_AGO,
formatDateTime(DATEADD(hour, -24 * 3, toStartOfFiveMinute(NOW('UTC'), 'UTC')), '%Y-%m-%d %R:%S') as THREE_DAYS_AGO,
9 * 60 as READY_STUCK_THRESHOLD, -- время в секундах, которое заявка должна провести в статусе ready_for_approval, чтобы считаться зависшей
20 as CLAIMS_CNT_IN_CITY_THRESHOLD_ALARM,

statuses as ( -- считаем кол-во выполеннных заявок в статусах
    SELECT
        claim_statuses.*,
        company_group as client_to_alert
    FROM (
        SELECT
            corp_client_id,
            status_code,
            count(claim_id) as cnt
        FROM concatYtTablesRange("//home/delivery-dwh/ods/cargo_claims/claim",
                                formatDateTime(toStartOfMonth(now('UTC')), '%Y-%m-%d'),
                                formatDateTime(toStartOfMonth(now('UTC')), '%Y-%m-%d'))
        WHERE 1
            AND utc_updated_dttm > HOUR_AGO
            AND dateDiff('hour', toDateTime(utc_created_dttm), toDateTime(utc_updated_dttm)) < 24
        GROUP BY
            corp_client_id,
            status_code
    ) as claim_statuses
    INNER JOIN "//home/taxi-delivery/analytics/dev/e-solovev/LOGDATA-528/config/alerts_dict" as alerts_dict
    ON claim_statuses.corp_client_id == alerts_dict._corp_client_id
),
cr_by_client_id as (
    SELECT
        client_to_alert,
        corp_client_id,
        round(sumIf(cnt, status_code in ('delivered_finish', 'returned_finish')) / sum(cnt), 3)  as cr_corp_client_id,
        sum(cnt) as claims_cnt_corp_client_id
    FROM statuses
    WHERE client_to_alert != 'NA'
    GROUP BY client_to_alert, corp_client_id
),
cr_by_company_group as (
    SELECT
        client_to_alert,
        round(sumIf(cnt, status_code in ('delivered_finish', 'returned_finish')) / sum(cnt), 3)  as cr_company_group,
        sum(cnt) as claims_total_cnt_company_group,
        toJSONString(map(
            'Завершенных заявок', sumIf(cnt, status_code in ('delivered_finish', 'returned_finish')),
            'Отмена клиентом', sumIf(cnt, status_code in ('cancelled')),
            'Отмена клиентом с оплатой', sumIf(cnt, status_code in ('cancelled_with_payment')),
            'Отмена клиентом без возврата товара', sumIf(cnt, status_code in ('cancelled_with_items_on_hands')),
            'Водитель не найден', sumIf(cnt, status_code in ('performer_not_found')),
            'Отмена сервисом', sumIf(cnt, status_code in ('cancelled_by_taxi')),
            'Ошибка оценки',  sumIf(cnt, status_code in ('estimating_failed')),
            'Ошибка', sumIf(cnt, status_code in ('failed'))
        )) as claims_cnt_by_status_company_group
    FROM statuses
    WHERE client_to_alert != 'NA'
    GROUP BY client_to_alert
),

-- считаем статистику по всем заявкам (общее количество новых заявок и доля зависших в ready_for_approval заявок)
latest_claim_status_changes as (
    SELECT
        *
    FROM (
        SELECT
            *
        FROM
            concatYtTablesRange("//home/delivery-dwh/ods/cargo_claims/claim_status_log",
                                formatDateTime(toStartOfMonth(now('UTC')), '%Y-%m-%d'),
                                formatDateTime(toStartOfMonth(now('UTC')), '%Y-%m-%d'))
        WHERE
            utc_created_dttm > THREE_HOURS_AGO
        ) as claim_status_log
    INNER JOIN "//home/taxi-delivery/analytics/dev/e-solovev/LOGDATA-528/config/alerts_dict" as alerts_dict
    ON claim_status_log.corp_client_id == alerts_dict._corp_client_id
),
claims as ( -- собираем текущее состояние заявок
    SELECT
        claim_id,
        any(tariff_zone_code) as tariff_zone_code,
        any(corp_client_id) as corp_client_id,

        argMax(new_claim_status, utc_created_dttm) as last_status,
        max(utc_created_dttm) as last_status_time,

        argMin(new_claim_status, utc_created_dttm) as first_status,
        min(utc_created_dttm) as first_status_time,

        dateDiff('second', toDateTime(last_status_time, 'UTC'), now('UTC')) as last_status_to_now_diff
    FROM latest_claim_status_changes
    GROUP BY
        claim_id
    HAVING 1
        and first_status_time > HOUR_AGO -- созданы за последний час
        and first_status == 'new'
),
all_corp_client_ids as ( -- все corp_client_id, по которым были успешные claim'ы за последние три дня
    SELECT
        distinct corp_client_id
    FROM (SELECT * FROM concatYtTablesRange("//home/delivery-dwh/ods/cargo_claims/claim",
                             formatDateTime(toStartOfMonth(now('UTC')), '%Y-%m-%d'),
                             formatDateTime(toStartOfMonth(now('UTC')), '%Y-%m-%d'))
          WHERE utc_created_dttm > THREE_DAYS_AGO
    ) as claim
    INNER JOIN "//home/taxi-delivery/analytics/dev/e-solovev/LOGDATA-528/config/alerts_dict" as alerts_dict
    ON claim.corp_client_id == alerts_dict._corp_client_id
),
agg_by_client_id as (
    select
        corp_client_id,
        count(*) as total_claims_cnt,
        1. * countIf(last_status == 'ready_for_approval' and last_status_to_now_diff > READY_STUCK_THRESHOLD) / count(*) as ready_for_approval_stuck_claims_share
    from
        claims
    group by corp_client_id
),
agg_by_all_client_id as ( -- добавляем corp_client_id, по которым нет заявок
    select
        all_corp_client_ids.corp_client_id as corp_client_id,
        total_claims_cnt,
        ready_for_approval_stuck_claims_share
    from
        agg_by_client_id
    right join
        all_corp_client_ids
    on
        agg_by_client_id.corp_client_id == all_corp_client_ids.corp_client_id
),

corp_client_id_stat as ( --объединяем посчитанные CR и число заявок/долю зависших
    select
        cr_corp_client_id,
        claims_cnt_corp_client_id,
        ready_for_approval_stuck_claims_share,
        total_claims_cnt,

        coalesce(cr_by_client_id.corp_client_id, agg_by_all_client_id.corp_client_id) as corp_client_id
    from
        cr_by_client_id
    full join
        agg_by_all_client_id
    on
        cr_by_client_id.corp_client_id == agg_by_all_client_id.corp_client_id
),

corp_client_id_stat_with_info as (
    SELECT
        corp_client_id_stat.*,
        company_group as client_to_alert,
        alert_end_hour,
        alert_start_hour,
        cr_threshold_alarm,
        cr_threshold_warning,
        claims_count_threshold_alarm,
        ready_share_threshold_alarm,
        manager_name,
        manager_staff_login,
        manager_telegram_id,
        manager_telegram_login,
        staff_for_direct_message,
        staff_to_mention
    FROM
        corp_client_id_stat
    INNER JOIN "//home/taxi-delivery/analytics/dev/e-solovev/LOGDATA-528/config/alerts_dict" as alerts_dict
    ON corp_client_id_stat.corp_client_id == alerts_dict._corp_client_id
),

corp_client_id_stat_with_company_group as ( -- добавляем CR по группам компаний
    select
        corp_client_id_stat_with_info.*,
        cr_company_group,
        claims_total_cnt_company_group,
        claims_cnt_by_status_company_group
    from
        corp_client_id_stat_with_info
    join
        cr_by_company_group
    on
        cr_by_company_group.client_to_alert == corp_client_id_stat_with_info.client_to_alert
),
corp_client_id_stat_final as ( -- убираем corp_client_id, за которыми не нужно следить
    select
        corp_client_id_stat_with_company_group.*
    from
        corp_client_id_stat_with_company_group
    left anti join
        "//home/taxi-delivery/analytics/dev/e-solovev/LOGDATA-528/config/corp_client_id_blacklist" as blacklist
    on corp_client_id_stat_with_company_group.corp_client_id == blacklist.corp_client_id
),

-- расчёт CR по городам
statuses_with_city as ( -- считаем кол-во выполеннных заявок в статусах в разбивке по corp_client_id и городу
    SELECT
        agg_statuses.*,
        alerts_dict.company_group as client_to_alert
    FROM (
        SELECT
            corp_client_id,
            status_code,
            dictGetOrDefault('regions', ('agglomeration_name'), tuple(COALESCE(tariff_zone, 'None')), ('NA')) as agglomeration_name,
            count(claim_id) as cnt
        FROM concatYtTablesRange("//home/delivery-dwh/ods/cargo_claims/claim", formatDateTime(toStartOfMonth(now('UTC')), '%Y-%m-%d'))
        WHERE 1
            AND utc_updated_dttm > HOUR_AGO
            AND dateDiff('hour', toDateTime(utc_created_dttm), toDateTime(utc_updated_dttm)) < 24
        GROUP BY
            corp_client_id,
            status_code,
            tariff_zone
    ) AS agg_statuses
    INNER JOIN "//home/taxi-delivery/analytics/dev/e-solovev/LOGDATA-528/config/alerts_dict" as alerts_dict
    ON agg_statuses.corp_client_id == alerts_dict._corp_client_id
),
cr_by_client_id_and_city as (
    SELECT
        t.*,
        alerts_dict.cr_threshold_alarm as cr_threshold_alarm
    FROM (
        SELECT
            corp_client_id,
            agglomeration_name,
            round(sumIf(cnt, status_code in ('delivered_finish', 'returned_finish')) / sum(cnt), 3)  as cr,
            sum(cnt) as claims_cnt_corp_client_id
        FROM statuses_with_city
        WHERE client_to_alert != 'NA'
        GROUP BY corp_client_id, agglomeration_name
        HAVING sum(cnt) > CLAIMS_CNT_IN_CITY_THRESHOLD_ALARM
    ) as t
    INNER JOIN "//home/taxi-delivery/analytics/dev/e-solovev/LOGDATA-528/config/alerts_dict" as alerts_dict
    ON t.corp_client_id == alerts_dict._corp_client_id
),
corp_clients_and_cities_with_low_cr as (
    select
        corp_client_id,
        toJSONString(groupArray(tuple(agglomeration_name, cr))) as low_cr_cities
    from
        cr_by_client_id_and_city
    where
        cr < cr_threshold_alarm
    group by
        corp_client_id
)

select
    corp_client_id_stat_final.*,
    if(low_cr_cities == '', '[]', low_cr_cities) as low_cr_cities -- делаем валидный json, если нет городов с низким CR
from
    corp_client_id_stat_final
left join
    corp_clients_and_cities_with_low_cr
on
    corp_client_id_stat_final.corp_client_id == corp_clients_and_cities_with_low_cr.corp_client_id
order by client_to_alert, corp_client_id
;
