import copy
import datetime
import json
import logging
import os
import io

from collections import defaultdict

from jinja2 import Environment, PackageLoader, select_autoescape
from PIL import Image
from business_models import hahn

import pytz
import requests

from projects.common.telegram import TelegramClient
from projects.common.images import images_merged_vertically_b
from projects.common.helpers import build_tvm_client_for_zora, build_gdocs, read_script_config_json

logging.basicConfig(format='%(asctime)s %(levelname)s %(name)s %(message)s', level=logging.INFO)
logger = logging.getLogger(__name__)


def get_pic_by_url(grafana_session, url):
    resp = grafana_session.get(url)
    if not resp.ok or resp.headers['Content-Type'] != 'image/png':
        raise RuntimeError('error loading grafana image')

    return resp.content


def row_is_alarm(row, current_company_group):
    low_cr = row['cr_corp_client_id'] < current_company_group['cr_threshold_alarm'] and \
             row['claims_cnt_corp_client_id'] >= MIN_CLAIMS_FOR_ALARM_THRESHOLD
    low_claims_cnt = row['total_claims_cnt'] < row['claims_count_threshold_alarm']
    high_stuck_share = row['ready_for_approval_stuck_claims_share'] > row['ready_share_threshold_alarm'] \
                       and row['total_claims_cnt'] >= MIN_CLAIMS_FOR_ALARM_THRESHOLD

    return low_cr or low_claims_cnt or high_stuck_share


def row_is_warning(row, current_company_group):
    low_cr = row['cr_corp_client_id'] < current_company_group['cr_threshold_warning'] and \
             row['claims_cnt_corp_client_id'] >= MIN_CLAIMS_FOR_WARNING_THRESHOLD
    return low_cr


config = read_script_config_json()

is_prod = config['is_production']
managers_sheet_id = config['sheet_id']

tvm_secret = os.environ['TVM_SECRET']
tg_token = os.environ['TELEGRAM_TOKEN']

GR1PICURL = 'https://grafana.yandex-team.ru/render/d-solo/3xXqkYi7z/cargo-statistics?orgId=1&refresh=1m&var-country=All&var-population_group=All&var-agglomeration=All&var-zone_id=All&var-c2c=All&var-corp_client_id={}&var-corp_name=All&var-is_delayed=All&from=3h&to=1m&panelId=7&width=1000&height=500&tz=Europe%2FMoscow'
GR2PICURL = 'https://grafana.yandex-team.ru/render/d-solo/P2Q1y8Gnk/cargo-statistics-2?orgId=1&refresh=1m&var-country=All&var-population_group=All&var-agglomeration=All&var-zone_id=All&var-c2c=All&var-corp_client_id={}&var-corp_name=All&var-is_delayed=All&from=3h&to=1m&panelId=3&width=1000&height=1000&tz=Europe%2FMoscow'

group_chat_id = config['group_chat_id']

MIN_CLAIMS_FOR_WARNING_THRESHOLD = 30
MIN_CLAIMS_FOR_ALARM_THRESHOLD = 30

LOAD_ALERTS_DATA_FROM_YQL = config['load_data_from_yql']


def main():
    logger.info(f'is_prod = {is_prod}, LOAD_ALERTS_DATA_FROM_YQL = {LOAD_ALERTS_DATA_FROM_YQL}')

    tvm_client = build_tvm_client_for_zora()
    tvm_ticket = tvm_client.get_service_ticket_for('gozora')

    proxy_conn_string = f'http://logdata:{tvm_ticket}@go.zora.yandex.net:1080'

    tg_client = TelegramClient(tg_token, dict(https=proxy_conn_string, http=proxy_conn_string))

    logger.info(f'passport auth')
    grafana_session = requests.Session()
    passport_auth_data = {'login': os.environ['ROBOT_YATEAM_LOGIN'], 'passwd': os.environ['ROBOT_YATEAM_PASSWORD']}
    grafana_session.post('https://passport.yandex-team.ru/passport?mode=auth', data=passport_auth_data)

    logger.info('Loading managers')

    gdocs = build_gdocs()

    managers = gdocs.read(
        table_name='managers',
        sheet_id=managers_sheet_id,
        header=1,
    ).to_dict(orient='records')

    if LOAD_ALERTS_DATA_FROM_YQL:
        logger.info('Loading alert data from YQL')

        current_hour = datetime.datetime.now(pytz.timezone('Europe/Moscow')).hour
        with open('calculate_statistics_chyt.sql', 'r') as f:
            query = f.read()

        alerts_data = hahn(query,
                           chyt=True,
                           chyt_clique='taxi-delivery-rt',
                           title='prepare alerts data').to_dict(orient='records')

    else:  # чтение из файла с тестовыми данными, может быть полезно для тестирования
        fname = 'input_data_example.json'

        logger.info(f'Reading example alert data from {fname}')
        current_hour = 12
        with open(fname) as f:  # граф, который собирал этот пример: https://nda.ya.ru/t/5fyt9HTP4jPVvU
            alerts_data = json.load(f)

    groups = defaultdict(dict)

    company_group_fields = ['cr_company_group', 'claims_total_cnt_company_group', 'alert_end_hour', 'alert_start_hour',
                            'cr_threshold_alarm', 'cr_threshold_warning', 'manager_name', 'manager_staff_login',
                            'manager_telegram_id', 'manager_telegram_login']

    # превращаем плоскую табличку из кликхауса в удобоваримый объект, который можно подать на вход шаблонизатору
    for row in alerts_data:
        current_company_group = groups[row['client_to_alert']]

        for field in company_group_fields:
            current_company_group[field] = row[field]

        staff_logins_to_mention = row['staff_to_mention'].split(',')
        tg_logins_to_mention = ['@' + r['telegram_login'] for r in managers if
                                r['staff_login'] in staff_logins_to_mention]
        current_company_group['telegram_mentions'] = ', '.join(tg_logins_to_mention)

        staff_for_direct_message = [x for x in row['staff_for_direct_message'].split(',') if len(x) >= 1]
        chat_ids_for_direct_message = [int(r['telegram_id'])
                                       for r in managers
                                       if r['staff_login'] in staff_for_direct_message and r['telegram_id'] not in ('', '-1')]

        current_company_group['chat_ids_for_direct_message'] = chat_ids_for_direct_message

        current_company_group['claims_cnt_by_status'] = {k: int(v) for k, v in json.loads(row['claims_cnt_by_status_company_group']).items()}

        if 'corp_clients' not in current_company_group:
            current_company_group['corp_clients'] = {}

        current_corp_client = current_company_group['corp_clients'][row['corp_client_id']] = {
            'cr': row['cr_corp_client_id'],
            'claims_cnt': row['claims_cnt_corp_client_id'],  # завершённых за последний час
            'is_alarm': row_is_alarm(row, current_company_group),
            'is_warning': row_is_warning(row, current_company_group),
            'low_cr_cities': json.loads(row['low_cr_cities']),
            'ready_for_approval_stuck_claims_share': row['ready_for_approval_stuck_claims_share'],
            'total_claims_cnt': row['total_claims_cnt'],  # новых за последний час
        }

        # если хотя бы один corp_client_id пробивает порог (и там достаточно заказов), алертим по всей группе компаний
        current_company_group['is_alarm'] = current_company_group.get('is_alarm', False) or (current_corp_client['is_alarm'])
        current_company_group['is_warning'] = current_company_group.get('is_warning', False) or current_corp_client['is_warning']

    # фильтруем groups по попаданию в текущий час
    groups = {k: v for k, v in groups.items() if v['alert_start_hour'] <= current_hour <= v['alert_end_hour']}

    jinja_env = Environment(loader=PackageLoader("__main__"), autoescape=select_autoescape())

    # собираем вместе всё, где есть ворнинги и алармы
    have_warnings = False
    for company_group_name, current_group in groups.items():
        if not (current_group['is_alarm'] or current_group['is_warning']):
            continue

        have_warnings = True

    # отображаем только те corp_client_id, где есть аларм или ворнинг

    groups_for_warning = copy.deepcopy(groups)
    for group_name in groups_for_warning:
        groups_for_warning[group_name]['corp_clients'] = {k: v
                                                          for k, v in groups_for_warning[group_name]['corp_clients'].items()
                                                          if v['is_alarm'] or v['is_warning'] or v['low_cr_cities']}

    if groups:  # чтобы не было сообщений с пустым списком компаний
        logger.info('Sending warnings')
        warning_template = jinja_env.get_template("warnings.template")
        msg = warning_template.render(groups=groups_for_warning, have_warnings=have_warnings)
        tg_client.send_message(msg, group_chat_id)
    else:
        logger.info('Will not send warnings, company group list is empty')

    logger.info('Sending alarms')
    # отправляем алармы

    for company_group_name, current_group in groups.items():
        if not current_group['is_alarm']:
            continue

        # достаём первый попавшийся алармящий corp_client_id, чтобы его заскриншотить
        corp_client_id_for_screenshots = [k for k, v in current_group['corp_clients'].items() if v['is_alarm']][0]

        grafana_image1 = get_pic_by_url(grafana_session, GR1PICURL.format(corp_client_id_for_screenshots))
        grafana_image2 = get_pic_by_url(grafana_session, GR2PICURL.format(corp_client_id_for_screenshots))

        image1_nb = Image.open(io.BytesIO(grafana_image1))
        image2_nb = Image.open(io.BytesIO(grafana_image2))

        grafana_image = images_merged_vertically_b([image2_nb, image1_nb])

        statuses_raw = list(current_group['claims_cnt_by_status'].items())
        claims_statuses = sorted(statuses_raw, key=lambda d: d[1], reverse=True)
        statuses_sorted = sorted(filter(lambda d: d[0] != 'Завершенных заявок', statuses_raw),
                                 key=lambda d: d[1],
                                 reverse=True)
        top_status = statuses_sorted[0][0]

        alert_template = jinja_env.get_template("alert.template")
        msg = alert_template.render(
            client_name=company_group_name,
            corp_clients=current_group['corp_clients'],
            claims_statuses=claims_statuses,
            top_status=top_status,
            claims_total_cnt=current_group['claims_total_cnt_company_group'],
            manager_staff_login=current_group['manager_staff_login'],
            manager_name=current_group['manager_name'],
            manager_telegram_login=current_group['manager_telegram_login'],
            cr_threshold_alarm=current_group['cr_threshold_alarm'],
            cr_threshold_warning=current_group['cr_threshold_warning'],
            telegram_mentions=current_group['telegram_mentions'],
        )

        manager_telegram_id = current_group['manager_telegram_id'] if is_prod else config['default_manager_telegram_id']

        chat_ids_for_alert = [group_chat_id,
                              manager_telegram_id]  # по дефолту отправляем в общий чат и ответственному менеджеру
        if is_prod:  # добавляем телеграмы из конфига
            chat_ids_for_alert.extend(current_company_group['chat_ids_for_direct_message'])

        chat_ids_for_alert = [int(v) for v in chat_ids_for_alert if v and int(v) != -1]
        chat_ids_for_alert = list(set(chat_ids_for_alert))
        logger.info(f'Chat_ids for alarm on {company_group_name}: {chat_ids_for_alert}')

        tg_client.send_photo_batch(grafana_image, msg, chat_ids_for_alert)


if __name__ == '__main__':
    logger.info('Start')
    main()
    logger.info('Finish')
