import json
import logging
import os
import warnings

import pandas as pd
from business_models import hahn, greenplum

from projects.common.helpers import build_gdocs, read_script_config_json

logging.basicConfig(format='%(asctime)s %(levelname)s %(name)s %(message)s', level=logging.INFO)
logger = logging.getLogger(__name__)

__location__ = os.path.realpath(os.path.join(os.getcwd(), os.path.dirname(__file__)))

def main():
    warnings.filterwarnings("ignore")
    gdocs = build_gdocs()

    config = read_script_config_json()
    config_base_path_yt = config['config_base_path_yt']
    config_base_path_gp = config['config_base_path_gp']
    sheet_id = config['sheet_id']

    tables = {
        'managers': ['yt'],
        'company_groups': ['yt'],
        'corp_client_id_blacklist': ['yt'],
    }

    for table in tables:
        logger.info(f'Replicating {table}')

        df = gdocs.read(
            table_name=table,
            sheet_id=sheet_id,
            header=1,
        )

        if table == 'company_groups':
            company_groups_schema = {
                'alert_end_hour': 'int64',
                'alert_start_hour': 'int64',
                'cr_threshold_warning': 'float64',
                'cr_threshold_alarm': 'float64',
                'claims_count_threshold_alarm': 'int64',
                'ready_share_threshold_alarm': 'float64',
            }

            df = df.astype(company_groups_schema)

        for destination in tables[table]:
            if destination == 'yt':
                output_yt_table = config_base_path_yt + table
                hahn.write_table(output_yt_table, df)
            elif destination == 'gp':
                output_gp_table = config_base_path_gp + table
                greenplum.write_table(output_gp_table,
                                      df,
                                      with_grant=True,
                                      operator='ALL',
                                      to='analyst')
            else:
                raise ValueError(destination)

    with open(os.path.join(__location__, 'company_groups.sql')) as f:
        query = f.read()

    company_groups_with_corp_client_ids = greenplum(query)

    with open(os.path.join(__location__, 'additional_company_groups.json')) as f:
        data = json.load(f)
        company_groups_with_corp_client_ids = company_groups_with_corp_client_ids.append(pd.DataFrame(data))

    hahn.write_table(config_base_path_yt + 'corp_client_ids', company_groups_with_corp_client_ids)

    with open(os.path.join(__location__, 'join_query.yql')) as f:
        join_query = f.read()
        hahn(join_query.format(base_path=config_base_path_yt))


if __name__ == '__main__':
    logger.info('Start')
    main()
    logger.info('Finish')
