drop table if exists ndd_deliveries;
CREATE TEMPORARY TABLE ndd_deliveries as
with t as (
    select request_id,
           max(corp_client_id) as corp_client_id,
           bool_or(creation_method = 'confirm_platform_offer') as api_ndd_flg
    from snb_delivery.logdata_ndd_requests_datamart a
    where true
      and (cast(created_dttm as timestamp) at time zone 'UTC' at time zone 'Europe/Moscow')::date
          between current_date-30 and current_date-1
      and mapped_event_type in ('Доставлен', 'Доставлен частично')
      and employer_code not in ('Eva','akk')
      and corp_client_id is not null
    group by 1
)
select corp_contract_id,
       count() as deliveries_cnt,
       bool_or(api_ndd_flg) as api_ndd_flg
from t
  join snb_b2b.dbcorp_client_info lk
    on lk.corp_client_id = t.corp_client_id
group by 1;

drop table if exists points;
CREATE TEMPORARY TABLE points as
select claim_id,
       count(distinct point_id) as deliveries_cnt
from delivery_ods_cargo_claims.claim_point
where utc_created_dttm >= current_date-45
  and point_type = 'destination'
  and visit_status = 'visited'
group by claim_id
distributed by (claim_id);

drop table if exists claims_api_sdd;
CREATE TEMPORARY TABLE claims_api_sdd as
select lk.corp_contract_id,
       sum(case when tariff_class_code = 'night' then deliveries_cnt end) as deliveries_cnt,
       bool_or(tariff_class_code in ('courier','express') and api_source_type = 'api' ) as api_express_flg,
       bool_or(tariff_class_code in ('night') and api_source_type = 'api' ) as api_sdd_flg
from delivery_ods_cargo_claims.claim
  join points
    on points.claim_id = claim.claim_id
  join taxi_cdm_marketplace.fct_order
    on fct_order.order_id = claim.taxi_order_id
  join snb_b2b.dbcorp_client_info lk
    on claim.corp_client_id = lk.corp_client_id
where true
  and claim.utc_created_dttm >= current_date-45
  and fct_order.utc_order_created_dttm >= current_date-31
  and msk_order_created_dt between current_date-30 and current_date-1
group by 1
distributed by (corp_contract_id);

drop table if exists main_deliveries;
CREATE TEMPORARY TABLE main_deliveries as
select corp_contract_id,
       tariff_class_code,
       sum(delivery_point_cnt) as deliveries_cnt
from taxi_cdm_marketplace.fct_order
where true
  and delivery_point_cnt > 0
  ---- 30 days
  and utc_order_created_dttm >= current_date - 31
  and msk_order_created_dt between current_date-30 and current_date-1
  and tariff_class_code <> 'night'
group by 1,2;

drop table if exists deliveries;
CREATE TEMPORARY TABLE deliveries as
with raw as (
    select corp_contract_id, 'NDD'::varchar as tariff_class_code, deliveries_cnt,
           NULL::bool as api_express_flg,
           NULL::bool as api_sdd_flg,
           api_ndd_flg
    from ndd_deliveries
    union all
    select corp_contract_id, 'SDD'::varchar as tariff_class_code, deliveries_cnt,
           api_express_flg, api_sdd_flg,
           NULL::bool as api_ndd_flg
    from claims_api_sdd
    union all
    select corp_contract_id, tariff_class_code, deliveries_cnt,
           NULL::bool as api_express_flg,
           NULL::bool as api_sdd_flg,
           NULL::bool as api_ndd_flg
    from main_deliveries
)
select corp_contract_id,
       sum(deliveries_cnt) as deliveries_cnt,
       sum(deliveries_cnt) filter ( where tariff_class_code in ('courier','express') ) as deliveries_express_cnt,
       sum(deliveries_cnt) filter ( where tariff_class_code in ('SDD') ) as deliveries_sdd_cnt,
       sum(deliveries_cnt) filter ( where tariff_class_code in ('NDD') ) as deliveries_ndd_cnt,
       bool_or(api_express_flg) as api_express_flg,
       bool_or(api_sdd_flg) as api_sdd_flg,
       bool_or(api_ndd_flg) as api_ndd_flg
from raw
group by corp_contract_id
distributed by (corp_contract_id);

drop table if exists res;
CREATE TEMPORARY TABLE res as
with leads_raw as (
    select coalesce(holding_name,company_name,lead_name) as client_name,
           first_value(market_segment) over (partition by coalesce(holding_name,company_name,lead_name) order by price desc nulls last) as market_segment,
           lead_id,
           price as potential,
           lead_status,
           pipeline_name,
           pipeline_status_name,
           manager_name,
           coalesce(dlv.deliveries_cnt,0) as deliveries_cnt,
           coalesce(dlv.deliveries_express_cnt,0) as deliveries_express_cnt,
           coalesce(dlv.deliveries_sdd_cnt,0) as deliveries_sdd_cnt,
           coalesce(dlv.deliveries_ndd_cnt,0) as deliveries_ndd_cnt,
           coalesce(api_express_flg,False) as api_express_flg,
           coalesce(api_sdd_flg,False) as api_sdd_flg,
           coalesce(api_ndd_flg,False) as api_ndd_flg
    from snb_b2b.amo_leads_info crm
      left join deliveries dlv
        on dlv.corp_contract_id = crm.contract_id
    where manager_metagroup = 'Тир 1'
      and pipeline_name not in ('МАГИСТРАЛЬ','ФЕНИКС','ДОГОВОРЫ ЯНДЕКС ТАКСИ','САМОСТОЯТЕЛЬНЫЕ КЛИЕНТЫ')
)
, clients_raw as (
    select bool_or(lead_status='open'
                    and pipeline_status_name not in ('РАСТОРГНУТ ПО ИНИЦИАТИВЕ КЛИЕНТА','РАСТОРГНУТ ПО ИНИЦИАТИВЕ ЯНДЕКСА')
                    and pipeline_name not in ('ХОЛОД')
               ) as client_open_flg,
           bool_or(pipeline_name in ('ЗАПУСК И РАЗВИТИЕ') and lead_status='open') as products_dash_flg,
           client_name,
           substring(client_name from '^([^/]*)/?') as client_name_short,
           max(market_segment) as market_segment,
           string_agg(lead_id::varchar,',') as lead_id_array,
           string_agg(distinct manager_name,',') as manager_name,
           bool_or(pipeline_name in ('ЗАПУСК И РАЗВИТИЕ')) as launch_flg,
           bool_or(pipeline_status_name in ('ПЕРЕГОВОРЫ', 'ПРИНЯТО РЕШЕНИЕ', 'ЖДЕМ ДОГОВОР')) as negotiations_flg,
           bool_or(pipeline_name in ('ПОДКЛЮЧЕНИЕ')) as connection_start_flg,
           bool_or(pipeline_name in ('ХОЛОД') and lead_status = 'open') as client_cold_open_flg,
           --
           bool_or(api_express_flg) as api_express_flg,
           bool_or(api_sdd_flg) as api_sdd_flg,
           bool_or(api_ndd_flg) as api_ndd_flg,
           --
           sum(potential) as potential,
           sum(deliveries_cnt) as deliveries_cnt,
           sum(deliveries_express_cnt) as deliveries_express_cnt,
           sum(deliveries_sdd_cnt) as deliveries_sdd_cnt,
           sum(deliveries_ndd_cnt) as deliveries_ndd_cnt
    from leads_raw
    group by client_name
)
select client_open_flg,
       products_dash_flg,
       client_name,
       client_name_short,
       coalesce(market_segment, 'N/A') as market_segment,
       lead_id_array, manager_name,
       potential,
       deliveries_cnt, deliveries_express_cnt, deliveries_sdd_cnt, deliveries_ndd_cnt,
       row_number() over (order by potential desc nulls last, deliveries_cnt desc nulls last) as client_rate,
       row_number() over (
           partition by client_open_flg, market_segment
           order by potential desc nulls last, deliveries_cnt desc nulls last) as segment_client_rate,
       case when products_dash_flg then
           row_number() over (
                    partition by client_open_flg, market_segment, products_dash_flg
                    order by potential desc nulls last, deliveries_cnt desc nulls last)
        end as segment_products_client_rate,
       case when client_open_flg then
           -- open wo cold
            case
                when deliveries_cnt >= potential * 0.7 then 1   -- dark green
                when deliveries_cnt > 0 then 2                  -- light green
                when launch_flg then 3                          -- light blue
                when negotiations_flg then 4                    -- yellow
                when connection_start_flg then 5                -- grey
                else 6 end                                      -- red
            -- 7                                                -- purple
            -- 8                                                -- orange
        else
            -- lost or cold
            case
                when client_cold_open_flg then 5
                else 6 end
        end as pipeline_type_id,
       case when deliveries_express_cnt>0 then 1 else 5 end as express_type_id,
       case when deliveries_sdd_cnt>0 then 1 else 5 end as sdd_type_id,
       case when deliveries_ndd_cnt>0 then 1 else 5 end as ndd_type_id,
       case api_express_flg::int + api_sdd_flg::int + api_ndd_flg::int
           when 3 then 1
           when 2 then 2
           when 1 then
            case when api_express_flg then 3
                 when api_sdd_flg then 4
                 when api_ndd_flg then 5 end
           when 0 then 6 end as api_type_id
from clients_raw;

select * from res;
