import os
from datetime import datetime
import uuid

import pandas as pd

from typing import List

import smtplib
from email.mime.text import MIMEText

import yt.wrapper as yt
from business_models.hahn import HahnDataLoader

def send_email(from_email: str, to_email: List[str], 
               email_text: str, email_subject: str) -> None:
    """Функция отправляет имейл от from_email на список адресатов 
    из to_email с текстом email_text и темой email_subject"""
    MAIL_HOST = 'outbound-relay.yandex.net'
    
    msg = MIMEText(email_text, 'html', _charset="utf-8")
    msg['Subject'] = email_subject
    msg['From'] = from_email
    msg['To'] = ', '.join(to_email)

    conn = smtplib.SMTP(MAIL_HOST)
    conn.sendmail(from_email, to_email, msg.as_string())
    conn.quit()
    
def main():
    print('START')
    yt.config['proxy']['url'] = 'hahn.yt.yandex.net'
    
    with open ('projects/projects/cargo/operations/manual_dispatch/claims_selector.sql') as f:
        sql = f.read()
        
    print('> Executing YQL script')
    hahn = HahnDataLoader(token=os.environ['YT_TOKEN'])
    hahn(sql)
    
    print('> YQL script executed')
    
    CLAIMS_PATH = '//home/taxi-delivery/analytics/production/cargo/b2b/manual_dispatch/new_claims'
    PROCESSED_CLAIMS = '//home/taxi-delivery/analytics/production/cargo/b2b/manual_dispatch/processed_claims'
    
    print('> Importing new claims')
    claims = pd.DataFrame(list(yt.read_table(yt.TablePath(CLAIMS_PATH))))
    print('> Claims imported')
    
    print('> Batching claims')
    batch = claims.groupby(by=['corp_client_id', 
                               'point_fullname', 
                               'claim_due']).agg({'claim_uuid': lambda x: set(x)}).reset_index()
    print('> Claims batched')

    cuttent_utc_dttm = datetime.utcnow().strftime('%Y-%m-%d %H:%M:%S')

    
    for i in range(len(batch)):
        batch_row = batch.iloc[i]
        batch_uuid = uuid.uuid4().hex

        batch_due = pd.to_datetime(batch_row['claim_due'], unit='s', origin='unix') + pd.Timedelta(3, unit='h')
        batch_due = batch_due.strftime('%H:%M:%S %d.%m.%Y')
        batch_point = batch_row['point_fullname']
        batch_client = batch_row['corp_client_id']
        batch_cars = len(batch_row['claim_uuid'])

        text = f'''
            НЕ РЕАГИРОВАТЬ НА ЭТОТ ТИКЕТ - ТЕСТИРУЕМ LOGDATA-541 В ПРОДЕ
            Дата/вермя due: {batch_due} (ВРЕМЯ МОСКОВСКОЕ)<br>
            Корп клиент: {batch_client}<br>
            Адрес точки А: {batch_point}<br>
            Количество машин: {batch_cars}<br><br>

            Заявки:<br>
        '''
        
        print(f'> Writing logs for batch {batch_uuid} to {PROCESSED_CLAIMS}')
        
        for claim_uuid in batch_row['claim_uuid']:
            text += f"https://tariff-editor.taxi.yandex-team.ru/corp-claims/show/{claim_uuid}/info<br>"

            yt.write_table(
                yt.TablePath(PROCESSED_CLAIMS, append=True),
                [
                    {
                        "claim_uuid": str(claim_uuid),
                        "claim_due": str(batch_due),
                        "point_fullname": str(batch_point),
                        "corp_client_id": str(batch_client),
                        "processed_utc_dttm": str(cuttent_utc_dttm),
                        "batch_uuid": str(batch_uuid)
                    },
                ],
            )
    
        print(f'> Sending email for batch {batch_uuid}')
        from_ = 'noreply@yandex-team.ru'
        to_ = ['ikhomyanin@yandex-team.ru', 'iksanov@yandex-team.ru', 'b2b-cargo-operations@go.yandex.ru']
        subject = 'LOGDATA-541 ТЕСТ В ПРОДЕ - НЕ ОТРАБАТЫВАТЬ'

        send_email(from_, to_, text, subject)
    print(f'> END')
         
if __name__ == '__main__':
    main()
