import time

import requests
from requests.adapters import HTTPAdapter
from urllib3.util.retry import Retry

from projects.common.clients.tvm import TVMClient


class SkorozvonAPI:
    """
    Skorozvon API for skorozvon needs
    https://app.skorozvon.ru/static/skorozvon/docs/SkorozvonAPI.pdf
    """

    def __init__(self, login, api_key, client_id, client_secret, tvm_secret):
        self.login = login
        self.api_key = api_key
        self.client_id = client_id
        self.client_secret = client_secret
        self.tvm_secret = tvm_secret

    def make_http_session(self):
        session = requests.Session()
        tvm = TVMClient(
            tvm_secret=self.tvm_secret,
            src_id=2027260,
            dsts={'gozora': 2023123},
        )
        tvm_ticket = tvm.get_ticket(dst_service_name='gozora')
        session.proxies = {
            'https': f'http://logdata:{tvm_ticket}@go.zora.yandex.net:1080',
        }
        retry = Retry(
            total=10,
            backoff_factor=0.1,
            status_forcelist=(500, 502, 503, 504),
            allowed_methods=('POST',),
        )
        session.mount('https://', HTTPAdapter(max_retries=retry))
        self.update_token(session)
        return session

    def update_token(self, session):
        response = session.post(
            url='https://app.skorozvon.ru/oauth/token',
            data=dict(
                grant_type='password',
                username=self.login,
                api_key=self.api_key,
                client_id=self.client_id,
                client_secret=self.client_secret,
            ),
            timeout=1,
            verify=False,
        )
        access_token = response.json()['access_token']
        session.headers = {'Authorization': f'Bearer {access_token}'}

    def iterate_calls_total(self, start_time=0, end_time=None):
        end_time = end_time or int(time.time())
        session = self.make_http_session()
        page = 1
        total_pages = None

        while total_pages is None or page <= total_pages:
            r = session.post(
                url='https://app.skorozvon.ru/api/reports/calls_total.json',
                json=dict(
                    length=100,
                    start_time=start_time,
                    end_time=end_time,
                    page=page,
                ),
                timeout=5,
                verify=False,
            )

            if r.status_code == 401:
                self.update_token(session)
                continue

            json_data = r.json()

            total_pages = json_data['total_pages']
            print(f'Loaded page {page} out of {total_pages}')
            for item in json_data['data']:
                yield item
            page += 1
