import sys

from typing import Optional, Dict

from yql.client.parameter_value_builder import (
    YqlParameterValueBuilder as ValueBuilder,
)
from yql.api.v1.client import YqlClient


def generate_yql_params(parameters: Dict[str, str]):
    """make yql param dict passable to yql client"""

    assert all(
        key.startswith('$') for key in parameters.keys()
    ), 'yql params must start with "$"'

    yql_params = {
        key: ValueBuilder.make_string(value=value)
        for key, value in parameters.items()
    }
    return yql_params


def run_yql(
        query: str,
        parameters: Optional[Dict[str, str]] = None,
        files: Optional[Dict[str, str]] = None,
):
    """
    run provided query with parameters
    :param query: string with query
    :param parameters: dict with parameters
    :param files: dict {name: content}
    """

    parameter_values = generate_yql_params(parameters) if parameters else {}

    client = YqlClient()
    request = client.query(query=query, syntax_version=1)
    files = files or {}

    for name, content in files.items():
        request.attach_file_data(data=content, name=name)

    request.run(parameters=ValueBuilder.build_json_map(parameter_values))

    results = request.get_results()
    if results.errors:
        for e in results.errors:
            print(e, file=sys.stderr)
        raise Exception('Errors during YQL')

    if results.status != 'COMPLETED':
        print(results.status, file=sys.stderr)
        raise Exception('Not success YQL')
