# coding=utf-8

import numpy as np
from typing import Union

EARTH_RADIUS_METERS = 6372795.0
FULL_CIRCLE_DEGREES = 360.0


def earth_distance(
        lat1: Union[float, np.array],
        lon1: Union[float, np.array],
        lat2: Union[float, np.array],
        lon2: Union[float, np.array],
        is_radians: bool = False,
) -> Union[float, np.array]:
    """
    Вычисление расстояния между точками заданными географическими координатами
    Если входные пареметры массивы, то возвращается расстояния первый с
    первым и тд
    """
    if not is_radians:
        lon1, lat1, lon2, lat2 = map(np.radians, (lon1, lat1, lon2, lat2))
    return (
        EARTH_RADIUS_METERS
        * 2.0
        * np.arcsin(
            np.sqrt(
                np.sin((lat2 - lat1) / 2.0) ** 2
                + np.cos(lat1)
                * np.cos(lat2)
                * np.sin((lon2 - lon1) / 2.0) ** 2,
            ),
        )
    )


def earth_pairwise_distance(
        lat1: np.ndarray,
        lon1: np.ndarray,
        lat2: np.ndarray,
        lon2: np.ndarray,
        is_radians: bool = False,
) -> np.ndarray:
    """
    Вычисление попарного расстояния между точками  заданными географическими
    координатами
    """
    if not is_radians:
        lon1, lat1, lon2, lat2 = map(np.radians, (lon1, lat1, lon2, lat2))
    dlon = lon2[np.newaxis, :] - lon1[:, np.newaxis]
    dlat = lat2[np.newaxis, :] - lat1[:, np.newaxis]
    xdist = np.sin(dlat / 2.0)
    ydist = np.sin(dlon / 2.0)
    ycoeff = np.cos(lat1)[:, np.newaxis] * np.cos(lat2)[np.newaxis, :]
    return (
        EARTH_RADIUS_METERS
        * 2.0
        * np.arcsin(np.sqrt(xdist ** 2 + ycoeff * ydist ** 2))
    )


def lonlat_to_cartesian(
        lat: Union[float, np.array], lon: Union[float, np.array],
):
    """
    converts lon/lat coordinates to cartesian 3d coordinates
    """
    lon = np.radians(lon)
    lat = np.radians(lat)
    x = EARTH_RADIUS_METERS * np.cos(lat) * np.cos(lon)
    y = EARTH_RADIUS_METERS * np.cos(lat) * np.sin(lon)
    z = EARTH_RADIUS_METERS * np.sin(lat)
    return x, y, z


def angle_clockwise(
        lat1: Union[float, np.array],
        lon1: Union[float, np.array],
        lat2: Union[float, np.array],
        lon2: Union[float, np.array],
        is_radians: bool = False,
) -> Union[float, np.array]:
    """
    clockwise angle for vector from 1st point to 2nd point
    north vector is reference
    """
    if not is_radians:
        lon1, lat1, lon2, lat2 = map(np.radians, (lon1, lat1, lon2, lat2))
    dlon = lon2 - lon1
    x = np.cos(lat1) * np.sin(lat2) - np.sin(lat1) * np.cos(lat2) * np.cos(
        dlon,
    )
    y = np.sin(dlon) * np.cos(lat2)
    angle = np.degrees(np.arctan2(y, x))
    return (angle + FULL_CIRCLE_DEGREES) % FULL_CIRCLE_DEGREES


def shift_coordinates(
        lat: Union[float, np.array],
        lon: Union[float, np.array],
        angle: Union[float, np.array],
        distance: Union[float, np.array],
        is_radians: bool = False,
):
    if not is_radians:
        lon, lat, angle = map(np.radians, (lon, lat, angle))
    d = distance / EARTH_RADIUS_METERS
    lat_shifted = np.arcsin(
        np.sin(lat) * np.cos(d) + np.cos(lat) * np.sin(d) * np.cos(angle),
    )
    lon_shifted = lon + np.arctan2(
        np.sin(angle) * np.sin(d) * np.cos(lat),
        np.cos(d) - np.sin(lat) * np.sin(lat_shifted),
    )
    if not is_radians:
        return np.degrees(lon_shifted), np.degrees(lat_shifted)
    return lon_shifted, lat_shifted
