from typing import Dict, Iterable, List, Set
import os
import sys
import platform
import pkg_resources
import subprocess

SCRIPT_DIR = os.path.dirname(os.path.realpath(__file__))
CACHE_DIR = os.path.join(
    SCRIPT_DIR,
    '.wheels_cache',
    'py' + '.'.join(map(str, sys.version_info[:2])),
    platform.system(),
)


def get_wheel_paths() -> Dict[str, str]:
    # to hide dependency on nile
    import wheel_inspect

    os.makedirs(CACHE_DIR, exist_ok=True)
    result = dict()
    for name in os.listdir(CACHE_DIR):
        if name.endswith('whl'):
            info = wheel_inspect.parse_wheel_filename(name)
            result[
                f'{pkg_resources.to_filename(info.project)}-{info.version}'
            ] = name
    return result


def _dfs(name: str, dependencies: Set[str]) -> None:
    dist = pkg_resources.get_distribution(name)
    key = f'{pkg_resources.to_filename(dist.project_name)}-{dist.version}'
    if key not in dependencies:
        dependencies.add(key)
        for requirement in dist.requires():
            _dfs(requirement.name, dependencies)


def get_deps_tree(names: Iterable[str]) -> Set[str]:
    dependencies = set()
    for name in names:
        _dfs(name, dependencies)
    return dependencies


def build_venv_packages(
        names: List[str],
        wheels_dir: str,
        verbose: bool = False,
        no_deps: bool = True,
) -> List[str]:
    # to hide dependency on nile
    from pip._internal.operations import freeze

    constraints_path = os.path.join(wheels_dir, 'constraints.txt')
    with open(constraints_path, 'w') as constraints_file:
        constraints_file.write('\n'.join(freeze.freeze(exclude_editable=True)))
    kwargs = dict()
    if not verbose:
        kwargs.update(
            stdout=open(os.devnull, 'wb'), stderr=open(os.devnull, 'wb'),
        )

    subprocess.check_call(
        [sys.executable]
        + ['-m', 'pip', 'wheel']
        + ['-i', 'https://pypi.yandex-team.ru/simple']
        + (['--no-deps'] if no_deps else [])
        + ['--no-cache-dir', '--default-timeout=180']
        + names
        + ['-w', wheels_dir]
        + ['-c', constraints_path],
        **kwargs,
    )
    return names


def build_deps_for_venv_packages(
        names: Iterable[str], verbose: bool = False,
) -> List[str]:
    """
    :param names: names of packages to get wheels
    :param verbose: flag to log when building wheels
    :return: list of packages with dependencies and versions
    Builds wheels if needed and caches them in CACHE_DIR
    """
    deps = get_deps_tree(names)
    wheel_paths = get_wheel_paths()
    keys = [key for key in deps if key not in wheel_paths]
    if keys:
        build_venv_packages(
            names=[key.split('-')[0] for key in keys],
            wheels_dir=CACHE_DIR,
            verbose=verbose,
            no_deps=True,
        )
    return list(deps)
