# -*- coding: utf-8 -*-
import math
import datetime

import haversine
import numpy as np
from sklearn.metrics import pairwise_distances
from scipy.spatial.distance import pdist
from geopy import distance as gp_distance

from projects.common import prof, geo

COURIER_SPEED_MAP = {
    'bicycle': 2.201287,
    'electric_bicycle': 2.810285,
    'motorcycle': 3.497776,
    'pedestrian': 1.572080,
    'vehicle': 3.173605,
    'cargo_s': 3.173605,
    'cargo_m': 3.173605,
    'cargo_l': 3.173605,
}
COURIER_MAX_VOLUME = {
    'bicycle': 40 * 25 * 40,
    'electric_bicycle': 40 * 25 * 40,
    'motorcycle': 40 * 25 * 40,
    'pedestrian': 40 * 25 * 40,
    'vehicle': 40 * 25 * 40,
    'cargo_s': 170 * 100 * 90,
    'cargo_m': 260 * 160 * 150,
    'cargo_l': 400 * 190 * 200,
}
COURIER_MAX_WEIGHT = {
    'bicycle': 10,
    'electric_bicycle': 10,
    'motorcycle': 10,
    'pedestrian': 10,
    'vehicle': 30,
    'cargo_s': np.inf,
    'cargo_m': np.inf,
    'cargo_l': np.inf,
}
COURIER_MAX_DELIVERIES_MAP = {
    'bicycle': 3,
    'electric_bicycle': 3,
    'motorcycle': 3,
    'pedestrian': 3,
    'vehicle': 5,
    'cargo_s': np.inf,
    'cargo_m': np.inf,
    'cargo_l': np.inf,
}
COURIER_MAX_TRAVEL_TIME_MAP = {
    'bicycle': 14400,
    'electric_bicycle': 14400,
    'motorcycle': 14400,
    'pedestrian': 14400,
    'vehicle': 28800,
    'cargo_s': 28800,
    'cargo_m': 28800,
    'cargo_l': 28800,
}
COURIER_DIST_COEF_MAP = {
    'bicycle': 1.408532,
    'electric_bicycle': 1.408532,
    'motorcycle': 2.017683,
    'pedestrian': 1.400771,
    'vehicle': 2.017683,
    'cargo_s': 2.017683,
    'cargo_m': 2.017683,
    'cargo_l': 2.017683,
}
COURIER_SH_COST = {
    'bicycle': 220,
    'electric_bicycle': 220,
    'motorcycle': 220,
    'pedestrian': 220,
    'vehicle': 375,
    'cargo_s': 500,
    'cargo_m': 500,
    'cargo_l': 500,
}

CARGO_HOURS_PAY_MAP = {
    2: 999,
    3: 1399,
    4: 1799,
    5: 2199,
    6: 2599,
    7: 2999,
    8: 3399,
}
AVERAGE_SURGE = 1.2
DOOR_TO_DOOR_PP_TO_USER = True
COURIER_HALT_COEF = 1.5
COURIER_BOARDING_SEC = 7 * 60
COURIER_TAKE_SEC = 2 * 60


def calculate_service_cost(
        min_cost,
        include_dist,
        include_time,
        km_fare,
        minute_fare,
        comission,
        inform_cost,
        nmfg,
        door_to_door,
        cargo_coef,
        cargo_flg,
        courier_dict_coef,
        dist_in_km,
        time_in_minutes,
        surge,
        door_to_door_flg,
):
    pay_dist = max(0, dist_in_km * courier_dict_coef - include_dist)
    pay_time = max(0, time_in_minutes - include_time)

    user_base_cost = (
        min_cost
        + pay_dist * km_fare
        + pay_time * minute_fare
        + door_to_door * door_to_door_flg
    )
    if cargo_flg:
        hours = round(math.ceil(time_in_minutes / 60))
        hours_pay = CARGO_HOURS_PAY_MAP.get(hours, np.inf)
        user_base_cost = min(hours_pay, user_base_cost)

    user_cost = surge * user_base_cost * cargo_coef
    service_cost = (
        max(nmfg, user_cost) * (1 - (comission * 1.2)) - inform_cost * 1.2
    )
    return service_cost


COURIER_PAY_FUNCTION = {
    'bicycle': (
        lambda dist_in_km, time_in_minutes, surge, door_to_door_flg: calculate_service_cost(
            min_cost=155,
            include_dist=2.5,
            include_time=0,
            km_fare=16,
            minute_fare=0,
            comission=0.18,
            inform_cost=5.5,
            nmfg=0,
            door_to_door=0,
            cargo_coef=1,
            cargo_flg=False,
            courier_dict_coef=COURIER_DIST_COEF_MAP['bicycle'],
            dist_in_km=dist_in_km,
            time_in_minutes=time_in_minutes,
            surge=surge,
            door_to_door_flg=door_to_door_flg,
        )
    ),
    'electric_bicycle': (
        lambda dist_in_km, time_in_minutes, surge, door_to_door_flg: calculate_service_cost(
            min_cost=155,
            include_dist=2.5,
            include_time=0,
            km_fare=16,
            minute_fare=0,
            comission=0.18,
            inform_cost=5.5,
            nmfg=300,
            door_to_door=0,
            cargo_coef=1,
            cargo_flg=False,
            courier_dict_coef=COURIER_DIST_COEF_MAP['electric_bicycle'],
            dist_in_km=dist_in_km,
            time_in_minutes=time_in_minutes,
            surge=surge,
            door_to_door_flg=door_to_door_flg,
        )
    ),
    'motorcycle': (
        lambda dist_in_km, time_in_minutes, surge, door_to_door_flg: calculate_service_cost(
            min_cost=155,
            include_dist=2.5,
            include_time=0,
            km_fare=16,
            minute_fare=0,
            comission=0.18,
            inform_cost=5.5,
            nmfg=300,
            door_to_door=0,
            cargo_coef=1,
            cargo_flg=False,
            courier_dict_coef=COURIER_DIST_COEF_MAP['motorcycle'],
            dist_in_km=dist_in_km,
            time_in_minutes=time_in_minutes,
            surge=surge,
            door_to_door_flg=door_to_door_flg,
        )
    ),
    'pedestrian': (
        lambda dist_in_km, time_in_minutes, surge, door_to_door_flg: calculate_service_cost(
            min_cost=155,
            include_dist=2.5,
            include_time=0,
            km_fare=16,
            minute_fare=0,
            comission=0.18,
            inform_cost=5.5,
            nmfg=300,
            door_to_door=0,
            cargo_coef=1,
            cargo_flg=False,
            courier_dict_coef=COURIER_DIST_COEF_MAP['pedestrian'],
            dist_in_km=dist_in_km,
            time_in_minutes=time_in_minutes,
            surge=surge,
            door_to_door_flg=door_to_door_flg,
        )
    ),
    'vehicle': (
        lambda dist_in_km, time_in_minutes, surge, door_to_door_flg: calculate_service_cost(
            min_cost=165,
            include_dist=3,
            include_time=5,
            km_fare=11,
            minute_fare=9,
            comission=0.18,
            inform_cost=5.5,
            nmfg=300,
            door_to_door=150,
            cargo_coef=1,
            cargo_flg=False,
            courier_dict_coef=COURIER_DIST_COEF_MAP['vehicle'],
            dist_in_km=dist_in_km,
            time_in_minutes=time_in_minutes,
            surge=surge,
            door_to_door_flg=door_to_door_flg,
        )
    ),
    'cargo_s': (
        lambda dist_in_km, time_in_minutes, surge, door_to_door_flg: calculate_service_cost(
            min_cost=699,
            include_dist=5,
            include_time=10,
            km_fare=2,
            minute_fare=9,
            comission=0.18,
            inform_cost=5.5,
            nmfg=1000,
            door_to_door=0,
            cargo_coef=1,
            cargo_flg=True,
            courier_dict_coef=COURIER_DIST_COEF_MAP['cargo_s'],
            dist_in_km=dist_in_km,
            time_in_minutes=time_in_minutes,
            surge=surge,
            door_to_door_flg=door_to_door_flg,
        )
    ),
    'cargo_m': (
        lambda dist_in_km, time_in_minutes, surge, door_to_door_flg: calculate_service_cost(
            min_cost=699,
            include_dist=5,
            include_time=10,
            km_fare=2,
            minute_fare=9,
            comission=0.18,
            inform_cost=5.5,
            nmfg=1000,
            door_to_door=0,
            cargo_coef=1.3,
            cargo_flg=True,
            courier_dict_coef=COURIER_DIST_COEF_MAP['cargo_m'],
            dist_in_km=dist_in_km,
            time_in_minutes=time_in_minutes,
            surge=surge,
            door_to_door_flg=door_to_door_flg,
        )
    ),
    'cargo_l': (
        lambda dist_in_km, time_in_minutes, surge, door_to_door_flg: calculate_service_cost(
            min_cost=699,
            include_dist=5,
            include_time=10,
            km_fare=2,
            minute_fare=9,
            comission=0.18,
            inform_cost=5.5,
            nmfg=1000,
            door_to_door=0,
            cargo_coef=1.6,
            cargo_flg=True,
            courier_dict_coef=COURIER_DIST_COEF_MAP['cargo_l'],
            dist_in_km=dist_in_km,
            time_in_minutes=time_in_minutes,
            surge=surge,
            door_to_door_flg=door_to_door_flg,
        )
    ),
}


class Delivery(object):
    def __init__(
            self,
            delivery_id,
            service_name,
            weight,
            volume,
            delivery_lat,
            delivery_lon,
            warehouse_lat,
            warehouse_lon,
            warehouse_id,
            delivery_plan_dttm,
            warehouse_ready_dttm,
            order_id_source,
    ):
        # Known before start parameters.
        self.id = delivery_id
        self.service_name = service_name
        self.location_lat = delivery_lat
        self.location_lon = delivery_lon
        self.warehouse_lat = warehouse_lat
        self.warehouse_lon = warehouse_lon
        self.weight = weight
        self.volume = volume
        self.delivery_plan_dttm = delivery_plan_dttm
        self.warehouse_ready_dttm = warehouse_ready_dttm
        self.warehouse_id = warehouse_id
        self.order_id_source = order_id_source
        # Parameters TBD over the course of simulation.
        self.sorting_center_id = None
        self.pickup_point_id = None
        self.pp_to_user_batched_flg = False
        self.delivered_flg = False
        self.courier_id = None
        self.pp_to_user_start_dttm = None
        self.pp_to_user_end_dttm = None
        self.size_group = None
        self.not_delivered_reason = None


class Courier(object):
    def __init__(self, courier_id, courier_type):
        self.id = courier_id
        self.type = courier_type

        self.stored_volume = 0
        self.stored_weight = 0
        self.distance_travelled = 0
        self.time_travelled = 0
        self.deliveries_made = 0
        self.payout = 0


# class SortingCenter(object):
#     def __init__(self, sorting_center_id, lat, lon):
#         self.id = sorting_center_id
#         self.lat = lat
#         self.lon = lon


# class Warehouse(object):
#     def __init__(self, warehouse_id, lat, lon):
#         self.id = warehouse_id
#         self.lat = lat
#         self.lon = lon


class PickupPoint(object):
    def __init__(
            self,
            pickup_point_id,
            lat,
            lon,
            take_start_tm,
            take_end_tm,
            capacity,
            pp_type,
            address,
            sla_sec,
            delivery_product_type,
    ):
        self.id = pickup_point_id
        self.lat = lat
        self.lon = lon
        self.take_start_tm = take_start_tm
        self.take_end_tm = take_end_tm
        self.stored_deliveries = 0
        self.potential_deliveries = 0
        self.closest_deliveries = 0
        self.capacity = capacity
        self.pp_type = pp_type
        self.address = address
        self.sla_sec = sla_sec
        self.potential_pickup_point_id = None
        self.delivery_product_type = delivery_product_type


class Simulation(object):
    def __init__(self, pp_work_time):
        self.pp_work_time = pp_work_time

        self.deliveries = []
        self.deliveries_dict = {}
        self.couriers = []
        self.couriers_dict = {}
        self.sorting_centers = []
        self.sorting_centers_dict = {}
        self.warehouses = []
        self.warehouses_dict = {}
        self.pickup_points = []
        self.pickup_points_dict = {}

        self.time_storage = {}

    def initialize_deliveries(self, deliveries_df):
        with prof.TimedContext('initialize_deliveries', self.time_storage):
            for row in deliveries_df.itertuples():
                delivery = Delivery(
                    delivery_id=len(self.deliveries),
                    service_name=row.service_name,
                    weight=row.weight,
                    volume=row.volume,
                    delivery_lat=row.delivery_lat,
                    delivery_lon=row.delivery_lon,
                    warehouse_lat=row.warehouse_lat,
                    warehouse_lon=row.warehouse_lon,
                    warehouse_id=row.warehouse_id,
                    delivery_plan_dttm=row.delivery_plan_dttm,
                    warehouse_ready_dttm=row.warehouse_ready_dttm,
                    order_id_source=row.order_id_source,
                )
                delivery.size_group = (
                    'small'
                    if delivery.weight <= 10
                    and delivery.volume <= 40 * 25 * 40
                    else 'large'
                )
                self.deliveries.append(delivery)
                self.deliveries_dict[delivery.id] = delivery

    # def initialize_sorting_centers(self, sorting_centers_df):
    #     with prof.TimedContext('initialize_sorting_centers', self.time_storage):
    #         for row in sorting_centers_df.itertuples():
    #             sorting_center = SortingCenter()
    #             self.sorting_centers.append(sorting_center)
    #             self.sorting_centers_dict[sorting_center.id] = sorting_center

    # def initialize_warehouses(self, warehouses_df):
    #     with prof.TimedContext('initialize_warehouses', self.time_storage):
    #         for row in warehouses_df.itertuples():
    #             warehouse = Warehouse()
    #             self.warehouses.append(warehouse)
    #             self.warehouses_dict[warehouse.id] = warehouse

    def initialize_pickup_points(self, pickup_points_df):
        with prof.TimedContext('initialize_pickup_points', self.time_storage):
            for row in pickup_points_df.itertuples():
                pickup_point = PickupPoint(
                    pickup_point_id=row.pickup_point_id,
                    lat=row.lat,
                    lon=row.lon,
                    take_start_tm=row.take_start_tm,
                    take_end_tm=row.take_end_tm,
                    capacity=row.capacity,
                    pp_type=row.pp_type,
                    address=row.address,
                    sla_sec=row.sla_sec,
                    delivery_product_type=row.delivery_product_type,
                )
                self.pickup_points.append(pickup_point)
                self.pickup_points_dict[pickup_point.id] = pickup_point

    def initialize_simulation(
            self,
            deliveries_df,
            pickup_points_df,
            sorting_centers_df=None,
            warehouses_df=None,
    ):
        with prof.TimedContext('initialize_simulation', self.time_storage):
            self.initialize_deliveries(deliveries_df)
            self.initialize_pickup_points(pickup_points_df)
            # self.initialize_sorting_centers(sorting_centers_df)
            # self.initialize_warehouses(warehouses_df)
        print(
            'Initialization finished in {}'.format(
                self.time_storage['initialize_simulation'],
            ),
        )

    def assign_deliveries_to_pickup_points(self):
        with prof.TimedContext(
                'assign_deliveries_to_pickup_point', self.time_storage,
        ):
            # Assign small deliveries.
            small_deliveries = [
                delivery
                for delivery in self.deliveries
                if delivery.size_group == 'small'
            ]
            large_deliveries = [
                delivery
                for delivery in self.deliveries
                if delivery.size_group == 'large'
            ]
            X_small = np.array(
                [
                    [delivery.location_lat, delivery.location_lon]
                    for delivery in small_deliveries
                ],
            )
            X_large = np.array(
                [
                    [delivery.location_lat, delivery.location_lon]
                    for delivery in large_deliveries
                ],
            )
            Y_all = np.array([[pp.lat, pp.lon] for pp in self.pickup_points])
            Y_pvz = np.array(
                [
                    [pp.lat, pp.lon]
                    for pp in self.pickup_points
                    if pp.pp_type == 'pvz'
                ],
            )
            pairwise_dists_small = (
                pairwise_distances(
                    X=X_small,
                    Y=Y_all,
                    metric=haversine.haversine,
                    n_jobs=-1,
                    force_all_finite=True,
                )
                * 1000.0
            )
            pairwise_dists_large = (
                pairwise_distances(
                    X=X_large,
                    Y=Y_pvz,
                    metric=haversine.haversine,
                    n_jobs=-1,
                    force_all_finite=True,
                )
                * 1000.0
            )
            closest_pp_index_small = pairwise_dists_small.argsort(axis=1)[
                :, :2,
            ].tolist()
            closest_pp_index_large = pairwise_dists_large.argsort(axis=1)[
                :, :2,
            ].tolist()

            for delivery, pp_index_list in zip(
                    small_deliveries, closest_pp_index_small,
            ):
                for c, i in enumerate(pp_index_list):
                    if c == 0:
                        self.pickup_points[i].closest_deliveries += 1
                    self.pickup_points[i].potential_deliveries += 1
                    delivery.potential_pickup_point_id = self.pickup_points[
                        i
                    ].id
                    if (
                            self.pickup_points[i].stored_deliveries
                            < self.pickup_points[i].capacity
                    ):
                        delivery.pickup_point_id = self.pickup_points[i].id
                        self.pickup_points[i].stored_deliveries += 1
                        break
                    else:
                        if c != len(pp_index_list) - 1:
                            self.pickup_points[i].potential_deliveries -= 1
                if not delivery.pickup_point_id:
                    delivery.not_delivered_reason = 'capacity'

            for delivery, pp_index_list in zip(
                    large_deliveries, closest_pp_index_large,
            ):
                pvz_pp = [
                    pp for pp in self.pickup_points if pp.pp_type == 'pvz'
                ]
                for c, i in enumerate(pp_index_list):
                    if c == 0:
                        pvz_pp[i].closest_deliveries += 1
                    pvz_pp[i].potential_deliveries += 1
                    delivery.potential_pickup_point_id = pvz_pp[i].id
                    if pvz_pp[i].stored_deliveries < pvz_pp[i].capacity:
                        delivery.pickup_point_id = pvz_pp[i].id
                        pvz_pp[i].stored_deliveries += 1
                        break
                    else:
                        if c != len(pp_index_list) - 1:
                            pvz_pp[i].potential_deliveries -= 1
                if not delivery.pickup_point_id:
                    delivery.not_delivered_reason = 'capacity'

    # def assign_pickup_points_to_sorting_centers(self):
    #     with prof.TimedContext('assign_pickup_points_to_sorting_centers', self.time_storage):
    #         X = np.array([(pp.lat, pp.lon) for pp in self.pickup_points])
    #         Y = np.array([(sc.lat, sc.lon) for sc in self.sorting_centers])
    #         metric = lambda p1, p2: gp_distance.distance(p1, p2).meters

    #         pairwise_dists = pairwise_distances(
    #             X=X, Y=Y, metric=metric, n_jobs=8, force_all_finite=True,
    #         )
    #         closest_sc_index = pairwise_dists.argmin(axis=1).tolist()

    #         for pp, sc_index in zip(self.pickup_points, closest_sc_index):
    #             pp.sorting_center_id = self.sorting_centers[sc_index].id

    # def assign_deliveries_to_sorting_centers(self):
    #     with prof.TimedContext('assign_deliveries_to_sorting_centers', self.time_storage):
    #         for delivery in self.deliveries:
    #             delivery.sorting_center_id = (
    #                 self.pickup_points_dict[delivery.pickup_point_id].sorting_center_id
    #             )

    def assign_objects(self):
        with prof.TimedContext('assign_objects', self.time_storage):
            self.assign_deliveries_to_pickup_points()
            # self.assign_pickup_points_to_sorting_centers()
            # self.assign_deliveries_to_sorting_centers()
        print(
            'Assigning finished in {}'.format(
                self.time_storage['assign_objects'],
            ),
        )

    def spawn_courier(self, courier_type):
        courier = Courier(
            courier_id=len(self.couriers), courier_type=courier_type,
        )
        self.couriers.append(courier)
        self.couriers_dict[courier.id] = courier
        return courier

    def batching_allowed(
            self, delivery_a, delivery_b, courier, point_pp, pickup_point,
    ):
        # If already batched then why bother.
        if delivery_b.pp_to_user_batched_flg:
            return False

        # If wait between points is too long there is no need to batch deliveries.
        max_allowed_time = (
            delivery_b.delivery_plan_dttm - delivery_a.delivery_plan_dttm
        ).total_seconds()
        if max_allowed_time > pickup_point.sla_sec:
            return False

        point_a = (delivery_a.location_lat, delivery_a.location_lon)
        point_b = (delivery_b.location_lat, delivery_b.location_lon)

        dist_a_to_b = gp_distance.distance(point_a, point_b).meters
        time_a_to_b = dist_a_to_b / COURIER_SPEED_MAP[courier.type]

        #########
        if pickup_point.delivery_product_type == 'ondemand':
            dist_pp_to_b = gp_distance.distance(point_pp, point_b).meters
            time_pp_to_b = (
                dist_pp_to_b / COURIER_SPEED_MAP[courier.type]
                + COURIER_BOARDING_SEC
                + COURIER_TAKE_SEC
            )
            delivery_b.time_pp_to_user = time_pp_to_b
            time_b_known = delivery_b.delivery_plan_dttm - datetime.timedelta(
                seconds=time_pp_to_b,
            )
            if time_b_known > delivery_a.pp_to_user_start_dttm:
                return False
        #########

        if (
                courier.deliveries_made + 1
                > COURIER_MAX_DELIVERIES_MAP[courier.type]
        ):
            return False
        if (
                courier.time_travelled + time_a_to_b
                > COURIER_MAX_TRAVEL_TIME_MAP[courier.type]
        ):
            return False
        if (
                courier.stored_weight + delivery_b.weight
                > COURIER_MAX_WEIGHT[courier.type]
        ):
            return False
        if (
                courier.stored_volume + delivery_b.volume
                > COURIER_MAX_VOLUME[courier.type]
        ):
            return False

        if time_a_to_b <= max_allowed_time:
            return True

        return False

    def batch_deliveries_on_pickup_point(self, pickup_point):
        pickup_point_deliveries = list(
            sorted(
                filter(
                    lambda delivery: delivery.pickup_point_id
                    == pickup_point.id,
                    self.deliveries,
                ),
                key=lambda delivery: delivery.delivery_plan_dttm,
            ),
        )
        point_pp = (pickup_point.lat, pickup_point.lon)

        for i, delivery in enumerate(pickup_point_deliveries):
            if delivery.pp_to_user_batched_flg:
                continue

            if delivery.size_group == 'large':
                courier = self.spawn_courier(courier_type='vehicle')
            else:
                courier = self.spawn_courier(courier_type='bicycle')
            delivery_a = delivery
            point_a = (delivery_a.location_lat, delivery_a.location_lon)

            dist_pp_to_user = gp_distance.distance(point_pp, point_a).meters
            time_pp_to_user = (
                dist_pp_to_user / COURIER_SPEED_MAP[courier.type]
                + COURIER_BOARDING_SEC
                + COURIER_TAKE_SEC
            )
            time_pp_to_user_driving = (
                dist_pp_to_user / COURIER_SPEED_MAP[courier.type]
            )
            delivery_a.time_pp_to_user = time_pp_to_user
            if time_pp_to_user > pickup_point.sla_sec:
                delivery.not_delivered_reason = 'time_to_user'
                continue

            delivery_a.pp_to_user_start_dttm = (
                delivery_a.delivery_plan_dttm
                - datetime.timedelta(seconds=time_pp_to_user_driving)
            )

            delivery_a.pp_to_user_end_dttm = delivery_a.delivery_plan_dttm
            delivery_a.delivered_flg = True
            delivery_a.courier_id = courier.id

            courier.distance_travelled += dist_pp_to_user
            courier.time_travelled += time_pp_to_user
            courier.deliveries_made += 1
            courier.stored_weight += delivery_a.weight
            courier.stored_volume += delivery_a.volume

            for delivery_b in pickup_point_deliveries[i + 1 :]:
                point_b = (delivery_b.location_lat, delivery_b.location_lon)

                if self.batching_allowed(
                        delivery_a,
                        delivery_b,
                        courier,
                        point_pp,
                        pickup_point,
                ):
                    delivery_a.pp_to_user_batched_flg = True
                    delivery_b.pp_to_user_batched_flg = True
                    dist_a_to_b = haversine.haversine(point_a, point_b) * 1000
                    time_a_to_b = dist_a_to_b / COURIER_SPEED_MAP[courier.type]

                    delivery_b.pp_to_user_start_dttm = (
                        delivery_a.pp_to_user_start_dttm
                    )
                    delivery_b.pp_to_user_end_dttm = (
                        delivery_a.pp_to_user_end_dttm
                        + datetime.timedelta(seconds=time_a_to_b)
                    )
                    delivery_b.delivered_flg = True
                    delivery_b.courier_id = courier.id

                    courier.distance_travelled += dist_a_to_b
                    courier.time_travelled += time_a_to_b
                    courier.deliveries_made += 1
                    courier.stored_weight += delivery_b.weight
                    courier.stored_volume += delivery_b.volume

    def batch_deliveries_pp_to_user(self):
        with prof.TimedContext(
                'batch_deliveries_pp_to_user', self.time_storage,
        ):
            for pickup_point in self.pickup_points:
                self.batch_deliveries_on_pickup_point(pickup_point)
        print(
            'Batching finished in {}'.format(
                self.time_storage['batch_deliveries_pp_to_user'],
            ),
        )

    def pay_courier(self, courier, surge, door_to_door_flg):
        courier.payout = COURIER_PAY_FUNCTION[courier.type](
            dist_in_km=courier.distance_travelled / 1000,
            time_in_minutes=courier.time_travelled / 60,
            surge=surge,
            door_to_door_flg=door_to_door_flg,
        )

    def pay_couriers(self):
        for courier in self.couriers:
            self.pay_courier(
                courier,
                surge=AVERAGE_SURGE,
                door_to_door_flg=DOOR_TO_DOOR_PP_TO_USER,
            )

    def run_simulation(self):
        self.batch_deliveries_pp_to_user()
        self.pay_couriers()

    def calculate_metrics(self, deliveries, pickup_points):
        success_deliveries = list(
            filter(lambda x: x.delivered_flg, deliveries),
        )
        couriers = [
            self.couriers_dict[cid]
            for cid in set([d.courier_id for d in deliveries if d.courier_id])
        ]

        self.metrics = {}
        self.metrics['closest_deliveries_cnt'] = sum(
            [pp.closest_deliveries for pp in pickup_points],
        )
        self.metrics['potential_deliveries_cnt'] = sum(
            [pp.potential_deliveries for pp in pickup_points],
        )
        self.metrics['success_deliveries_cnt'] = len(success_deliveries)
        self.metrics['%_success_deliveries'] = (
            self.metrics['success_deliveries_cnt']
            / self.metrics['potential_deliveries_cnt']
        )
        self.metrics['success_deliveries_bicycle_cnt'] = sum(
            [c.deliveries_made for c in couriers if c.type == 'bicycle'],
        )
        self.metrics['success_deliveries_vehicle_cnt'] = sum(
            [c.deliveries_made for c in couriers if c.type == 'vehicle'],
        )
        self.metrics['failed_deliveries_due_to_capacity'] = sum(
            [
                pp.potential_deliveries - pp.stored_deliveries
                for pp in pickup_points
            ],
        )
        # self.metrics['failed_deliveries_due_to_capacity'] = len(
        #     [d for d in deliveries if d.not_delivered_reason == 'capacity']
        # )
        self.metrics['failed_deliveries_due_to_large_radius'] = len(
            [
                d
                for d in deliveries
                if d.not_delivered_reason == 'time_to_user'
            ],
        )
        self.metrics['success_trips_cnt'] = len(
            set([d.courier_id for d in success_deliveries]),
        )

        self.metrics['fraction_of_batched_deliveries'] = (
            len(
                list(
                    filter(
                        lambda x: x.delivered_flg and x.pp_to_user_batched_flg,
                        success_deliveries,
                    ),
                ),
            )
            / self.metrics['success_deliveries_cnt']
        )

        self.metrics['utilized_pickup_points_cnt'] = 0
        self.metrics['all_pickup_points_cnt'] = len(pickup_points)
        for pp in self.pickup_points:
            pickup_point_deliveries = list(
                sorted(
                    filter(
                        lambda delivery: delivery.pickup_point_id == pp.id
                        and delivery.delivered_flg
                        and not delivery.pp_to_user_batched_flg,
                        deliveries,
                    ),
                    key=lambda delivery: delivery.delivery_plan_dttm,
                ),
            )

            if pickup_point_deliveries:
                self.metrics['utilized_pickup_points_cnt'] += 1

        self.metrics['total_sh'] = (
            sum([(c.time_travelled) * COURIER_HALT_COEF for c in couriers])
            / 3600
        )
        self.metrics['total_sh_bicycle'] = (
            sum(
                [
                    (c.time_travelled) * COURIER_HALT_COEF
                    for c in couriers
                    if c.type == 'bicycle'
                ],
            )
            / 3600
        )
        self.metrics['total_sh_vehicle'] = (
            sum(
                [
                    (c.time_travelled) * COURIER_HALT_COEF
                    for c in couriers
                    if c.type == 'vehicle'
                ],
            )
            / 3600
        )

        self.metrics['total_sh_cost'] = (
            self.metrics['total_sh_vehicle'] * COURIER_SH_COST['vehicle']
            + self.metrics['total_sh_bicycle'] * COURIER_SH_COST['bicycle']
        )
        self.metrics['total_sh_cost_bicycle'] = (
            self.metrics['total_sh_bicycle'] * COURIER_SH_COST['bicycle']
        )
        self.metrics['total_sh_cost_vehicle'] = (
            self.metrics['total_sh_vehicle'] * COURIER_SH_COST['vehicle']
        )

        self.metrics['delivery_cost_sh_based'] = (
            self.metrics['total_sh_cost']
            / self.metrics['success_deliveries_cnt']
        )
        self.metrics['delivery_cost_sh_based_bicycle'] = self.metrics[
            'total_sh_cost_bicycle'
        ] / (self.metrics['success_deliveries_bicycle_cnt'] + 1e-6)
        self.metrics['delivery_cost_sh_based_vehicle'] = self.metrics[
            'total_sh_cost_vehicle'
        ] / (self.metrics['success_deliveries_vehicle_cnt'] + 1e-6)

        self.metrics['avg_work_time'] = self.pp_work_time
        self.metrics['success_deliveries_per_point'] = (
            self.metrics['success_deliveries_cnt']
            / self.metrics['utilized_pickup_points_cnt']
        )
        self.metrics['success_trips_per_point'] = (
            self.metrics['success_trips_cnt']
            / self.metrics['utilized_pickup_points_cnt']
        )
        self.metrics['avg_trips_per_hour'] = (
            self.metrics['success_trips_per_point']
            / self.metrics['avg_work_time']
        )
        return self.metrics
