import os.path

import urllib3
import json
import click
import logging
import yt.wrapper as yt

from gptransfer_client import GPTransferClient, GPTransferRequestTimeout, GPTransferException


logging.basicConfig(format='%(asctime)s %(levelname)s %(name)s %(message)s', level=logging.INFO)
logger = logging.getLogger(__name__)

urllib3.disable_warnings(urllib3.exceptions.InsecureRequestWarning)

@click.command()
@click.argument('gp_path', type=str)
@click.argument('yt_path', type=str)
@click.argument('yt_schema', type=str)
@click.argument('action_if_exists', type=str)
def main(gp_path, yt_path, yt_schema, action_if_exists):
    if action_if_exists not in ['drop', 'append']:
        raise ValueError(action_if_exists)

    yt_schema = json.loads(yt_schema)

    if yt.exists(yt_path) and action_if_exists == 'drop':
        logger.info(f'Dropping table {yt_path}')
        yt.remove(yt_path)

    if not yt.exists(yt_path):
        logger.info(f'Creating table {yt_path} with schema {yt_schema}')
        yt.create('table', path=yt_path, attributes={"schema": yt_schema})


    columns = [column['name'] for column in yt_schema]
    logger.info(f'Columns for replication: {columns}')

    if not columns:
        raise RuntimeError('Resulting table will have no columns, aborting')

    logger.info('Start replication')

    with open(os.path.expanduser('~/mylib_config.json')) as f:
        mylib_config = json.load(f)

    gptransfer_token = mylib_config['gptransfer_token']
    gp_token = mylib_config['gp_token']

    gptransfer_client = GPTransferClient(
        token=gptransfer_token,
        gp_user=os.environ['USER'],
        gp_password=gp_token,
        host='https://dwh-gptransfer-api.taxi.yandex.net',
        verify_https=False,
    )

    try:
        process_uuid = gptransfer_client.gp_to_yt(
            yt_table_path=yt_path,
            gp_table_name=gp_path,
            column_list=columns,
            connect_timeout=30,
            read_timeout=15,
        )
    except GPTransferRequestTimeout as exc:
        raise exc
    except GPTransferException as exc:
        if 'UUID: ' in str(exc):
            process_uuid = str(exc).split('UUID: ')[1]
            gptransfer_client.wait_till_finish(process_uuid, connect_timeout=30, read_timeout=15)

    gptransfer_client.wait_till_finish(process_uuid)

    status = gptransfer_client.check_status(process_uuid)
    logger.info(f'Finish: {status}')

    if status.error:
        raise RuntimeError(status)

    logger.info('Done')


if __name__ == '__main__':
    main()
