begin;
drop table if exists snb_delivery.international_dash;
create table snb_delivery.international_dash as
with
currency_rates as (
    SELECT date,
           target_cur as currency,
           rate
    FROM dds.dim_currency_rate
    where date >= '2021-06-01'
      and source_cur = 'RUB'
      and target_cur in ('USD', 'RUB')
),

fact_rub as (
    select
        'fact'::text as source,
        'day'::text as scale,
        date::date as date,
        country,
        same_or_next_day,
        c2c_smb_b2b_flg as business,
        case when tariff = 'Delivery+Courier' then 'Delivery' else tariff end as tariff,
        metric,
        sum(val) as value
    from (
        select
            unnest(array [deliveries, orders, trips, gmv, commissions, subsidies, decoupling, net_inflow]) as val,
            unnest(array ['deliveries', 'orders', 'trips', 'gmv', 'commissions', 'subsidies', 'decoupling', 'net_inflow']) as metric,
            t.*
        from snb_delivery.dash_bizdev_business_overview t
    ) as raw_fact
    group by date, tariff, same_or_next_day, metric, country, c2c_smb_b2b_flg
),

base_budget_and_ambition as (
    select
        'budget'::varchar as source,
        b.*
    from
        snb_delivery.plans_bud_202207 b
    union all
    select
        'ambition'::varchar as source,
        a.*
    from
        snb_delivery.plans_amb_202207 a
),
budget_and_amb as (
-- budget_and_amb_rub
    select
      source,
     'RUB'::text as currency,
     'day'::text as scale,
     date,
     country,
     same_or_next_day,
     tariff,
     metric,
     business,
     sum(case
          when metric in ('deliveries', 'orders', 'trips') then greatest(0,value) -- crutch for negative values
          else value
         end) as value
    from base_budget_and_ambition
    group by source,
             date,
             country,
             same_or_next_day,
             tariff,
             metric,
             business
union
-- budget_and_amb_usd
    select
        source,
        'USD'::text as currency,
        'day'::text as scale,
        date,
        country,
        same_or_next_day,
        tariff,
        metric,
        business,
        sum(case
            when metric in ('deliveries', 'orders', 'trips') then greatest(0,value) -- crutch for negative values
            else
                case
                  when date >= '2022-07-01' then value/75.0
                  else value*rate
                end
        end)
        as value
    from base_budget_and_ambition
      left join
          (select * from currency_rates where currency = 'USD') cr
            using(date)
    group by
        source,
        date,
        country,
        same_or_next_day,
        tariff,
        metric,
        business
)
,fact_rub_usd as (
    select
        source,
        currency,
        scale,
        date,
        country,
        same_or_next_day,
        tariff,
        metric,
        business,
        case when metric in ('deliveries', 'orders', 'trips') then value else value * rate end as value
    from
        fact_rub
    join
        currency_rates
    using(date)
),

day as (
    select * from budget_and_amb
    union
    select * from fact_rub_usd
),

week as (
   select
       source,
       currency,
       'week'::text as scale,
       date_trunc('week', date)::date as date,
       country,
       same_or_next_day,
       tariff,
       metric,
       business,
       sum(value) as value
   from day
   group by
       source,
       currency,
       date_trunc('week', date),
       country,
       same_or_next_day,
       tariff,
       metric,
       business
),
month as (
   select
       source,
       currency,
       'month'::text as scale,
       date_trunc('month', date)::date as date,
       country,
       same_or_next_day,
       tariff,
       metric,
       business,
       sum(value) as value
   from day
   group by
       source,
       currency,
       date_trunc('month', date),
       country,
       same_or_next_day,
       tariff,
       metric,
       business
),
all_data as (
    select * from day
    union
    select * from week
    union
    select * from month
),
gmv as (
    select * from all_data where metric = 'gmv'
),
deliveries as (
    select * from all_data where metric = 'deliveries'
),
trips as (
    select * from all_data where metric = 'trips'
),
orders as (
    select * from all_data where metric = 'orders'
),
pre_final as (
    select
        all_data.*,
        gmv.value as gmv,
        deliveries.value as deliveries,
        orders.value as orders,
        trips.value as trips
    from all_data
    left join gmv
    using(source, currency, scale, date, country, same_or_next_day, tariff, business)
    left join deliveries
    using(source, currency, scale, date, country, same_or_next_day, tariff, business)
    left join orders
    using(source, currency, scale, date, country, same_or_next_day, tariff, business)
    left join trips
    using(source, currency, scale, date, country, same_or_next_day, tariff, business)

    where
        country not in ('Великобритания', 'Russia', 'Россия','США','Австралия','Эстония','Латвия','Литва')
),

with_prev_values as (
    select
        pre_final.*
--         ,lag(date) over w as prev_date
        ,case when scale = 'day'::text then lag(date,7) over w else lag(date) over w end as prev_date
--         ,lag(value) over w as prev_value
         ,case when scale = 'day'::text then null else lag(value) over w end as prev_value
--          ,lag(deliveries) over w as prev_deliveries
         ,case when scale = 'day'::text then lag(deliveries,7) over w else lag(deliveries) over w end as prev_deliveries
--          ,lag(gmv) over w as prev_gmv
         ,case when scale = 'day'::text then lag(gmv,7) over w else lag(gmv) over w end as prev_gmv
--          ,lag(orders) over w as prev_orders
         ,case when scale = 'day'::text then lag(orders,7) over w else lag(orders) over w end as prev_orders

        ,False
            or lag(date) over w is null
            or (scale = 'day'::text and lag(date,7) over w = (date - interval '1 week')::date)
            or (scale = 'week'::text and lag(date) over w = (date - interval '1 week')::date)
            or (scale = 'month'::text and lag(date) over w = (date - interval '1 month')::date)
        as _correct_prev_date
    from pre_final
    window w as (partition by source, currency, scale, country, same_or_next_day, tariff, metric, business order by date)
),

with_correct_prev_values as (
    select
        source, currency, scale, country, same_or_next_day, tariff, metric, business, date,
        value, gmv, orders, deliveries, trips,

        case when _correct_prev_date then prev_date end as prev_date,
        case when _correct_prev_date then prev_gmv end as prev_gmv,
        case when _correct_prev_date then prev_deliveries end as prev_deliveries,
        case when _correct_prev_date then prev_orders end as prev_orders,
        case when _correct_prev_date then prev_value end as prev_value
    from
        with_prev_values
),


deliveries_daily as (
    select
        country,
        date,
        extract(isodow from date) as dow,
        sum(value) as deliveries
    from
        with_correct_prev_values
    where True
        and source = 'fact'
        and currency = 'USD'
        and scale = 'day'
        and metric = 'deliveries'
        and date between date_trunc('week', now()) - interval '4 week' and date_trunc('week', now()) - interval '1 week 1 day'
    group by
        country, date
),
deliveries_dow as (
    select
        country,
        dow,
        sum(deliveries) as deliveries
    from deliveries_daily
    group by
        country,
        dow
),
seasonality_by_dow as (
    select
        country,
        dow,
        deliveries / nullif(sum(deliveries) over (partition by country),0) as dow_seasonality
    from
        deliveries_dow
    order by country, dow
),
running_seasonality as (
    select
        country,
        dow,
        sum(dow_seasonality) over (partition by country order by dow) as running_seasonality_sum
    from
        seasonality_by_dow
),



max_available_dow as (
 select
      source, currency, scale, country, same_or_next_day, tariff, business,
      date_trunc('week', date)::date as week_start,
      extract(isodow from max(date)) as max_dow
 from
      with_correct_prev_values
 where True
      and scale = 'day' and source = 'fact'
      and date_trunc('week', date)::date = date_trunc('week', now()::date)::date
 group by
      source, currency, scale, country, same_or_next_day, tariff, business,
      date_trunc('week', date)
),

with_max_available_dow as (
    select
        dash.*,
        mad.max_dow
    from with_correct_prev_values dash
    left join
        max_available_dow mad
    on True
        and dash.scale = 'week'
        and dash.date = mad.week_start
        and dash.source = mad.source
        and dash.currency = mad.currency
        and dash.country = mad.country
        and dash.same_or_next_day = mad.same_or_next_day
        and dash.tariff = mad.tariff
        and dash.business = mad.business
),
with_running_seasonality_sum as (
    select
        with_max_available_dow.*,
        coalesce(running_seasonality.running_seasonality_sum, max_dow / 7.0) as rss -- если новая страна и для неё ещё нет сезонности, фолбечимся на равномерную внутринедельную сезонность
    from with_max_available_dow
    left join running_seasonality
    on with_max_available_dow.max_dow = running_seasonality.dow and with_max_available_dow.country = running_seasonality.country
),
absolute_values as (
    select
        case when rss is not null then 'run rate' else source end as source,
        currency, scale, country,
        case
          when country not in ('Молдавия', 'Белоруссия', 'Беларусь', 'Казахстан',
                               'Узбекистан', 'Азербайджан', 'Армения', 'Грузия',
                               'Киргизия', 'Молдова','Финляндия')
          then 'New Markets'
          else 'Established Markets'
        end as market_type,
        same_or_next_day, tariff,
        metric,
        business,
        date,
        value as _source_value,
        case when rss is not null then value / nullif(rss,0) else value end as value,
        case when rss is not null then gmv / nullif(rss,0) else gmv end as gmv,
        case when rss is not null then orders / nullif(rss,0) else orders end as orders,
        case when rss is not null then trips / nullif(rss,0) else trips end as trips,
        case when rss is not null then deliveries / nullif(rss,0) else deliveries end as deliveries,
        prev_date,
        prev_gmv,
        prev_deliveries,
        prev_orders,
        prev_value
    from
        with_running_seasonality_sum
)
,relative_dates as (
select
    country
    ,case
        when country in ('Кот-д’Ивуар','Кот-д''Ивуар') then to_date('07/03/2022','dd/mm/yyyy')
        else min(date)::date
    end as min_country_date
from absolute_values v
where 0<1
  and source = 'fact'
  and metric = 'deliveries'
  and scale = 'day'
  and value > 20
group by
  country
)
select
    market_type
    ,d.country_name_en as country
    ,date, metric, value
    ,source, currency, scale
    ,same_or_next_day, tariff, business
    ,gmv, orders, deliveries, trips
    ,prev_date, prev_gmv, prev_deliveries, prev_orders, prev_value
    ,case scale
      when 'day' then date::date-r.min_country_date::date
      when 'week' then ceil(date::date-date_trunc('week',r.min_country_date::date)::date)/7
      when 'month' then (date_part('year',date)-date_part('year',r.min_country_date))*12
                        +(date_part('month',date)-date_part('month',r.min_country_date))
    end as relative_date
from absolute_values v
  left join snb_delivery.countries_dict d
    on lower(d.country_name_ru) = lower(v.country)
  left join relative_dates r
    on lower(r.country) = lower(v.country)
where not (scale = 'day' and source != 'fact');
commit;
