import os
import requests
import json
import numpy as np
from datetime import datetime, timedelta
from requests.adapters import HTTPAdapter
from urllib3.util.retry import Retry

import yt.wrapper as yt
import ticket_parser2_py3.api.v1.exceptions as tp2_exceptions
import ticket_parser2_py3.api.v1 as tp2

class TvmRequestError(Exception):
    """Error during performing request to tvm."""


class TVMClient:
    def __init__(self, tvm_secret: str, src_id: int, dsts: dict):
        self._tvm_client = tp2.TvmClient(
            tp2.TvmApiClientSettings(
                self_client_id=src_id,
                enable_service_ticket_checking=True,
                enable_user_ticket_checking=tp2.BlackboxEnv.Test,
                self_secret=tvm_secret,
                dsts=dsts,
            ),
        )

    def get_ticket(self, dst_service_name):
        try:
            return self._tvm_client.get_service_ticket_for(dst_service_name)
        except tp2_exceptions.TvmException as ex:
            raise TvmRequestError(
                f'cannot perform request to {dst_service_name}: {ex}',
            )
            
def create_session(dst_tvm_id):
    session = requests.Session()
    tvm = TVMClient(
        tvm_secret=os.environ.get('TVM_SECRET'),
        src_id=2027260,
        dsts={'logplatform': dst_tvm_id},
    )
    tvm_ticket = tvm.get_ticket(
        dst_service_name='logplatform',
    )
    session.headers = {'X-Ya-Service-Ticket': tvm_ticket}
    retry = Retry(
        total=5,
        backoff_factor=1,
        status_forcelist=(500, 502, 503, 504),
        allowed_methods=('POST', 'GET'),
    )
    session.mount('https://', HTTPAdapter(max_retries=retry))
    return session

def get_date_range(day, min_shift, max_shift):
    date_range = []
    for i in np.arange(min_shift, max_shift + 1):
        date_range.append((day + timedelta(days=int(i))))#.strftime('%Y-%m-%d'))
    return date_range
    
def save_results(stations_data):
    YT_TABLE_PATH = '//home/taxi-delivery/analytics/production/k2/lavka_capacity/k2_and_t2_lavkas'
    SCHEMA = [
        {"name" : "day_minus_4_and_less", "type" : "int32", "strict": "false"}, 
        {"name" : "day_minus_3", "type" : "int32", "strict": "false"}, 
        {"name" : "day_minus_2", "type" : "int32", "strict": "false"},
        {"name" : "day_minus_1", "type" : "int32", "strict": "false"},
        {"name" : "day_0", "type" : "int32", "strict": "false"},
        {"name" : "day_plus_1", "type" : "int32", "strict": "false"},
        {"name" : "day_plus_2", "type" : "int32", "strict": "false"},
        {"name" : "day_plus_3", "type" : "int32", "strict": "false"},
        {"name" : "day_plus_4", "type" : "int32", "strict": "false"},
        {"name" : "day_plus_5_and_more", "type" : "int32", "strict": "false"},
        {"name" : "capacity_day_0", "type" : "double", "strict": "false"},
        {"name" : "capacity_day_1", "type" : "double", "strict": "false"},
        {"name" : "capacity_day_2", "type" : "double", "strict": "false"},
        {"name" : "capacity_day_3", "type" : "double", "strict": "false"},
        {"name" : "capacity_day_4", "type" : "double", "strict": "false"},
        {"name" : "station_capacity", "type" : "int32", "strict": "false"},
        {"name" : "station_operator_id", "type" : "string", "strict": "false"},
        {"name" : "station_delivery_service_id", "type" : "string", "strict": "false"},
        {"name" : "station_id", "type" : "string", "strict": "false"},
        {"name" : "stations_locality", "type" : "string", "strict": "false"},
        {"name" : "station_full_name", "type" : "string", "strict": "false"},
        {"name" : "station_lat", "type" : "double", "strict": "false"},
        {"name" : "station_lon", "type" : "double", "strict": "false"},
        {"name" : "source", "type" : "string", "strict": "false"},
    ]
    yt.config["proxy"]["url"] = "hahn.yt.yandex.net"
    yt.create("table", YT_TABLE_PATH, attributes={"schema": SCHEMA}, force=True)
    yt.write_table(
        YT_TABLE_PATH,
        stations_data,
    )
    
def analyze_stations(stations, k2_flg = True):
    MIN_DELIVERY_DATE_SHIFT = 1
    MAX_DELIVERY_DATE_SHIFT = 4
    ORDER_STAY_DAYS = 3

    if k2_flg:
        KFREMOVE = 0.7
    else:
        KFREMOVE = 0.9

    FORBIDEN_DSID = ['93730']

    now = datetime.now()

    today = now.strftime('%Y-%m-%d')

    days_of_interest = get_date_range(now, 
                                      -ORDER_STAY_DAYS, 
                                      MAX_DELIVERY_DATE_SHIFT)
    days_of_interest_formatted = []
    for i in days_of_interest:
        days_of_interest_formatted.append(i.strftime('%Y-%m-%d'))

    days_names = [
        'day_minus_3'
        , 'day_minus_2'
        , 'day_minus_1'
        , 'day_0'
        , 'day_plus_1'
        , 'day_plus_2'
        , 'day_plus_3'
        , 'day_plus_4'
    ]

    days_mapping = dict(zip(days_of_interest_formatted, days_names))


    capacity_names = [
        'capacity_day_0'
        , 'capacity_day_0'
        , 'capacity_day_0'
        , 'capacity_day_0'
        , 'capacity_day_1'
        , 'capacity_day_2'
        , 'capacity_day_3'
        , 'capacity_day_4'
    ]

    capacity_mapping = dict(zip(days_of_interest_formatted, capacity_names))
    
    stations_data = []

    for i, station in enumerate(stations):
        station_operator_id = station['operator_id']
        station_delivery_service_id = station['delivery_service_id']

        station_is_deprecated = station['deprecated']
        station_is_enabled = station['enabled']
        station_is_enabled_in_platform = station['enabled_in_platform']
        station_enabled_robust = (station['enabled'] \
                                  and not station['deprecated'] \
                                  and station['enabled_in_platform'])

        station_isValidAsLavka = (station_operator_id == 'lavka')
        station_isValidAsPVZ = (station_delivery_service_id not in FORBIDEN_DSID)
        check_station = (not station_isValidAsLavka) and (not station_isValidAsPVZ)

        station_integrations = station['integrations']

        if (station_enabled_robust \
            and station_isValidAsLavka):# \
            #and len(station_integrations) == 0):

            station_capacity = station['capacity']
            station_id = station['station_id']

            stations_locality = station['location_details']['locality']
            station_full_name = station['station_full_name']

            station_lat = station['location_ll']['lat']
            station_lon = station['location_ll']['lon']

            try:
                station_requests_count_by_day = station['requests_count_by_day']

                days_data = {}
                capacity_data = {}

                loaded_capacity = 0.0

                for day in station_requests_count_by_day.keys():
                    if day < today and day not in days_of_interest_formatted:
                        if 'day_minus_4_and_less' not in days_data:
                            days_data['day_minus_4_and_less'] = station_requests_count_by_day[day]
                        else:
                            days_data['day_minus_4_and_less'] += station_requests_count_by_day[day]
                            
                        loaded_capacity += station_requests_count_by_day[day]
                    elif day in days_of_interest_formatted:
                        if days_mapping[day] not in days_data:
                            days_data[days_mapping[day]] = station_requests_count_by_day[day]
                        else:
                            days_data[days_mapping[day]] += station_requests_count_by_day[day]

                        if day <= today:
                            loaded_capacity += station_requests_count_by_day[day]
                        elif day > today:
                            loaded_capacity = loaded_capacity * (1.0 - KFREMOVE) + station_requests_count_by_day[day]

                        capacity_data[capacity_mapping[day]] = loaded_capacity

                    elif day > max(days_of_interest_formatted):
                        if 'day_plus_5_and_more' not in days_data:
                            days_data['day_plus_5_and_more'] = station_requests_count_by_day[day]
                        else:
                            days_data['day_plus_5_and_more'] += station_requests_count_by_day[day]

                for item in days_mapping.keys():
                    if days_mapping[item] not in days_data:
                        days_data[days_mapping[item]] = 0

                for item in capacity_mapping.keys():
                    if capacity_mapping[item] not in capacity_data:
                        capacity_data[capacity_mapping[item]] = 0.0

                station_data = {**days_data, **capacity_data}
                station_data['station_capacity'] = station['capacity']
                station_data['station_operator_id'] = station_operator_id
                station_data['station_delivery_service_id'] = station_delivery_service_id
                station_data['station_id'] = station_id
                station_data['stations_locality'] = stations_locality
                station_data['station_full_name'] = station_full_name
                station_data['station_lat'] = station_lat
                station_data['station_lon'] = station_lon

                stations_data.append(station_data)
            except:
                pass
        
    return stations_data
    
def main():
    params = {
            'port': 80,
            'capacity': True,
            'limit': 10000,
            'dump': 'eventlog',
        }
    
    k2_session = create_session(2024161)
    print('K2 SESSION CREATED')
    k2_response = k2_session.get(
            'http://logistic-platform.taxi.yandex.net/api/admin/station/list',
            params=params,
        )
    k2_response.raise_for_status()
    k2_stations = k2_response.json()['stations']
    print('GOT K2 STATIONS')
    
    ts_session = create_session(2028266)
    print('TS SESSION CREATED')
    ts_response = ts_session.get(
            'http://logistic-platform-market.taxi.yandex.net/api/admin/station/list',
            params=params,
        )
    ts_response.raise_for_status()
    ts_stations = ts_response.json()['stations']
    print('GOT TS STATIONS')
    
    k2_lavkas = analyze_stations(k2_stations, k2_flg = True)
    ts_lavkas = analyze_stations(ts_stations, k2_flg = False)
    for station in k2_lavkas:
        station['source'] = 'logistic-platform'

    for station in ts_lavkas:
        station['source'] = 'logistic-platform-market'
    all_lavkas = k2_lavkas + ts_lavkas
    print('STATIONS ANALYZED')
    
    save_results(all_lavkas)
    print('RESULTS SAVED')
    
if __name__ == '__main__':
    main()
