import os
import requests
import json
import numpy as np
from datetime import datetime, timedelta
from requests.adapters import HTTPAdapter
from urllib3.util.retry import Retry

import yt.wrapper as yt
import ticket_parser2_py3.api.v1.exceptions as tp2_exceptions
import ticket_parser2_py3.api.v1 as tp2

class TvmRequestError(Exception):
    """Error during performing request to tvm."""


class TVMClient:
    def __init__(self, tvm_secret: str, src_id: int, dsts: dict):
        self._tvm_client = tp2.TvmClient(
            tp2.TvmApiClientSettings(
                self_client_id=src_id,
                enable_service_ticket_checking=True,
                enable_user_ticket_checking=tp2.BlackboxEnv.Test,
                self_secret=tvm_secret,
                dsts=dsts,
            ),
        )

    def get_ticket(self, dst_service_name):
        try:
            return self._tvm_client.get_service_ticket_for(dst_service_name)
        except tp2_exceptions.TvmException as ex:
            raise TvmRequestError(
                f'cannot perform request to {dst_service_name}: {ex}',
            )
            
def create_session():
    session = requests.Session()
    tvm = TVMClient(
        tvm_secret=os.environ.get('TVM_SECRET'),
        src_id=2027260,
        dsts={'logplatform': 2024161},
    )
    tvm_ticket = tvm.get_ticket(
        dst_service_name='logplatform',
    )
    session.headers = {'X-Ya-Service-Ticket': tvm_ticket}
    retry = Retry(
        total=5,
        backoff_factor=1,
        status_forcelist=(500, 502, 503, 504),
        allowed_methods=('POST', 'GET'),
    )
    session.mount('https://', HTTPAdapter(max_retries=retry))
    return session
    
def get_date_range(day, min_shift, max_shift):
    date_range = []
    for i in np.arange(min_shift, max_shift + 1):
        date_range.append((day + timedelta(days=int(i))))#.strftime('%Y-%m-%d'))
    return date_range
    
def save_results(stations_data):
    YT_TABLE_PATH = '//home/taxi-delivery/analytics/production/k2/lavka_capacity/latest'
    SCHEMA = [
        {"name" : "station_old_requests_cnt", "type" : "int64"}, 
        {"name" : "station_operator_id", "type" : "string"},
        {"name" : "station_delivery_service_id", "type" : "string"},
        {"name" : "station_capacity", "type" : "int64"},
        {"name" : "station_id", "type" : "string"},
        {"name" : "stations_locality", "type" : "string"},
        {"name" : "station_full_name", "type" : "string"},
        {"name" : "station_lat", "type" : "float"},
        {"name" : "station_lon", "type" : "float"},
        {"name" : "station_current_orders_cnt", "type" : "int64"},
        {"name" : "station_oldest_request_date", "type" : "string"},
        {"name" : "station_possible_to_place_order_1", "type" : "boolean"},
        {"name" : "station_possible_to_place_order_2", "type" : "boolean"},
        {"name" : "station_possible_to_place_order_3", "type" : "boolean"},
        {"name" : "station_possible_to_place_order_4", "type" : "boolean"},
    ]
    yt.config["proxy"]["url"] = "hahn.yt.yandex.net"
    yt.create("table", YT_TABLE_PATH, attributes={"schema": SCHEMA}, force=True)
    yt.write_table(
        YT_TABLE_PATH,
        stations_data,
    )
    
def main():
    session = create_session()
    
    print('SESSION CREATED')

    params = {
            'port': 80,
            'capacity': True,
            'limit': 10000,
            'dump': 'eventlog',
        }

    response = session.get(
            'http://logistic-platform.taxi.yandex.net/api/admin/station/list',
            params=params,
        )

    response.raise_for_status()
    
    print('GOT RESPONSE')
    
    stations = response.json()['stations']
    
    print('GOT STATIONS')
    
    now = datetime.now()

    MIN_DELIVERY_DATE_SHIFT = 1
    MAX_DELIVERY_DATE_SHIFT = 4
    ORDER_STAY_DAYS = 3
    KFREMOVE = 0.7

    FORBIDEN_DSID = ['93730']

    stations_data = []
    for i, station in enumerate(stations):
    #     print(i, station['station_id'])
        station_operator_id = station['operator_id']
        station_delivery_service_id = station['delivery_service_id']

        station_is_deprecated = station['deprecated']
        station_is_enabled = station['enabled']
        station_is_enabled_in_platform = station['enabled_in_platform']
        station_enabled_robust = (station['enabled'] \
                                  and not station['deprecated'] \
                                  and station['enabled_in_platform'])

        station_isValidAsLavka = (station_operator_id == 'lavka')
        station_isValidAsPVZ = (station_delivery_service_id not in FORBIDEN_DSID)
        check_station = (not station_isValidAsLavka) and (not station_isValidAsPVZ)

        station_integrations = station['integrations']

        if (station_enabled_robust \
            and station_isValidAsLavka \
            and len(station_integrations) == 0):

            station_capacity = station['capacity']
            station_id = station['station_id']

            stations_locality = station['location_details']['locality']
            station_full_name = station['station_full_name']

            station_lat = station['location_ll']['lat']
            station_lon = station['location_ll']['lon']

            station_possible_to_place_order = {
                1: False,
                2: False,
                3: False,
                4: False
            }

            try:
                station_current_orders_cnt = station['orders_count']

                station_requests_count_by_day = station['requests_count_by_day']
                station_oldest_request_date = list(station_requests_count_by_day.keys())[0]

                possible_days_of_delivery = get_date_range(datetime.now(), 
                                                           MIN_DELIVERY_DATE_SHIFT, 
                                                           MAX_DELIVERY_DATE_SHIFT)
                
                oldest_possible_day = (now - timedelta(days=3)).strftime('%Y-%m-%d')
                station_old_requests_cnt = 0
                for day in station_requests_count_by_day.keys():
                    requests_cnt = station_requests_count_by_day[day]
                    if day < oldest_possible_day:
                        station_old_requests_cnt += requests_cnt
                    
                for j in range(len(possible_days_of_delivery)):
                    day_of_delivery = possible_days_of_delivery[j].strftime('%Y-%m-%d')
                    permanent_count = 0
                    today = now.strftime('%Y-%m-%d')

                    station_count_by_days = {}

                    for day in station_requests_count_by_day.keys():
                        requests_cnt = station_requests_count_by_day[day]
                        max_day = max(day, today)
                        if max_day not in station_count_by_days:
                            station_count_by_days[max_day] = requests_cnt
                        else:
                            station_count_by_days[max_day] += requests_cnt

                    if day_of_delivery not in station_count_by_days:
                        station_count_by_days[day_of_delivery] = 1
                    else:
                        station_count_by_days[day_of_delivery] += 1

                    current_sum = 0

                    days_to_check = get_date_range(possible_days_of_delivery[j], -ORDER_STAY_DAYS, ORDER_STAY_DAYS)

                    for n_day in days_to_check:
                        day = n_day.strftime('%Y-%m-%d')
                        if day in station_count_by_days:
                            current_orders_cnt = station_count_by_days[day]
                            current_sum += current_orders_cnt
                        if (day >= day_of_delivery and current_sum > station_capacity):
#                             print(f'{day} - capacity exhasted')
                            break

                        current_sum *= (1 - KFREMOVE)
                        if day == day_of_delivery:
                            station_possible_to_place_order[j + 1] = True

                station_data = {}
                station_data['station_old_requests_cnt'] = station_old_requests_cnt
                station_data['station_operator_id'] = station_operator_id
                station_data['station_delivery_service_id'] = station_delivery_service_id
                station_data['station_capacity'] = station_capacity
                station_data['station_id'] = station_id
                station_data['stations_locality'] = stations_locality
                station_data['station_full_name'] = station_full_name
                station_data['station_lat'] = station_lat
                station_data['station_lon'] = station_lon
                station_data['station_possible_to_place_order'] = station_possible_to_place_order
                station_data['station_current_orders_cnt'] = station_current_orders_cnt
                station_data['station_oldest_request_date'] = station_oldest_request_date

                stations_data.append(station_data)
            except:
                pass
            
    for station in stations_data:
        station['station_possible_to_place_order_1'] = station['station_possible_to_place_order'][1]
        station['station_possible_to_place_order_2'] = station['station_possible_to_place_order'][2]
        station['station_possible_to_place_order_3'] = station['station_possible_to_place_order'][3]
        station['station_possible_to_place_order_4'] = station['station_possible_to_place_order'][4]
        station.pop('station_possible_to_place_order', None)
        
        
    print('PARSED STATIONS')
    
    save_results(stations_data)
    
    print('RESULTS SAVED')
    
if __name__ == '__main__':
    main()
