import os

import json

import requests
from requests.adapters import HTTPAdapter
from urllib3.util.retry import Retry

import yt.wrapper as yt

from business_models import hahn

TRACKER_URL = 'https://st-api.yandex-team.ru'
OAUTH_TOKEN = OAUTH_TOKEN = os.environ.get('ST_TOKEN')

def make_http_session():
    session = requests.Session()

    session.headers = {
        'Authorization' : f'OAuth {OAUTH_TOKEN}'
    }
    
    retry = Retry(
        total=10,
        backoff_factor=0.1,
        status_forcelist=(500, 502, 503, 504),
        allowed_methods=('POST',),
    )
    session.mount('https://', HTTPAdapter(max_retries=retry))
    return session

def get_ticket(session, ticket_key):
    ticket = session.get(url=TRACKER_URL + f'/v2/issues/{ticket_key}')
    
    return ticket.json()

def create_comment(session, ticket_id, text):
    data = {
        'text': text
    }

    resp = session.post(url=TRACKER_URL + f'/v2/issues/{ticket_id}/comments', data=json.dumps(data))
    
    return resp.json()

def find_robot_log_analytics_comments(session, ticket_id):
    resp = session.get(url=TRACKER_URL + f'/v2/issues/{ticket_id}/comments')
    
    comments_data = resp.json()
    
    has_robot_comment = False

    for comment in comments_data:
        if comment['createdBy']['id'] == 'robot-log-analytics':
            has_robot_comment = True
            break
            
    return has_robot_comment

def can_be_closed(session, ticket_key):
    """
    Возвращает True и статус закрытия, если тикет можно закрыть
    Иначе возвращает False и статус перевода в работу
    """
    ticket_transitions = session.get(url=TRACKER_URL + f'/v2/issues/{ticket_key}/transitions', verify=False).json()
    intermediate_step = None
    for status in ticket_transitions:
        if 'Закрыть' == status['display']:
            return status, True
        elif 'В работу' == status['display']:
            intermediate_step = status
    return intermediate_step, False

def modify_ticket(session, ticket_key):
    """
    Закрывает тикет если его можно закрыть.
    Если нельзя, то переводит в работу, а потом закрывает
    """
    status, is_final = can_be_closed(session, ticket_key)
    transition = status['id']
    resp = session.post(url=TRACKER_URL + f'/v2/issues/{ticket_key}/transitions/{transition}/_execute').json()
    if is_final:
        return resp
    else:
        status, is_final = can_be_closed(session, ticket_key)
        transition = status['id']
        resp = session.post(url=TRACKER_URL + f'/v2/issues/{ticket_key}/transitions/{transition}/_execute').json()
        return resp

    

def main():
    print('START')
    
    # Get all processed requests
    yt.config['proxy']['url'] = 'hahn.yt.yandex.net'
    PROCESSED_REQUESTS_PATH = '//home/taxi-delivery/analytics/production/ndd/alerts/no_yd_id_processed'
    processed_requests = list(yt.read_table(PROCESSED_REQUESTS_PATH))
    print('Got all request')
    
    # Create session
    session = make_http_session()
    print('Session created')
    
    # Find opened tickets
    requests_to_check = []
    for request in processed_requests:
        ticket_key = request['ticket_key']

        ticket = get_ticket(session, ticket_key)
        
        if ticket['status']['display'] != 'Закрыт':
            requests_to_check.append({
                'request_code': request['request_code'],
                'ticket_key': ticket_key,
                'ticket_id': ticket['id']
            })
    print('Filtered opened tickets')
            
    # Extract request_code's
    request_codes_to_check = []
    for request in requests_to_check:
        request_codes_to_check.append(request['request_code'])

    # Get external_order_id's for request_code's
    request_codes = str(request_codes_to_check)

    with open('check_requests.txt', 'r') as f:
        script = f.read()

    full_script = script.format(request_codes = request_codes)
    
    df = hahn(full_script)
    print('Got external orders')
    
    # Filter requests with external orders
    tickets_to_close = df[df['external_order_id_null'] == 0].reset_index(drop=True)
    
    # Filter tickets to comment
    requests_to_close = []
    for i in range(len(tickets_to_close)):
        row = tickets_to_close.iloc[i]
        request_code = row['request_code']
        external_code_timestamp = row['min_history_timestamp']
        external_order_id = row['external_order_id']

        for request in requests_to_check:
            if request['request_code'] == request_code:
                requests_to_close.append({
                    'request_code': request_code,
                    'external_code_timestamp': external_code_timestamp,
                    'external_order_id': external_order_id,
                    'ticket_key': request['ticket_key'],
                    'ticket_id': request['ticket_id']
                })
                
    # Comment tickets
    for request in requests_to_close:
        timestamp = request['external_code_timestamp']
        external_order_id = request['external_order_id']
        text = f'Заказ (external_order_id = {external_order_id}) после ретрая прогрузился {timestamp}'
        if not find_robot_log_analytics_comments(session, request['ticket_id']):
            create_comment(session, request['ticket_id'], text)
            print('Added comment to ' + request['ticket_key'])
            modify_ticket(session, request['ticket_key'])
            print('Closed ticket ' + request['ticket_key'])
            
    print('END')
            
if __name__ == '__main__':
    main()
