#!/usr/bin/env python

import json
import time
import psycopg2
import psycopg2.extras

import nirvana.job_context as nv


def retry(n_tries=3, exception_cls=Exception, sleep=0):
    def inner(f):
        def wrapped(*args, **kwargs):
            n = 0
            while True:
                try:
                    return f(*args, **kwargs)
                except exception_cls:
                    if n < n_tries:
                        n += 1
                        time.sleep(sleep)
                        continue
                    else:
                        raiser
        return wrapped
    return inner


def ResultIter(cursor_buffered):
    for result in cursor_buffered:
        yield {k: v for k, v in result.items()}

class Greenplum(object):
    def __init__(self, server, port, database, user, token):
        connection_string = 'postgresql://{}:{}@{}:{}/{}'.format(user, token, server, port, database)
        self.connection = psycopg2.connect(dsn=connection_string)

    def run(self, query):
        cursor = self.connection.cursor(cursor_factory=psycopg2.extras.DictCursor)
        cursor.execute(query)
        return cursor

    def commit(self):
        self.connection.commit()

    def close(self):
        self.connection.close()


def main():
    job_context = nv.context()
    parameters = job_context.get_parameters()
    inputs = job_context.get_inputs()
    outputs = job_context.get_outputs()

    with open(inputs.get('query')) as f:
        query = f.read()

    @retry(n_tries=parameters['n-tries'], sleep=10, exception_cls=psycopg2.OperationalError)
    def run(query, parameters):
        cursor = None
        greenplum = None
        try:
            greenplum = Greenplum(
                server=parameters['server'],
                port=parameters['port'],
                database=parameters['database'],
                user=parameters['user'],
                token=parameters['token']
            )

            cursor = greenplum.run(query)
            if cursor.description is not None:
                result = list(ResultIter(cursor))
            else:
                result = None
            greenplum.commit()
            return result
        finally:
            if cursor:
                cursor.close()
            if greenplum:
                greenplum.close()

    with open(outputs.get('out_json'), 'w') as f:
        result = run(query, parameters)
        json.dump(result, f)

if __name__ == '__main__':
    main()
