/*
    Purpose of the script:
    Get MPH data for SDD taxi orders

    Author:
    @shamilalburin

    Ticket:
    LOGDATA-2084
*/

PRAGMA AnsiInForEmptyOrNullableItemsCollections;
PRAGMA OrderedColumns;
PRAGMA SimpleColumns;
PRAGMA yson.DisableStrict;
PRAGMA yt.Owners = 'shamilalburin';
PRAGMA yt.InferSchema = '1';

USE hahn; 

-- Working directory for this task

$WORKING_DIRECTORY = '//home/taxi-delivery/analytics/production/product/SDD';

-- List of tables

$SDD_DATA = $WORKING_DIRECTORY || '/data_for_SDD_dashboard';
$FCT_ORDER_METRICS = '//home/taxi-dwh/cdm/marketplace/fct_order_metric';
$FCT_SUPPLY_HIST = '//home/taxi-dwh/cdm/supply/fct_supply_state_hist';


-- Dates
$DT_START = '2022-03-25';
$DT_END =  CAST(CurrentUtcDate() AS String);
$MTH_START = SUBSTRING($DT_START, 0, 7) || '-01';
$MTH_END = SUBSTRING($DT_END, 0, 7) || '-01';

-- Result path

$RESULT_PATH = $WORKING_DIRECTORY || '/MPH_night_data';

-- Data collection

$sdd_waybills = (
    SELECT
        order_id,
        SOME(order_type) AS order_type
    FROM $SDD_DATA
    GROUP BY taxi_order_id AS order_id
);

$all_night_orders = (
    SELECT
        utc_order_created_dttm AS utc_created_dttm,
        T1.order_id AS order_id,
        CAST(delivery_point_cnt > 0 AS Int64) AS any_success_delivery,
        driver_net_income_incl_paid_cancel_amt * currency_rate AS driver_net_income,
        executor_profile_sk,
        tariff_class_code,
        tariff_geo_zone_code,
        T2.order_type AS order_type
    FROM RANGE($FCT_ORDER_METRICS, $MTH_START, $MTH_END) AS T1
        INNER JOIN $sdd_waybills AS T2 ON T1.order_id = T2.order_id
);

$time_diff = ($dttm1, $dttm2) -> {
    $tstmp1 = DateTime::MakeTimestamp(DateTime::ParseIso8601($dttm1));
    $tstmp2 = DateTime::MakeTimestamp(DateTime::ParseIso8601($dttm2));
    $sec1 = CAST(DateTime::ToSeconds($tstmp1) AS int64);
    $sec2 = CAST(DateTime::ToSeconds($tstmp2) AS int64);

    RETURN ($sec2 - $sec1) * 1.0 / 60
};

$supply_data = (
    SELECT
        executor_profile_sk,
        order_id,
        SUM($time_diff(utc_valid_from_dttm, utc_valid_to_dttm)) AS time_spent_mnt
    FROM RANGE($FCT_SUPPLY_HIST, $DT_START, $DT_END)
    WHERE TRUE
        AND executor_status_code IN ['driving', 'waiting', 'transporting']
        AND order_id IS NOT NULL
    GROUP BY 
        order_id,
        executor_profile_sk
);

INSERT INTO $RESULT_PATH WITH TRUNCATE
SELECT
    utc_created_dttm,
    T1.order_id AS order_id,
    order_type,
    T1.executor_profile_sk AS executor_profile_sk,
    driver_net_income,
    time_spent_mnt,
    driver_net_income * 60.0 / time_spent_mnt AS MPH,
    any_success_delivery,
    tariff_class_code,
    tariff_geo_zone_code
FROM $all_night_orders AS T1
    LEFT JOIN $supply_data AS T2 ON (
        T1.order_id = T2.order_id
        AND T1.executor_profile_sk = T2.executor_profile_sk
    )
