/*
    Purpose of the script:
    Collect all SDD data

    Authors:
    @shamilalburin

    Ticket:
    LOGDATA-1637
*/

PRAGMA AnsiInForEmptyOrNullableItemsCollections;
PRAGMA OrderedColumns;
PRAGMA SimpleColumns;
PRAGMA yson.DisableStrict;
PRAGMA yt.Pool = 'taxi-delivery';
PRAGMA yt.Owners = 'shamilalburin';
PRAGMA yt.InferSchema = '1';

USE hahn; 

-- Working directory for this task

$WORKING_DIRECTORY = '//home/taxi-delivery/analytics/production/product/SDD';

-- List of tables

$CLAIM_POINTS = '//home/delivery-dwh/ods/cargo_claims/claim_point';
$POINTS = '//home/delivery-dwh/ods/cargo_claims/point';
$CLAIMS = '//home/delivery-dwh/ods/cargo_claims/claim';
$CLAIM_STATUS_LOGS = '//home/delivery-dwh/ods/cargo_claims/claim_status_log';
$CLAIMS_ADD_INFO = '//home/taxi/production/replica/postgres/cargo_claims/raw/additional_info';
$TAXI_PERFORMERS_INFO ='//home/delivery-dwh/ods/cargo_claims/taxi_performer_info';
$CORP_CLIENT_NAMES = '//home/taxi/production/replica/mongo/struct/corp/corp_clients';
$SUPPORT_TICKETS = '//home/taxi-dwh/ods/chatterbox_support_taxi/ticket';
$CURRENCY_DATA = '//home/taxi-dwh/dds/dim_currency_rate';
$C2C_ORDERS = '//home/delivery-dwh/ods/cargo_c2c/order';

-- DateTime format

$dttm_format = DateTime::Format('%Y-%m-%d %H:%M:%S');

-- Dates
$DT_START = '2022-03-25';
$DT_END =  CAST(CurrentUtcDate() AS String);
$MTH_START = SUBSTRING($DT_START, 0, 7) || '-01';
$MTH_END = SUBSTRING($DT_END, 0, 7) || '-01';
$YEAR_START = SUBSTRING($DT_START, 0, 4) || '-01-01';
$YEAR_END = SUBSTRING($DT_END, 0, 4) || '-01-01';

-- Result path

$RESULT_PATH = $WORKING_DIRECTORY || '/data_for_SDD_dashboard';

-- Data collection

$sdd_claims = (
    SELECT
        Yson::ConvertToInt64(doc.claim_id) AS claim_id,
        DateTime::MakeTimestamp(DateTime::ParseIso8601(Yson::ConvertToString(
            Yson::ParseJson(Yson::ConvertToString(doc.same_day_data)).delivery_interval.from
        ))) AS delivery_period_start,
        DateTime::MakeTimestamp(DateTime::ParseIso8601(Yson::ConvertToString(
            Yson::ParseJson(Yson::ConvertToString(doc.same_day_data)).delivery_interval.to
        ))) AS delivery_period_end,
        IF(
            Yson::ConvertToString(doc.same_day_data) IS NOT NULL,
            'B2C',
            'C2C'
        ) AS order_type
    FROM $CLAIMS_ADD_INFO
    WHERE FALSE
        OR Yson::ConvertToString(doc.same_day_data) IS NOT NULL
        OR Yson::ConvertToString(doc.taxi_class) = 'sdd'
);

$claims_info = (
    SELECT
        claim_id,
        cargo_ref_id,
        utc_created_dttm,
        api_source_type,
        claim_source_type,
        corp_client_id,
        final_price_w_vat_cur,
        price_multiplier_val,
        status_code,
        tariff_zone,
        currency_code
    FROM RANGE($CLAIMS, $MTH_START, $MTH_END)
    WHERE TRUE
        AND SUBSTRING(utc_created_dttm, 0, 10) >= $DT_START
        AND SUBSTRING(utc_created_dttm, 0, 10) <= $DT_END
);

$performer_lookup_info = (
    SELECT
        claim_id,
        DateTime::MakeTimestamp(DateTime::ParseIso8601(
            MIN(IF(new_claim_status = 'performer_draft', utc_created_dttm))
        )) AS performer_lookup_start_dttm,
        DateTime::MakeTimestamp(DateTime::ParseIso8601(
            MAX(IF(new_claim_status = 'performer_found', utc_created_dttm))
        )) AS performer_lookup_end_dttm,
        DateTime::MakeTimestamp(DateTime::ParseIso8601(
            COALESCE(
                MAX(IF(new_claim_status = 'delivered', utc_created_dttm)),
                MAX(IF(new_claim_status = 'returned', utc_created_dttm)),
                MAX(IF(new_claim_status = 'delivered_finish', utc_created_dttm)),
                MAX(IF(new_claim_status = 'returned_finish', utc_created_dttm)),
                MAX(utc_created_dttm)
            )
        )) AS end_or_delivered_dttm
    FROM RANGE($CLAIM_STATUS_LOGS, $MTH_START, $MTH_END)
    WHERE TRUE
        AND SUBSTRING(utc_created_dttm, 0, 10) >= $DT_START
        AND SUBSTRING(utc_created_dttm, 0, 10) <= $DT_END
    GROUP BY claim_id
);

$taxi_exec_info = (
    SELECT
        claim_id,
        MAX_BY(tariff_class_code, utc_created_dttm) AS taxi_tariff_class_code,
        MAX_BY(taxi_order_id, utc_created_dttm) AS taxi_order_id
    FROM RANGE($TAXI_PERFORMERS_INFO, $MTH_START, $MTH_END)
    WHERE TRUE
        AND SUBSTRING(utc_created_dttm, 0, 10) >= $DT_START
        AND SUBSTRING(utc_created_dttm, 0, 10) <= $DT_END
    GROUP BY claim_id
);

$claim_points_data = (
    SELECT
        claim_point_id,
        point_id,
        claim_id,
        point_type,
        visit_status
    FROM RANGE($CLAIM_POINTS, $MTH_START, $MTH_END)
    WHERE TRUE
        AND point_type = 'destination'
        AND SUBSTRING(utc_created_dttm, 0, 10) >= $DT_START
        AND SUBSTRING(utc_created_dttm, 0, 10) <= $DT_END
);

$points_data = (
    SELECT
        point_id,
        city_name_ru,
        country_name_ru
    FROM RANGE($POINTS, $MTH_START, $MTH_END)
    WHERE TRUE
        AND substring(utc_created_dttm, 0, 10) >= $DT_START
        AND substring(utc_created_dttm, 0, 10) <= $DT_END
);

$corp_clients_dict = (
    SELECT
        id AS corp_client_id,
        MAX_BY(name, updated) AS corp_client_name
    FROM $CORP_CLIENT_NAMES
    GROUP BY id 
);

$courier_complaints = (
    SELECT
        DISTINCT(order_id) AS order_id,
        1 AS courier_has_complaint
    FROM RANGE($SUPPORT_TICKETS, $MTH_START, $MTH_END)
    WHERE TRUE
        AND calc_ticket_category_name = 'driver'
        AND tariff_class_name IN ['express', 'courier', 'night', 'sdd']
);

$curr_dict = (
    SELECT
        `date` AS dt,
        rate,
        source_cur
    FROM $CURRENCY_DATA
    WHERE target_cur = 'RUB'
);

$c2c_platform_data = (
    SELECT
        app_platform_name AS app_platform,
        public_claim_id
    FROM RANGE($C2C_ORDERS, $YEAR_START, $YEAR_END)
);

$dttm_diff = ($dttm_1, $dttm_2) -> {
    $sec_1 = DateTime::ToSeconds($dttm_1);
    $sec_2 = DateTime::ToSeconds($dttm_2);

    RETURN ($sec_2 - $sec_1) * 1.0 / 60
};

INSERT INTO $RESULT_PATH WITH TRUNCATE
SELECT
    T1.claim_id AS claim_id,
    COALESCE(
        $dttm_format(T1.delivery_period_start),
        T2.utc_created_dttm
    ) AS utc_created_dttm,
    T1.order_type AS order_type,
    T2.cargo_ref_id as cargo_ref_id,
    T2.utc_created_dttm AS claim_utc_created_dttm,
    IF(T1.order_type = 'C2C', T10.app_platform, T2.api_source_type) AS api_source_type,
    T2.claim_source_type AS claim_source_type,
    T2.corp_client_id AS corp_client_id,
    T2.final_price_w_vat_cur * T9.rate AS final_price_w_vat_cur,
    T2.price_multiplier_val AS price_multiplier_val,
    T2.status_code AS status_code,
    T2.tariff_zone AS tariff_zone,
    CAST(T1.delivery_period_end < T3.end_or_delivered_dttm AS Uint32) AS is_delivery_late,
    IF(
        T1.delivery_period_end < T3.end_or_delivered_dttm,
        $dttm_diff(T1.delivery_period_end, T3.end_or_delivered_dttm),
        NULL
    ) AS late_for_mnt,
    $dttm_diff(T3.performer_lookup_start_dttm, T3.performer_lookup_end_dttm) AS executor_search_time_mnt,
    T4.taxi_tariff_class_code AS taxi_tariff_class_code,
    T4.taxi_order_id as taxi_order_id,
    T5.claim_point_id AS claim_point_id,
    T5.point_id AS point_id,
    T5.point_type AS point_type,
    T5.visit_status AS visit_status,
    T6.city_name_ru as city_name_ru,
    T6.country_name_ru AS country_name_ru,
    T7.corp_client_name AS corp_client_name,
    COALESCE(T8.courier_has_complaint, 0) AS courier_has_complaint
FROM $sdd_claims AS T1
    LEFT JOIN $claims_info AS T2 ON T1.claim_id = T2.claim_id
    LEFT JOIN $performer_lookup_info AS T3 ON T1.claim_id = T3.claim_id
    LEFT JOIN $taxi_exec_info AS T4 ON T1.claim_id = T4.claim_id
    LEFT JOIN $claim_points_data AS T5 ON T1.claim_id = T5.claim_id
    LEFT JOIN $points_data AS T6 ON T5.point_id = T6.point_id
    LEFT JOIN $corp_clients_dict AS T7 ON T2.corp_client_id = T7.corp_client_id
    LEFT JOIN $courier_complaints AS T8 ON T4.taxi_order_id = T8.order_id
    LEFT JOIN $curr_dict AS T9 ON (
        SUBSTRING(T2.utc_created_dttm, 0, 10) = T9.dt
        AND T2.currency_code = T9.source_cur
    )
    LEFT JOIN $c2c_platform_data AS T10 ON T2.cargo_ref_id = T10.public_claim_id
