/*
    Назначение скрипта:
    Обновить финальную витрину по шэрингу в Go и SMS за прошлый месяц

    Автор:
    @shamilalburin

    Тикет:
    LOGDATA-1450
*/

PRAGMA AnsiInForEmptyOrNullableItemsCollections;
PRAGMA yson.DisableStrict;

USE hahn;

-- Working directory for this task

$WORKING_DIRECTORY = '//home/taxi-delivery/analytics/production/product/sms_and_go_sharing';

-- List of tables

$CARGO_LOGS = '//home/logfeller/logs/taxi-cargo-c2c-yandex-taxi-cargo-c2c-log/1d';
$SMS_LOGS = '//home/logfeller/logs/taxi-ucommunications-user-sms-log/1d';
$SMS_STATUS_LOG = '//home/logfeller/logs/taxi-ucommunications-user-sms-status-log/1d';
$PUSH_LOGS = '//home/logfeller/logs/taxi-ucommunications-user-notfications-push-log/1d';
$PUSH_STATUS_LOGS = '//home/logfeller/logs/taxi-ucommunications-user-notfications-push-status-log/1d';
$SHARING_EVENTS = $WORKING_DIRECTORY || '/sharing_from_appmetrica';
$USERS = '//home/taxi-dwh/ods/mdb/user/user';
$USER_PHONES = '//home/taxi-dwh/ods/mdb/user_phone/user_phone';
$METRICA = '//statbox/cooked_logs/visit-cooked-log/v1/1d';
$CLIENTS_ORDERS = '//home/taxi/production/replica/postgres/cargo_c2c/clients_orders';
$CARGO_ORDERS = '//home/taxi/production/replica/postgres/cargo_c2c/orders';
$CARGO_CLAIMS = '//home/delivery-dwh/ods/cargo_claims/claim';
$FCT_ORDER = '//home/taxi-dwh/cdm/marketplace/fct_order';
$USER_APPSESSION = '//home/taxi-dwh/cdm/demand/user_appsession';
$INSTALLATION_LOGS = '//logs/metrika-mobile-install-log/1d';

-- DateTime format

$dttm_format = DateTime::Format('%Y-%m-%d %H:%M:%S');
$dt_format = DateTime::Format('%Y-%m-%d');

-- Dates

$DT_START = $dt_format(DateTime::StartOfMonth(DateTime::ShiftMonths(CurrentUtcDate(), -1)));
$DT_END = $dt_format(
    DateTime::MakeDatetime((DateTime::StartOfMonth(CurrentUtcDateTime())))
    - DateTime::IntervalFromDays(1)
);
$DT_START_M_80 = $dt_format(DateTime::MakeTimestamp(DateTime::ParseIso8601($DT_START)) - Interval('P80D'));
$DT_END_P_7 = $dt_format(DateTime::MakeTimestamp(DateTime::ParseIso8601($DT_END)) + Interval('P7D'));
$MTH_START = SUBSTRING($DT_START, 0, 7) || '-01';
$MTH_END = SUBSTRING($DT_END, 0, 7) || '-01';
$MTH_START_M_6 = $dt_format(DateTime::ShiftMonths(DateTime::ParseIso8601($MTH_START), -6));
$MTH_END_P_1 = $dt_format(DateTime::ShiftMonths(DateTime::ParseIso8601($MTH_END), 1));;

-- Result path

$RESULT_PATH = $WORKING_DIRECTORY || '/data_for_tableau/' || $MTH_START;

-- Communication handler

$handler = (
    SELECT
        meta_order_id AS order_id,
        personal_phone_id AS phone_pd_id,
        CASE
            WHEN text LIKE '%order_provider_id = cargo-c2c%' THEN 'cargo-c2c'
            WHEN text LIKE '%order_provider_id = taxi%' THEN 'taxi'
            ELSE NULL
        END AS order_provider_id,
        CASE
            WHEN text LIKE 'sending sms or push cargo_c2c_recipient_new_order%'
                OR text LIKE 'sending sms or push cargo_c2c_sender_new_order%'
                THEN 'SmsOrPush'
            WHEN text LIKE 'sending sms cargo_c2c_recipient_new_order%'
                OR text LIKE 'sending sms cargo_c2c_sender_new_order%'
                THEN 'JustSms'
            ELSE NULL
        END AS handler
    FROM (
        SELECT
            meta_order_id,
            personal_phone_id,
            SOME(text) AS text
        FROM RANGE($CARGO_LOGS, $DT_START, $DT_END)
        WHERE TRUE
            AND (
                FALSE
                OR text LIKE 'sending sms or push cargo_c2c_recipient_new_order%'
                OR text LIKE 'sending sms or push cargo_c2c_sender_new_order%'
                OR text LIKE 'sending sms cargo_c2c_recipient_new_order%'
                OR text LIKE 'sending sms cargo_c2c_sender_new_order%'
            )
            AND (
                FALSE
                OR text LIKE '%order_provider_id = cargo-c2c%'
                OR text LIKE '%order_provider_id = taxi%'
            )
        GROUP BY
            meta_order_id,
            personal_phone_id
    )
);

-- SMS

$sms = (
    SELECT
        message_id,
        phone_id AS phone_pd_id,
        Yson::ConvertToString(Yson::ParseJson(meta).order_id) AS order_id,
        Yson::ConvertToString(Yson::ParseJson(meta).order_provider_id) AS order_provider_id,
    FROM RANGE($SMS_LOGS, $DT_START, $DT_END)
    WHERE TRUE
        AND status = 'sent'
        AND Yson::ConvertToString(Yson::ParseJson(meta).order_provider_id) IS NOT NULL
        AND Yson::ConvertToString(Yson::ParseJson(meta).order_id) IS NOT NULL
);

$sms_statuses = (
    SELECT
        DISTINCT(message_id),
        0 AS sms_delivered
    FROM RANGE($SMS_STATUS_LOG, $DT_START, $DT_END)
    WHERE status = 'undelivered'
);

$sms_total = (
    SELECT
        order_id,
        order_provider_id,
        phone_pd_id,
        SUM(COALESCE(sms_delivered, 1)) AS sms_delivered
    FROM $sms AS T1
        LEFT JOIN $sms_statuses AS T2 ON T1.message_id = T2.message_id
    GROUP BY
        T1.order_id AS order_id,
        T1.order_provider_id AS order_provider_id,
        T1.phone_pd_id AS phone_pd_id
);

-- Notifications

$notifications = (
    SELECT
        notification_id,
        IF(
            FALSE
            OR intent LIKE '%recipient_new_order%'
            OR intent LIKE '%sender_new_order%'
            OR intent LIKE '%recipient-new-order%'
            OR intent LIKE '%sender-new-order%',
            1,
            0
        ) AS new_order_notif,
        Yson::ConvertToString(Yson::ParseJson(meta).order_id) AS order_id,
        Yson::ConvertToString(Yson::ParseJson(meta).order_provider_id) AS order_provider_id,
        Yson::ConvertToString(Yson::ParseJson(meta).phone_pd_id) AS phone_pd_id
    FROM RANGE($PUSH_LOGS, $DT_START, $DT_END)
    WHERE TRUE
        AND Yson::ConvertToString(Yson::ParseJson(meta).order_provider_id) IS NOT NULL
        AND Yson::ConvertToString(Yson::ParseJson(meta).order_id) IS NOT NULL
        AND Yson::ConvertToString(Yson::ParseJson(meta).phone_pd_id) IS NOT NULL
);

$notif_statuses = (
    SELECT
        DISTINCT(notification_id),
        1 AS notif_delivered
    FROM RANGE($PUSH_STATUS_LOGS, $DT_START, $DT_END)
    WHERE status = 'delivered'
);

$notif_total = (
    SELECT
        order_id,
        order_provider_id,
        phone_pd_id,
        COUNT(IF(new_order_notif = 1, T1.notification_id)) AS new_order_notif_sent,
        SUM(IF(new_order_notif = 1, COALESCE(notif_delivered, 0))) AS new_order_notif_delivered,
        COUNT(IF(new_order_notif = 0, T1.notification_id)) AS other_notif_sent,
        SUM(IF(new_order_notif = 0, COALESCE(notif_delivered, 0))) AS other_notif_delivered
    FROM $notifications AS T1
        LEFT JOIN $notif_statuses AS T2 ON T1.notification_id = T2.notification_id
    GROUP BY
        T1.order_id AS order_id,
        T1.order_provider_id AS order_provider_id,
        T1.phone_pd_id AS phone_pd_id
);

-- All communications

$all_communications = (
    SELECT
        COALESCE(T1.order_id, T2.order_id) AS order_id,
        COALESCE(T1.order_provider_id, T2.order_provider_id) AS order_provider_id,
        COALESCE(T1.phone_pd_id, T2.phone_pd_id) AS phone_pd_id,
        COALESCE(sms_delivered, 0) AS sms_delivered,
        IF(sms_delivered > 0, 1, 0) AS notified_by_sms,
        COALESCE(new_order_notif_sent, 0) AS new_order_notif_sent,
        COALESCE(new_order_notif_delivered, 0) AS new_order_notif_delivered,
        COALESCE(other_notif_sent, 0) AS other_notif_sent,
        COALESCE(other_notif_delivered, 0) AS other_notif_delivered,
        CASE
            WHEN sms_delivered > 0 THEN IF(new_order_notif_sent > 0, 'SmsOrPush, SMS', 'JustSMS')
            ELSE IF(new_order_notif_sent > 0, 'SmsOrPush, Push', 'No communication')
        END AS communication_type
    FROM $sms_total AS T1
        FULL JOIN $notif_total AS T2 ON (
            T1.order_id = T2.order_id
            AND T1.order_provider_id = T2.order_provider_id
            AND T1.phone_pd_id = T2.phone_pd_id
        )
);

-- Go sharing

$go_sharing = (
    SELECT
        order_id,
        order_provider_id,
        phone_pd_id,
        MIN_OF(1, SUM(CAST(event_name LIKE '%PreviewOrderList%' AS Uint64))) AS order_list_shown,
        MIN(IF(event_name LIKE '%PreviewOrderList%', utc_event_dttm)) AS order_list_dttm,
        MIN_OF(1, SUM(CAST(event_name LIKE '%SharedOrderCard%' AS Uint64))) AS card_shown,
        MIN(IF(event_name LIKE '%SharedOrderCard%', utc_event_dttm)) AS card_dttm,
        MIN_OF(1, SUM(CAST(event_name LIKE '%SharedOrderCard.Opened%' AS Uint64))) AS card_opened,
        MIN(IF(event_name LIKE '%SharedOrderCard.Opened%', utc_event_dttm)) AS card_opened_dttm,
        MIN_OF(1, SUM(CAST(event_name LIKE '%SharedOrderPush%' AS Uint64))) AS push_shown
    FROM RANGE($SHARING_EVENTS, $MTH_START, $DT_END)
    GROUP BY
        order_id,
        order_provider_id,
        phone_pd_id
);

-- Sms sharing

$convenient_date = ($timestamp_in_sec, $date_format) -> {
    $dttm_timestamp = DateTime::FromSeconds(CAST($timestamp_in_sec AS Uint32));
    $convenient_format = CAST($date_format($dttm_timestamp) AS String);

    RETURN $convenient_format
};

$sms_sharing = (
    SELECT
        sharing_key,
        $convenient_date(MIN(UTCStartTime), $dttm_format) AS sms_tracking_dttm,
        1 AS sms_tracking_opened
    FROM RANGE($METRICA, $DT_START, $DT_END)
    WHERE TRUE
        AND CounterID = 62915509
        AND ListHas(Yson::ConvertToUint64List(Goals_ID), 228969032)
    GROUP BY Yson::ConvertToString(ParsedParams_Key2[0]) AS sharing_key
);

-- Collect all communication and sharing data

$order_roles_full_info = (
    SELECT
        T1.order_id AS order_id,
        T1.order_provider_id AS order_provider_id,
        T1.phone_pd_id AS phone_pd_id,
        T2.*,
        T3.*,
        T4.*,
        handler,
        IF(ListHas(Yson::ConvertToStringList(roles), 'recipient'), 1, 0) AS is_recipient,
        IF(ListHas(Yson::ConvertToStringList(roles), 'initiator'), 1, 0) AS is_initiator,
        IF(ListHas(Yson::ConvertToStringList(roles), 'sender'), 1, 0) AS is_sender
        WITHOUT
            T2.order_id,
            T2.order_provider_id,
            T2.phone_pd_id,
            T3.order_id,
            T3.order_provider_id,
            T3.phone_pd_id,
            T4.sharing_key
    FROM $CLIENTS_ORDERS AS T1
        LEFT JOIN $all_communications AS T2 ON (
            T1.order_id = T2.order_id
            AND T1.order_provider_id = T2.order_provider_id
            AND T1.phone_pd_id = T2.phone_pd_id
        )
        LEFT JOIN $go_sharing AS T3 ON (
            T1.order_id = T3.order_id
            AND T1.order_provider_id = T3.order_provider_id
            AND T1.phone_pd_id = T3.phone_pd_id
        )
        LEFT JOIN $sms_sharing AS T4 ON T1.sharing_key = T4.sharing_key
        LEFT JOIN $handler AS T5 ON (
            T1.order_id = T5.order_id
            AND T1.order_provider_id = T5.order_provider_id
            AND T1.phone_pd_id = T5.phone_pd_id
        )
    WHERE TRUE
        AND $convenient_date(created_ts, $dt_format) >= $DT_START
        AND $convenient_date(created_ts, $dt_format) <= $DT_END
);

-- Join data to taxi orders

$cargo_c2c_orders = (
    SELECT
        T1.*,
        cargo_ref_id
    FROM (
        SELECT *
        FROM $order_roles_full_info
        WHERE order_provider_id = 'cargo-c2c'
    ) AS T1
        LEFT JOIN (
            SELECT
                order_id,
                cargo_ref_id
            FROM $CARGO_ORDERS
        ) AS T2  ON T1.order_id = T2.order_id
);

$cargo_claims_orders = (
    SELECT
        T1.*,
        order_id AS cargo_ref_id
    FROM $order_roles_full_info AS T1
    WHERE order_provider_id = 'cargo-claims'
);

$non_taxi_orders = (
    SELECT
        taxi_order_id,
        T1.* WITHOUT T1.cargo_ref_id,
    FROM (
        SELECT *
        FROM $cargo_c2c_orders
        UNION ALL
        SELECT *
        FROM $cargo_claims_orders
    ) AS T1
        LEFT JOIN (
            SELECT
                taxi_order_id,
                cargo_ref_id
            FROM RANGE($CARGO_CLAIMS, $MTH_START, $MTH_END)
        ) AS T2 ON T1.cargo_ref_id = T2.cargo_ref_id
);

$all_orders_w_taxi_order_id = (
    SELECT
        T1.*,
        1 AS is_in_clients_orders
    FROM (
        SELECT
            T1.*,
            order_id AS taxi_order_id
        FROM $order_roles_full_info AS T1
        WHERE order_provider_id = 'taxi'
        UNION ALL
        SELECT *
        FROM $non_taxi_orders
    ) AS T1
);

-- All deliveries with additional info

$end_time = ($start_time, $completion_time_mnt) -> {
    $start_time_tstmp = DateTime::MakeTimestamp(DateTime::ParseIso8601($start_time));
    $end_time_sec = (
        DateTime::ToSeconds($start_time_tstmp)
        + ($completion_time_mnt + 3) * 60 -- add 3 minutes to be sure
    );
    $end_time = $dttm_format(Datetime::FromSeconds(CAST($end_time_sec AS Uint32)));

    RETURN CAST($end_time AS String)
};

$all_deliveries = (
    SELECT
        order_id AS taxi_order_id,
        utc_order_created_dttm,
        $end_time(utc_order_created_dttm, order_completion_time_mnt) AS order_ended_dttm,
        IF(COALESCE(corp_order_flg, FALSE), 1, 0) AS is_corp_order,
        T1.corp_client_id AS corp_client_id,
        order_source_code,
        tariff_class_code,
        COALESCE(skip_client_notify_flg, 0) AS skip_client_notify_flg,
        destination_plan_geohash_cnt,
        IF(destination_plan_geohash_cnt > 1, 1, 0) AS is_multipoint_order,
        tariff_geo_zone_code AS geo_zone_code,
        country_name_ru,
        user_phone_pd_id
    FROM RANGE($FCT_ORDER, $MTH_START, $MTH_END) AS T1
        LEFT JOIN (
            SELECT
                taxi_order_id,
                SOME(IF(skip_client_notify_flg, 1, 0)) AS skip_client_notify_flg
            FROM RANGE($CARGO_CLAIMS, $MTH_START, $MTH_END)
            WHERE TRUE
                AND claim_id IS NOT NULL
                AND SUBSTRING(utc_created_dttm, 0, 10) >= $DT_START
                AND SUBSTRING(utc_created_dttm, 0, 10) <= $DT_END
            GROUP BY taxi_order_id
        ) AS T2 ON T1.order_id = T2.taxi_order_id
    WHERE TRUE
        AND SUBSTRING(utc_order_created_dttm, 0, 10) >= $DT_START
        AND SUBSTRING(utc_order_created_dttm, 0, 10) <= $DT_END
        AND delivery_point_cnt > 0
        AND tariff_class_code IN ['express', 'courier', 'cargo', 'cargocorp']
        AND NOT fake_order_flg
        AND NOT mqc_order_flg
);

-- Join deliveries and roles info

$final_result_without_activity = (
    SELECT
        T1.*,
        IF(
            TRUE
            AND order_source_code = 'yandex'
            AND tariff_class_code IN ['express', 'courier', 'cargo']
            AND is_in_clients_orders IS NULL,
            1,
            COALESCE(is_in_clients_orders, 0)
        ) AS is_in_clients_orsers, -- orders with only initiator don't appear in clients_orders table
                                   -- this flag will be used to show whether order is growing or not
         IF(
            TRUE
            AND order_source_code = 'yandex'
            AND tariff_class_code IN ['express', 'courier', 'cargo']
            AND is_in_clients_orders IS NULL,
            user_phone_pd_id,
            phone_pd_id
        ) AS phone_pd_id,
        IF(
            TRUE
            AND order_source_code = 'yandex'
            AND tariff_class_code IN ['express', 'courier', 'cargo']
            AND is_in_clients_orders IS NULL,
            1,
            is_initiator
        ) AS is_initiator,
        IF(card_shown = 1 AND card_dttm <= order_ended_dttm, 1, 0) AS card_shown,
        IF(card_opened = 1 AND card_opened_dttm <= order_ended_dttm, 1, 0) AS card_opened,
        IF(order_list_shown = 1 AND order_list_dttm <= order_ended_dttm, 1, 0) AS order_list_shown,
        IF(sms_tracking_opened = 1 AND sms_tracking_dttm <= order_ended_dttm, 1, 0) AS sms_tracking_opened,
        IF(
            IF(order_list_shown = 1 AND order_list_dttm <= order_ended_dttm, 1, 0) = 1
            OR IF(card_shown = 1 AND card_dttm <= order_ended_dttm, 1, 0) = 1,
            1,
            0
        ) AS Go_app_opened,
        IF(push_shown + card_shown + card_opened + order_list_shown > 0, 1, 0) AS any_Go_activity,
        T2.*
        WITHOUT
            T2.is_in_clients_orders,
            T2.is_initiator,
            T2.phone_pd_id,
            T2.card_shown,
            T2.card_dttm,
            T2.push_shown,
            T2.card_opened,
            T2.card_opened_dttm,
            T2.order_list_shown,
            T2.order_list_dttm,
            T2.sms_tracking_opened,
            T2.sms_tracking_dttm
    FROM $all_deliveries AS T1
        LEFT JOIN $all_orders_w_taxi_order_id AS T2 ON T1.taxi_order_id = T2.taxi_order_id
);

-- User activity data

$user_to_phone_pd = (
    SELECT
        user_id,
        phone_pd_id,
        yandex_uuid,
        T1.user_phone_id AS phone_id
    FROM (
        SELECT
            user_id,
            yandex_uuid,
            user_phone_id
        FROM $USERS
    ) AS T1
        INNER JOIN (
            SELECT
                user_phone_id,
                phone_pd_id
            FROM $USER_PHONES
        ) AS T2 ON T1.user_phone_id = T2.user_phone_id
);

$user_activity = (
    SELECT
        phone_pd_id,
        utc_session_start_dttm AS activity_time
    FROM RANGE($USER_APPSESSION, $DT_START_M_80, $DT_END) AS T1
    FLATTEN BY (Yson::ConvertToStringList(user_id_list) AS user_id)
    INNER JOIN $user_to_phone_pd AS T2 ON T1.user_id = T2.user_id
);

$time_diff = ($dttm1, $dttm2) -> {
    $tstmp1 = DateTime::MakeTimestamp(DateTime::ParseIso8601($dttm1));
    $tstmp2 = DateTime::MakeTimestamp(DateTime::ParseIso8601($dttm2));
    $sec1 = CAST(DateTime::ToSeconds($tstmp1) AS int64);
    $sec2 = CAST(DateTime::ToSeconds($tstmp2) AS int64);

    RETURN ($sec2 - $sec1) * 1.0 / (60 * 60 * 24)
};

$activity_data =(
    SELECT
        taxi_order_id,
        phone_pd_id,
        1 AS go_is_active
    FROM $final_result_without_activity AS T1
    LEFT JOIN $user_activity AS T2 ON T1.phone_pd_id = T2.phone_pd_id
    WHERE TRUE
        AND $time_diff(activity_time, utc_order_created_dttm) <= 80
        AND $time_diff(activity_time, utc_order_created_dttm) >= 0
    GROUP BY
        T1.taxi_order_id AS taxi_order_id,
        T1.phone_pd_id AS phone_pd_id
);

-- Reactivation data

$reactivation_data = (
    SELECT
        taxi_order_id,
        phone_pd_id,
        CAST(COUNT(IF(
            TRUE
            AND $time_diff(T2.utc_order_created_dttm, T1.utc_order_created_dttm) <= 30
            AND $time_diff(T2.utc_order_created_dttm, T1.utc_order_created_dttm) > 0,
            T2.order_id
        )) > 0 AS Uint32) AS order_during_last_1_m,
        CAST(COUNT(IF(
            TRUE
            AND $time_diff(T2.utc_order_created_dttm, T1.utc_order_created_dttm) <= 90
            AND $time_diff(T2.utc_order_created_dttm, T1.utc_order_created_dttm) > 0,
            T2.order_id
        )) > 0 AS Uint32) AS order_during_last_3_m,
        CAST(COUNT(IF(
            TRUE
            AND $time_diff(T2.utc_order_created_dttm, T1.utc_order_created_dttm) <= 180
            AND $time_diff(T2.utc_order_created_dttm, T1.utc_order_created_dttm) > 0,
            T2.order_id
        )) > 0 AS Uint32) AS order_during_last_6_m,
        CAST(COUNT(IF(
            TRUE
            AND $time_diff(T2.utc_order_created_dttm, T1.utc_order_created_dttm) >= -1
            AND $time_diff(T2.utc_order_created_dttm, T1.utc_order_created_dttm) < 0,
            T2.order_id
        )) > 0 AS Uint32) AS order_during_next_1_d,
        CAST(COUNT(IF(
            TRUE
            AND $time_diff(T2.utc_order_created_dttm, T1.utc_order_created_dttm) >= -3
            AND $time_diff(T2.utc_order_created_dttm, T1.utc_order_created_dttm) < 0,
            T2.order_id
        )) > 0 AS Uint32) AS order_during_next_3_d,
        CAST(COUNT(IF(
            TRUE
            AND $time_diff(T2.utc_order_created_dttm, T1.utc_order_created_dttm) >= -7
            AND $time_diff(T2.utc_order_created_dttm, T1.utc_order_created_dttm) < 0,
            T2.order_id
        )) > 0 AS Uint32) AS order_during_next_7_d
    FROM $final_result_without_activity AS T1
    LEFT JOIN RANGE($FCT_ORDER, $MTH_START_M_6, $MTH_END_P_1) AS T2 ON T1.phone_pd_id = T2.user_phone_pd_id
    GROUP BY
        T1.taxi_order_id AS taxi_order_id,
        T1.phone_pd_id AS phone_pd_id
);

-- Go installation data

$installation_data = (
    SELECT
        taxi_order_id,
        phone_pd_id,
        CAST(COUNT(IF(
            TRUE
            AND $time_diff(utc_order_created_dttm, inst_dttm) <= 1
            AND $time_diff(utc_order_created_dttm, inst_dttm) > 0,
            1
        )) > 0 AS Uint32) AS install_during_next_1_d,
        CAST(COUNT(IF(
            TRUE
            AND $time_diff(utc_order_created_dttm, inst_dttm) <= 3
            AND $time_diff(utc_order_created_dttm, inst_dttm) > 0,
            1
        )) > 0 AS Uint32) AS install_during_next_3_d,
        CAST(COUNT(IF(
            TRUE
            AND $time_diff(utc_order_created_dttm, inst_dttm) <= 7
            AND $time_diff(utc_order_created_dttm, inst_dttm) > 0,
            1
        )) > 0 AS Uint32) AS install_during_next_7_d
    FROM $final_result_without_activity AS T1
    LEFT JOIN $user_to_phone_pd AS T2 ON T1.phone_pd_id = T2.phone_pd_id
    LEFT JOIN (
        SELECT
            $dttm_format(DateTime::FromSeconds(CAST(InstallationTimestamp AS Uint32))) AS inst_dttm,
            `UUID` AS yandex_uuid
        FROM RANGE($INSTALLATION_LOGS, $DT_START, $DT_END_P_7)
    ) AS T3 ON T2.yandex_uuid = T3.yandex_uuid
    GROUP BY
        T1.taxi_order_id AS taxi_order_id,
        T1.phone_pd_id AS phone_pd_id
);

-- Collect final table

$final_result = (
    SELECT
        T1.*,
        COALESCE(go_is_active, 0) AS go_is_active,
        COALESCE(order_during_last_1_m, 0) AS order_during_last_1_m,
        COALESCE(order_during_last_3_m, 0) AS order_during_last_3_m,
        COALESCE(order_during_last_6_m, 0) AS order_during_last_6_m,
        COALESCE(order_during_next_1_d, 0) AS order_during_next_1_d,
        COALESCE(order_during_next_3_d, 0) AS order_during_next_3_d,
        COALESCE(order_during_next_7_d, 0) AS order_during_next_7_d,
        COALESCE(install_during_next_1_d, 0) AS install_during_next_1_d,
        COALESCE(install_during_next_3_d, 0) AS install_during_next_3_d,
        COALESCE(install_during_next_7_d, 0) AS install_during_next_7_d
    FROM $final_result_without_activity AS T1
    LEFT JOIN $activity_data AS T2 ON (
        T1.taxi_order_id = T2.taxi_order_id
        AND T1.phone_pd_id = T2.phone_pd_id
    )
    LEFT JOIN $reactivation_data AS T3 ON (
        T1.taxi_order_id = T3.taxi_order_id
        AND T1.phone_pd_id = T3.phone_pd_id
    )
    LEFT JOIN $installation_data AS T4 ON (
        T1.taxi_order_id = T4.taxi_order_id
        AND T1.phone_pd_id = T4.phone_pd_id
    )
);

INSERT INTO $RESULT_PATH WITH TRUNCATE
SELECT *
FROM $final_result;
