import os

import pandas as pd
import numpy as np

import yt.wrapper as yt

FEATURES_LIST = [
        'delivery_orders_sum', # кол-во заказов в express и курьер за 52 недели
        'delivery_maximin_3', # маскимум из минимумов, которые рассчитаны окном в 3 месяца
        'delivery_maximin_6',  # маскимум из минимумов, которые рассчитаны окном в 6 месяцев
        'delivery_maximin_12',  # маскимум из минимумов, которые рассчитаны окном в 12 месяцев (одно такое окно)
        'delivery_orders_max', # максимум среди всех 4х недельных периодов, следующих друг за другом
        'delivery_orders_mean', # среднее среди всех 4х недельных периодов, следующих друг за другом
        'delivery_orders_std', # дисперсия среди всех 4х недельных периодов, следующих друг за другом
        'active_month_0', # кол-во 4х недельных периодов, следующих друг за другом, когда было больше 0 трипов
        'active_month_1', # кол-во 4х недельных периодов, следующих друг за другом, когда было больше 1 трипа
        'active_month_3', # кол-во 4х недельных периодов, следующих друг за другом, когда было больше 3 трипов
        'active_month_6', # кол-во 4х недельных периодов, следующих друг за другом, когда было больше 6 трипов
        'normalized_delivery_orders_sum', # features['delivery_orders_sum'] / (features['active_month_0'] + 1)
        'door_to_door_share', # доля заказов Доставки с опцией "от двери до двери"
        'smoothed_ba', # сглаженное отношение уникальных точек Б к уникальным точкам А
        'corp_delivery_orders_share', # доля заказов в Доставке, которые мы считаем корпоративными
        'city_name_ru', # Аггломерация, из которой чаще всего заказы Доставки идут
        'population_group', # Размер аггломерации
        'delivery_share_in_total', # Доля заказов Доставки во заказах Такси и Доставки
        'share_working_days', # Доля заказов Доставки, совершенная в рабочие дни (пн - пт)
        'share_working_time', # Доля заказов Доставки, совершенная в рабочее время (9 - 17 в локальном времени)
        'share_delivery_cancel_before', # Доля заказов, отмененных до назначения исполнителя
        'share_delivery_cancel_after' # Доля заказов, отмененных после назначения исполнителя
    #     'active_month_12', # кол-во 4х недельных периодов, следующих друг за другом, когда было больше 12 трипов
    #     'thermobag_share', # кол-во 4х недельных периодов, следующих друг за другом, когда было больше 6 трипов      
    #     'country_name_ru', # доля заказов Доставки с опцией "термосумка"
    #     'rubric', # Рубрика из Справочника
    #     'in_spravochnik', # Наличие телефона в Справочнике
    #     'in_caller', # Наличие телефона в АОН 
    ]

def fill_na(df):
    # features = df.copy()
    features = df
    
    features['city_name_ru'] = np.where(
        features['city_name_ru'].isna(), 
        'unknown city', 
        features['city_name_ru']
    )

    features['population_group'] = np.where(
        features['population_group'].isna(), 
        'XX_unknown', 
        features['population_group']
    )
    features['population_group'] = features['population_group'].str[3:]

    features['total_avg_points_in_order'] = np.where(
        features['total_avg_points_in_order'].isna(), 
        1, 
        features['total_avg_points_in_order']
    )

    features['top_freq_b'] = np.where(
        features['top_freq_b'].isna(), 
        'unknown b', 
        features['top_freq_b']
    )

    features = features.fillna(0)
    
    return features
    
def create_new_features(df):
    # features = df.copy()
    features = df
    
    delivery_orders = [
        'total_delivery_4week_orders',
        'total_delivery_8week_orders',
        'total_delivery_12week_orders',
        'total_delivery_16week_orders',
        'total_delivery_20week_orders',
        'total_delivery_24week_orders',
        'total_delivery_28week_orders',
        'total_delivery_32week_orders',
        'total_delivery_36week_orders',
        'total_delivery_40week_orders',
        'total_delivery_44week_orders',
        'total_delivery_48week_orders',
        'total_delivery_52week_orders',
    ]
    
    # SUM
    features['delivery_orders_sum'] = np.sum(features[delivery_orders], axis=1)
    
    # Share Cancel before
    features['share_delivery_cancel_before'] = (
        features['delivery_cancelled_by_client_before'] + \
        features['corp_delivery_cancelled_by_client_before']
    ) / (features['delivery_orders_sum'] + 1)

    # Share Cancel after
    features['share_delivery_cancel_after'] = (
        features['delivery_cancelled_by_client_after'] + \
        features['corp_delivery_cancelled_by_client_after']
    ) / (features['delivery_orders_sum'] + 1)
    
    # MAX_MIN
    periods = [3, 6, 12]

    maximin_dict = {}
    for period in periods:
        period_mins = []
        for i in range(13-period+1):
            delivery_buckets = delivery_orders[i:i+period]

            mins = np.min(features[delivery_buckets], axis=1)

            period_mins.append(mins)
        maximin_dict[f'delivery_maximin_{period}'] = np.max(period_mins, axis=0)

    for key, value in maximin_dict.items():
        features[key] = value
        
    # MAX
    features['delivery_orders_max'] = np.max(features[delivery_orders], axis=1)

    # MEAN
    features['delivery_orders_mean'] = np.mean(features[delivery_orders], axis=1)

    # STD
    features['delivery_orders_std'] = np.std(features[delivery_orders], axis=1)
    
    # Active month
    active_month_counts = [0, 1, 3, 6, 12]

    am_dict = {}

    for count in active_month_counts:
        active_months = np.zeros(len(features))
        for i in delivery_orders:
             active_months += features[i] > count
        am_dict[f'active_month_{count}'] = active_months

    for key, value in am_dict.items():
        features[key] = value
        
    # Orders Normalization
    features['normalized_delivery_orders_sum'] = features['delivery_orders_sum'] / \
                                                 (features['active_month_0'] + 1)
    
    # requirements shares
    features['door_to_door_share'] = features['total_doortodoor_cnt'] / \
                                     (features['delivery_orders_sum'] + 1)
    features['thermobag_share'] = features['total_thermobag_cnt'] / \
                                  (features['delivery_orders_sum'] + 1)
    
    # smoothed B/A ratio
    features['smoothed_ba'] = (
        features['delivery_uniq_b'] + np.mean(features['delivery_uniq_b'])
    ) / (
        features['delivery_uniq_a'] + np.mean(features['delivery_uniq_a'])
    )
    
    # corp deliveries share
    features['corp_delivery_orders_share'] = features['corp_delivery_orders'] / \
        (features['corp_delivery_orders'] + features['delivery_orders'])

    # delivery share in total orders
    features['delivery_share_in_total'] = features['delivery_orders_sum'] / \
        (features['not_delivery_orders'] + features['delivery_orders_sum'] + 1)
    
    # share in working days
    features['share_working_days'] = features['delivery_orders_in_working_days_cnt'] / \
                                     (df['delivery_orders'] + 1)

    # share in working time 
    features['share_working_time'] = features['delivery_orders_in_working_time_cnt'] / \
                                     (features['delivery_orders'] + 1)
    
    return features

def get_data(table_path):
    ytc = yt.YtClient(proxy='hahn.yt.yandex.net', token=os.environ['YT_TOKEN'])

    df = pd.DataFrame(list(ytc.read_table(ytc.TablePath(table_path))))
    
    return create_new_features(fill_na(df))
