-- $start_date
CREATE TEMPORARY TABLE start_date
    ON COMMIT DROP
AS (
    SELECT (NOW() - INTERVAL '6 month')::Date AS start_date
) DISTRIBUTED RANDOMLY;
ANALYZE start_date;


-- $end_date
CREATE TEMPORARY TABLE end_date
    ON COMMIT DROP
AS (
    SELECT (NOW() - INTERVAL '1 day')::Date AS end_date
) DISTRIBUTED RANDOMLY;
ANALYZE end_date;


-- $link_ticket_order_act
CREATE TEMPORARY TABLE link_ticket_order_act
    ON COMMIT DROP
AS (
    SELECT ticket_id
         , max(country_name_ru)       AS country
         , max(agglomeration_name_ru) AS city
         , max(corp_client_id)        AS corp_client_id
    FROM taxi_cdm_contactcenter.link_ticket_order_act
    GROUP BY ticket_id
);


-- $logistics_sector_meta_information
CREATE TEMPORARY TABLE logistics_sector_meta_information
    ON COMMIT DROP
AS (
    SELECT sector_code
         , max(service_name_en) AS service_name_en
    FROM taxi_ods_support_gd.sector_meta_information
    WHERE service_name_en = 'logistics'
    GROUP BY sector_code
);


-- $fct_ticket_session
CREATE TEMPORARY TABLE fct_ticket_session
    ON COMMIT DROP
AS (
    SELECT a.utc_first_event_dttm  AS created
         , a.ticket_id             AS ticket_id
         , a.last_sector_name      AS sector
         , a.first_meta_event_name AS first_event
         , a.csat_value            AS csat
         , b.country               AS country
         , b.city                  AS city
         , b.corp_client_id        AS corp_client_id
    FROM taxi_cdm_contactcenter.fct_ticket_session a
             LEFT JOIN link_ticket_order_act b
                       ON b.ticket_id = a.ticket_id
    WHERE a.utc_first_event_dttm::DATE between (SELECT start_date FROM start_date) and (SELECT end_date FROM end_date)
) DISTRIBUTED BY (ticket_id);
ANALYZE fct_ticket_session;


-- $chatterbox_history
CREATE TEMPORARY TABLE chatterbox_history
    ON COMMIT DROP
AS (
    SELECT a.utc_event_dttm          AS created
         , a.event_number
         , CASE
               WHEN a.meta_event_name IN ('create', 'appear', 'reopen', 'reopen_after_dismiss', 'new_message')
                   THEN a.utc_event_dttm
        END                          AS creation_dttm
         , CASE
               WHEN a.meta_event_name IN ('dismiss', 'autoreply', 'export') OR
                    (meta_event_name = 'close' AND
                     event_comment_hidden NOT ILIKE '%%Task%%closed because source%%is closed%%')
                   THEN date_trunc('hour', a.utc_event_dttm)
        END                          AS close_dttm
         , a.ticket_id               AS ticket_id
         , a.sector_name             AS sector
         , a.csat_value              AS csat
         , a.meta_event_name         AS meta_action
         , a.event_comment           AS comment
         , a.event_comment_hidden    AS hidden_comment
         , a.staff_login             AS login
         , a.full_event_tag_list     AS added_tags
         , a.full_event_tag_list     AS all_added_tags
         , a.utc_call_answered_dttm  AS call_answered_at
         , a.utc_call_completed_dttm AS call_completed_at
         , b.target_sla_mnt * 60     AS sla_good
         , d.country                 AS country
         , d.city                    AS city
         , d.corp_client_id          AS corp_client_id
    FROM taxi_cdm_contactcenter.fct_ticket_action_markup a
             LEFT JOIN taxi_ods_chatterbox_support_taxi.sector_mapping b
                       ON b.sector_name = a.sector_name
        --              LEFT JOIN logistics_sector_meta_information c
--                        ON c.sector_code = a.sector_line
             LEFT JOIN link_ticket_order_act d
                       ON d.ticket_id = a.ticket_id
    WHERE a.utc_event_dttm::DATE between (SELECT start_date FROM start_date) and (SELECT end_date FROM end_date)
) DISTRIBUTED BY (ticket_id);
ANALYZE chatterbox_history;


-- $max_created
CREATE TEMPORARY TABLE max_created
    ON COMMIT DROP
AS (
    SELECT MAX(created) AS max_created
    FROM chatterbox_history
) DISTRIBUTED RANDOMLY;
ANALYZE max_created;


-- $current_datetime
CREATE TEMPORARY TABLE current_datetime
    ON COMMIT DROP
AS (
    SELECT NOW() AS current_datetime
) DISTRIBUTED RANDOMLY;
ANALYZE current_datetime;


-- $first_reply
CREATE TEMPORARY TABLE first_reply
    ON COMMIT DROP
AS (
    SELECT created
         , event_number
         , creation_dttm
         , CASE
               WHEN meta_action IN
                    ('create', 'new_message', 'reopen', 'reopen_after_dismiss', 'answer_on_proactive') AND
                    (LEAD(meta_action) OVER w IN ('tech_close', 'finish_archive'))
                   THEN date_trunc('hour', last_value(created) OVER w)
               WHEN
                           meta_action IN
                           ('create', 'new_message', 'reopen', 'reopen_after_dismiss', 'answer_on_proactive')
                       AND (LEAD(meta_action) OVER w IN
                            ('dismiss', 'comment', 'communicate', 'call_success', 'close', 'temp_answer', 'comment_ml',
                             'communicate_ml',
                             'close_ml', 'defer_with_answer', 'close_backlog'))
                   THEN date_trunc('hour', LEAD(created) OVER w)
               ELSE NULL
        END AS close_dttm
         , last_meta_action
         , sla_good
         , ticket_id
         , country
         , city
         , corp_client_id
         , CASE
               WHEN
                           meta_action IN
                           ('create', 'new_message', 'reopen', 'reopen_after_dismiss', 'answer_on_proactive')
                       AND LEAD(meta_action) OVER w IN
                           ('dismiss', 'comment', 'communicate', 'call_success', 'close', 'temp_answer', 'comment_ml',
                            'communicate_ml',
                            'close_ml', 'defer_with_answer', 'close_backlog')
                   THEN LEAD(sector) OVER w
               WHEN (LEAD(meta_action) OVER w IS NULL AND (lead_meta_action != 'appear' OR lead_meta_action IS NULL))
                   THEN sector
               WHEN LEAD(meta_action) OVER w IS NULL AND lead_meta_action = 'appear' THEN last_sector
        END AS sector
         , meta_action
         , added_tags
         , CASE
               WHEN meta_action IN
                    ('create', 'new_message', 'reopen', 'reopen_after_dismiss', 'answer_on_proactive') AND
                    (LEAD(meta_action) OVER w IS NULL
                        OR LEAD(meta_action) OVER w = 'close_backlog')
                   THEN NULL
               WHEN
                           meta_action IN
                           ('create', 'new_message', 'reopen', 'reopen_after_dismiss', 'answer_on_proactive')
                       AND LEAD(meta_action) OVER w IN
                           ('dismiss', 'comment', 'communicate', 'call_success', 'close', 'temp_answer', 'comment_ml',
                            'communicate_ml',
                            'close_ml', 'defer_with_answer')
                   THEN (LEAD(created) OVER w - created) / INTERVAL '1 second'
               ELSE -1
        END AS first_reply
         , (
                meta_action IN ('create', 'new_message', 'reopen', 'reopen_after_dismiss', 'answer_on_proactive') AND
                (LEAD(meta_action) OVER w IS NULL) AND
                ((SELECT max_created FROM max_created) - created) / INTERVAL '1 second' <= COALESCE(sla_good, 0)
        )   AS backlog_good_sla
    FROM (
             SELECT created
                  , event_number
                  , creation_dttm
                  , meta_action
                  , sla_good
                  , ticket_id
                  , country
                  , city
                  , corp_client_id
                  , added_tags
                  , sector
                  , LEAD(meta_action) OVER w       AS lead_meta_action
                  , last_value(meta_action) OVER w AS last_meta_action
                  , last_value(sector) OVER w      AS last_sector
             FROM chatterbox_history
             WHERE meta_action IN
                   ('dismiss', 'create', 'appear', 'new_message', 'reopen', 'reopen_after_dismiss',
                    'answer_on_proactive', 'comment', 'communicate', 'call_success',
                    'close', 'temp_answer', 'comment_ml', 'communicate_ml', 'close_ml', 'defer_with_answer',
                    'close_backlog',
                    'tech_close', 'proactive_message', 'finish_archive')
               AND (all_added_tags IS NULL OR 'lvk_us__first_response' != ALL (all_added_tags)) --TAXIDWH-15035
                 WINDOW w AS (PARTITION BY ticket_id
                     ORDER BY event_number
                     ROWS BETWEEN CURRENT ROW AND UNBOUNDED FOLLOWING
                     )
         ) t
    WHERE meta_action IN
          ('create', 'new_message', 'reopen', 'reopen_after_dismiss', 'answer_on_proactive', 'dismiss',
           'comment', 'communicate', 'call_success', 'close', 'temp_answer', 'comment_ml', 'communicate_ml', 'close_ml',
           'defer_with_answer', 'close_backlog', 'tech_close', 'proactive_message', 'finish_archive')
        WINDOW w AS (PARTITION BY ticket_id
            ORDER BY event_number)
) DISTRIBUTED BY (ticket_id);
ANALYZE first_reply;


-- $first_reply__total
CREATE TEMPORARY TABLE first_reply__total
    ON COMMIT DROP
AS (
    SELECT country
         , city
         , corp_client_id
         , sector
         , date_trunc('hour', created)                                  AS creation_dt
         , 'first_reply__5m'::VARCHAR                                   AS metric_name
         , SUM(CASE WHEN first_reply <= 300 THEN 1 ELSE 0 END)::NUMERIC AS metric_value
         , count(*)::BIGINT                                             AS sub_metric_value
    FROM first_reply
    WHERE first_reply != -1
    GROUP BY country, city, corp_client_id, sector, date_trunc('hour', created)
    UNION ALL
    SELECT country
         , city
         , corp_client_id
         , sector
         , date_trunc('hour', created)                                  AS creation_dt
         , 'first_reply__5m_perc'::VARCHAR                              AS metric_name
         , SUM(CASE WHEN first_reply <= 300 THEN 1 ELSE 0 END)::NUMERIC AS metric_value
         , count(*)::BIGINT                                             AS sub_metric_value
    FROM first_reply
    WHERE first_reply != -1
    GROUP BY country, city, corp_client_id, sector, date_trunc('hour', created)
    UNION ALL
    SELECT country
         , city
         , corp_client_id
         , sector
         , date_trunc('hour', created)                                                        AS creation_dt
         , 'first_reply__10m'::VARCHAR                                                        AS metric_name
         , SUM(CASE WHEN first_reply > 300 and first_reply <= 600 THEN 1 ELSE 0 END)::NUMERIC AS metric_value
         , count(*)::BIGINT                                                                   AS sub_metric_value
    FROM first_reply
    WHERE first_reply != -1
    GROUP BY country, city, corp_client_id, sector, date_trunc('hour', created)
    UNION ALL
    SELECT country
         , city
         , corp_client_id
         , sector
         , date_trunc('hour', created)                                  AS creation_dt
         , 'first_reply__10m_perc'::VARCHAR                             AS metric_name
         , SUM(CASE WHEN first_reply <= 600 THEN 1 ELSE 0 END)::NUMERIC AS metric_value
         , count(*)::BIGINT                                             AS sub_metric_value
    FROM first_reply
    WHERE first_reply != -1
    GROUP BY country, city, corp_client_id, sector, date_trunc('hour', created)
    UNION ALL
    SELECT country
         , city
         , corp_client_id
         , sector                                                                                 sector
         , date_trunc('hour', created)                                                         AS creation_dt
         , 'first_reply__1h'::VARCHAR                                                          AS metric_name
         , SUM(CASE WHEN first_reply > 600 and first_reply <= 3600 THEN 1 ELSE 0 END)::NUMERIC AS metric_value
         , count(*)::BIGINT                                                                    AS sub_metric_value
    FROM first_reply
    WHERE first_reply != -1
    GROUP BY country, city, corp_client_id, sector, date_trunc('hour', created)
    UNION ALL
    SELECT country
         , city
         , corp_client_id
         , sector
         , date_trunc('hour', created)                                   AS creation_dt
         , 'first_reply__1h_perc'::VARCHAR                               AS metric_name
         , SUM(CASE WHEN first_reply <= 3600 THEN 1 ELSE 0 END)::NUMERIC AS metric_value
         , count(*)::BIGINT                                              AS sub_metric_value
    FROM first_reply
    WHERE first_reply != -1
    GROUP BY country, city, corp_client_id, sector, date_trunc('hour', created)
    UNION ALL
    SELECT country
         , city
         , corp_client_id
         , sector                                                                                  sector
         , date_trunc('hour', created)                                                          AS creation_dt
         , 'first_reply__2h'::VARCHAR                                                           AS metric_name
         , SUM(CASE WHEN first_reply > 3600 and first_reply <= 7200 THEN 1 ELSE 0 END)::NUMERIC AS metric_value
         , count(*)::BIGINT                                                                     AS sub_metric_value
    FROM first_reply
    WHERE first_reply != -1
    GROUP BY country, city, corp_client_id, sector, date_trunc('hour', created)
    UNION ALL
    SELECT country
         , city
         , corp_client_id
         , sector
         , date_trunc('hour', created)                                   AS creation_dt
         , 'first_reply__2h_perc'::VARCHAR                               AS metric_name
         , SUM(CASE WHEN first_reply <= 7200 THEN 1 ELSE 0 END)::NUMERIC AS metric_value
         , count(*)::BIGINT                                              AS sub_metric_value
    FROM first_reply
    WHERE first_reply != -1
    GROUP BY country, city, corp_client_id, sector, date_trunc('hour', created)
    UNION ALL
    SELECT country
         , city
         , corp_client_id
         , sector
         , date_trunc('hour', created)                                  AS creation_dt
         , 'first_reply__more_2h'::VARCHAR                              AS metric_name
         , SUM(CASE WHEN first_reply > 7200 THEN 1 ELSE 0 END)::NUMERIC AS metric_value
         , count(*)::BIGINT                                             AS sub_metric_value
    FROM first_reply
    WHERE first_reply != -1
    GROUP BY country, city, corp_client_id, sector, date_trunc('hour', created)
    UNION ALL
    SELECT country
         , city
         , corp_client_id
         , sector
         , date_trunc('hour', created)                                                        AS creation_dt
         , 'first_reply__backlog_good_sla'::VARCHAR                                           AS metric_name
         , SUM(CASE WHEN first_reply IS NULL AND backlog_good_sla THEN 1 ELSE 0 END)::NUMERIC AS metric_value
         , count(*)::BIGINT                                                                   AS sub_metric_value
    FROM first_reply
    WHERE first_reply != -1
       OR first_reply is null
    GROUP BY country, city, corp_client_id, sector, date_trunc('hour', created)
    UNION ALL
    SELECT country
         , city
         , corp_client_id
         , sector
         , date_trunc('hour', created)                                                            AS creation_dt
         , 'first_reply__backlog_bad_sla'::VARCHAR                                                AS metric_name
         , SUM(CASE WHEN first_reply IS NULL AND NOT backlog_good_sla THEN 1 ELSE 0 END)::NUMERIC AS metric_value
         , count(*)::BIGINT                                                                       AS sub_metric_value
    FROM first_reply
    WHERE first_reply != -1
       OR first_reply is null
    GROUP BY country, city, corp_client_id, sector, date_trunc('hour', created)
) DISTRIBUTED BY (country, city, corp_client_id, sector, creation_dt);
ANALYZE first_reply__total;


-- $sector_sla
CREATE TEMPORARY TABLE sector_sla
    ON COMMIT DROP
AS (
    SELECT created
         , creation_dttm
         , CASE
               WHEN meta_action IN
                    ('create', 'appear', 'new_message', 'reopen', 'reopen_after_dismiss', 'answer_on_proactive')
                   THEN date_trunc('hour', LEAD(created) OVER w)
        END AS close_dttm
         , sla_good
         , ticket_id
         , country
         , city
         , corp_client_id
         , sector
         , meta_action
         , added_tags
         , CASE
               WHEN meta_action IN
                    ('create', 'appear', 'new_message', 'reopen', 'reopen_after_dismiss', 'answer_on_proactive') AND
                    (LEAD(meta_action) OVER w IS NULL
                        OR LEAD(meta_action) OVER w = 'close_backlog')
                   THEN NULL
               WHEN meta_action IN
                    ('create', 'appear', 'new_message', 'reopen', 'reopen_after_dismiss', 'answer_on_proactive')
                   AND (LEAD(meta_action) OVER w IN
                        ('comment', 'communicate', 'call_success', 'close', 'temp_answer', 'comment_ml',
                         'communicate_ml',
                         'close_ml', 'defer_with_answer'))
                   THEN (LEAD(created) OVER w - created) / INTERVAL '1 second'
               ELSE -1
        END AS sector_sla
         , (meta_action IN ('create', 'appear', 'new_message', 'reopen', 'reopen_after_dismiss', 'answer_on_proactive')
        AND (LEAD(meta_action) OVER w IS NULL)
        AND EXTRACT(EPOCH FROM ((SELECT max_created FROM max_created) - created)) <= COALESCE(sla_good, 0)
        )   AS backlog_good_sla
    FROM chatterbox_history
    WHERE meta_action IN
          ('create', 'appear', 'new_message', 'reopen', 'reopen_after_dismiss', 'answer_on_proactive', 'comment',
           'communicate', 'call_success',
           'close', 'temp_answer', 'comment_ml', 'communicate_ml', 'close_ml', 'defer_with_answer', 'close_backlog',
           'tech_close', 'proactive_message', 'finish_archive')
        WINDOW w AS (PARTITION BY ticket_id ORDER BY event_number)
) DISTRIBUTED BY (ticket_id);
ANALYZE sector_sla;


-- $reaction_time
CREATE TEMPORARY TABLE reaction_time
    ON COMMIT DROP
AS (
    SELECT created
         , ticket_id
         , country
         , city
         , corp_client_id
         , sector
         , date_trunc('hour', LEAD(created) OVER w) AS close_dttm
         , CASE
               WHEN meta_action IN
                    ('create', 'appear', 'new_message', 'reopen', 'reopen_after_dismiss', 'reopen_after_defer',
                     'answer_on_proactive'
                        ) AND (LEAD(meta_action) OVER w IS NULL)
                   THEN NULL
               WHEN meta_action IN
                    ('create', 'appear', 'new_message', 'reopen', 'reopen_after_dismiss', 'reopen_after_defer',
                     'answer_on_proactive'
                        ) AND
                    (LEAD(meta_action) OVER w = 'close_backlog')
                   THEN NULL
               WHEN meta_action IN
                    ('create', 'appear', 'new_message', 'reopen', 'reopen_after_dismiss', 'reopen_after_defer',
                     'answer_on_proactive')
                   AND LEAD(meta_action) OVER w IN
                       ('comment', 'communicate', 'call_success', 'close', 'temp_answer', 'comment_ml',
                        'communicate_ml',
                        'close_ml', 'defer_with_answer', 'dismiss', 'defer', 'forward_task', 'call_fail',
                        'hidden_comment',
                        'export', 'move', 'tracker_copy', 'call_undefined', 'forward')
                   THEN (LEAD(created) OVER w - created) / INTERVAL '1 second'
               ELSE -1
        END                                         AS reaction_time -- Имя метрики: reaction_time
         , (meta_action IN
            ('create', 'appear', 'new_message', 'reopen', 'reopen_after_dismiss', 'reopen_after_defer',
             'answer_on_proactive') AND
            (LEAD(meta_action) OVER w IS NULL) AND
            ((SELECT max_created FROM max_created) - created) / INTERVAL '1 second' <= COALESCE(sla_good, 0)
        )                                           AS backlog_good_sla
    FROM chatterbox_history
    WHERE meta_action IN
          ('create', 'appear', 'new_message', 'reopen', 'reopen_after_dismiss', 'reopen_after_defer',
           'answer_on_proactive',
           'comment', 'communicate', 'call_success', 'close', 'temp_answer', 'comment_ml', 'communicate_ml', 'close_ml',
           'defer_with_answer', 'close_backlog', 'tech_close', 'proactive_message', 'finish_archive', 'dismiss',
           'defer',
           'forward_task', 'call_fail', 'hidden_comment', 'export', 'move', 'tracker_copy', 'call_undefined')
       OR (meta_action = 'forward' AND login != 'superuser')
        WINDOW w AS (PARTITION BY ticket_id ORDER BY event_number)
) DISTRIBUTED BY (ticket_id);
ANALYZE reaction_time;


-- $first_reply_no_ml
CREATE TEMPORARY TABLE first_reply_no_ml
    ON COMMIT DROP
AS (
    SELECT created
         , creation_dttm
         , CASE
               WHEN meta_action IN
                    ('create', 'new_message', 'reopen', 'reopen_after_dismiss', 'answer_on_proactive') AND
                    (LEAD(meta_action) OVER w IN ('tech_close', 'finish_archive'))
                   THEN date_trunc('hour', last_value(created) OVER w)
               WHEN
                           meta_action IN
                           ('create', 'new_message', 'reopen', 'reopen_after_dismiss', 'answer_on_proactive')
                       AND (LEAD(meta_action) OVER w IN
                            ('dismiss', 'comment', 'communicate', 'call_success', 'close', 'temp_answer',
                             'defer_with_answer',
                             'close_backlog'))
                   THEN date_trunc('hour', LEAD(created) OVER w)
               ELSE NULL
        END AS close_dttm
         , last_meta_action
         , sla_good
         , ticket_id
         , country
         , city
         , corp_client_id
         , CASE
               WHEN
                           meta_action IN
                           ('create', 'new_message', 'reopen', 'reopen_after_dismiss', 'answer_on_proactive')
                       AND LEAD(meta_action) OVER w IN
                           ('dismiss', 'comment', 'communicate', 'call_success', 'close', 'temp_answer',
                            'defer_with_answer', 'close_backlog')
                   THEN LEAD(sector) OVER w
               WHEN (LEAD(meta_action) OVER w IS NULL AND (lead_meta_action != 'appear' OR lead_meta_action IS NULL))
                   THEN sector
               WHEN LEAD(meta_action) OVER w IS NULL AND lead_meta_action = 'appear' THEN last_sector
        END AS sector
         , meta_action
         , added_tags
         , CASE
               WHEN meta_action IN
                    ('create', 'new_message', 'reopen', 'reopen_after_dismiss', 'answer_on_proactive') AND
                    (LEAD(meta_action) OVER w IS NULL
                        OR LEAD(meta_action) OVER w = 'close_backlog')
                   THEN NULL
               WHEN
                           meta_action IN
                           ('create', 'new_message', 'reopen', 'reopen_after_dismiss', 'answer_on_proactive')
                       AND LEAD(meta_action) OVER w IN
                           ('dismiss', 'comment', 'communicate', 'call_success', 'close', 'temp_answer',
                            'defer_with_answer')
                   THEN (LEAD(created) OVER w - created) / INTERVAL '1 second'
               ELSE -1
        END AS first_reply
         , (
                meta_action IN ('create', 'new_message', 'reopen', 'reopen_after_dismiss', 'answer_on_proactive') AND
                (LEAD(meta_action) OVER w IS NULL) AND
                ((SELECT max_created FROM max_created) - created) / INTERVAL '1 second' <= COALESCE(sla_good, 0)
        )   AS backlog_good_sla
    FROM (
             SELECT created
                  , event_number
                  , creation_dttm
                  , meta_action
                  , sla_good
                  , ticket_id
                  , added_tags
                  , sector
                  , country
                  , city
                  , corp_client_id
                  , LEAD(meta_action) OVER w       AS lead_meta_action
                  , last_value(meta_action) OVER w AS last_meta_action
                  , last_value(sector) OVER w      AS last_sector
             FROM chatterbox_history
             WHERE meta_action IN
                   ('dismiss', 'create', 'appear', 'new_message', 'reopen', 'reopen_after_dismiss',
                    'answer_on_proactive',
                    'comment', 'communicate', 'call_success', 'close', 'temp_answer', 'defer_with_answer',
                    'close_backlog', 'tech_close', 'proactive_message', 'finish_archive')
                 WINDOW w AS (PARTITION BY ticket_id
                     ORDER BY event_number
                     ROWS BETWEEN CURRENT ROW AND UNBOUNDED FOLLOWING
                     )
         ) t
    WHERE meta_action IN
          ('dismiss', 'create', 'new_message', 'reopen', 'reopen_after_dismiss', 'answer_on_proactive',
           'comment', 'communicate', 'call_success', 'close', 'temp_answer', 'defer_with_answer',
           'close_backlog', 'tech_close', 'proactive_message', 'finish_archive')
        WINDOW w AS (PARTITION BY ticket_id
            ORDER BY event_number)
) DISTRIBUTED BY (ticket_id);
ANALYZE first_reply_no_ml;


-- $first_reply_no_ml__total
CREATE TEMPORARY TABLE first_reply_no_ml__total
    ON COMMIT DROP
AS (
    SELECT country
         , city
         , corp_client_id
         , sector
         , date_trunc('hour', created)                                  AS creation_dt
         , 'first_reply_no_ml__5m'::VARCHAR                             AS metric_name
         , SUM(CASE WHEN first_reply <= 300 THEN 1 ELSE 0 END)::NUMERIC AS metric_value
         , count(*)::BIGINT                                             AS sub_metric_value
    FROM first_reply_no_ml
    WHERE first_reply != -1
    GROUP BY country, city, corp_client_id, sector, date_trunc('hour', created)
    UNION ALL
    SELECT country
         , city
         , corp_client_id
         , sector
         , date_trunc('hour', created)                                  AS creation_dt
         , 'first_reply_no_ml__5m_perc'::VARCHAR                        AS metric_name
         , SUM(CASE WHEN first_reply <= 300 THEN 1 ELSE 0 END)::NUMERIC AS metric_value
         , count(*)::BIGINT                                             AS sub_metric_value
    FROM first_reply_no_ml
    WHERE first_reply != -1
    GROUP BY country, city, corp_client_id, sector, date_trunc('hour', created)
    UNION ALL
    SELECT country
         , city
         , corp_client_id
         , sector
         , date_trunc('hour', created)                                                        AS creation_dt
         , 'first_reply_no_ml__10m'::VARCHAR                                                  AS metric_name
         , SUM(CASE WHEN first_reply > 300 and first_reply <= 600 THEN 1 ELSE 0 END)::NUMERIC AS metric_value
         , count(*)::BIGINT                                                                   AS sub_metric_value
    FROM first_reply_no_ml
    WHERE first_reply != -1
    GROUP BY country, city, corp_client_id, sector, date_trunc('hour', created)
    UNION ALL
    SELECT country
         , city
         , corp_client_id
         , sector
         , date_trunc('hour', created)                                  AS creation_dt
         , 'first_reply_no_ml__10m_perc'::VARCHAR                       AS metric_name
         , SUM(CASE WHEN first_reply <= 600 THEN 1 ELSE 0 END)::NUMERIC AS metric_value
         , count(*)::BIGINT                                             AS sub_metric_value
    FROM first_reply_no_ml
    WHERE first_reply != -1
    GROUP BY country, city, corp_client_id, sector, date_trunc('hour', created)
    UNION ALL
    SELECT country
         , city
         , corp_client_id
         , sector
         , date_trunc('hour', created)                                                         AS creation_dt
         , 'first_reply_no_ml__1h'::VARCHAR                                                    AS metric_name
         , SUM(CASE WHEN first_reply > 600 and first_reply <= 3600 THEN 1 ELSE 0 END)::NUMERIC AS metric_value
         , count(*)::BIGINT                                                                    AS sub_metric_value
    FROM first_reply_no_ml
    WHERE first_reply != -1
    GROUP BY country, city, corp_client_id, sector, date_trunc('hour', created)
    UNION ALL
    SELECT country
         , city
         , corp_client_id
         , sector
         , date_trunc('hour', created)                                   AS creation_dt
         , 'first_reply_no_ml__1h_perc'::VARCHAR                         AS metric_name
         , SUM(CASE WHEN first_reply <= 3600 THEN 1 ELSE 0 END)::NUMERIC AS metric_value
         , count(*)::BIGINT                                              AS sub_metric_value
    FROM first_reply_no_ml
    WHERE first_reply != -1
    GROUP BY country, city, corp_client_id, sector, date_trunc('hour', created)
    UNION ALL
    SELECT country
         , city
         , corp_client_id
         , sector
         , date_trunc('hour', created)                                                          AS creation_dt
         , 'first_reply_no_ml__2h'::VARCHAR                                                     AS metric_name
         , SUM(CASE WHEN first_reply > 3600 and first_reply <= 7200 THEN 1 ELSE 0 END)::NUMERIC AS metric_value
         , count(*)::BIGINT                                                                     AS sub_metric_value
    FROM first_reply_no_ml
    WHERE first_reply != -1
    GROUP BY country, city, corp_client_id, sector, date_trunc('hour', created)
    UNION ALL
    SELECT country
         , city
         , corp_client_id
         , sector
         , date_trunc('hour', created)                                   AS creation_dt
         , 'first_reply_no_ml__2h_perc'::VARCHAR                         AS metric_name
         , SUM(CASE WHEN first_reply <= 7200 THEN 1 ELSE 0 END)::NUMERIC AS metric_value
         , count(*)::BIGINT                                              AS sub_metric_value
    FROM first_reply_no_ml
    WHERE first_reply != -1
    GROUP BY country, city, corp_client_id, sector, date_trunc('hour', created)
    UNION ALL
    SELECT country
         , city
         , corp_client_id
         , sector
         , date_trunc('hour', created)                                  AS creation_dt
         , 'first_reply_no_ml__more_2h'::VARCHAR                        AS metric_name
         , SUM(CASE WHEN first_reply > 7200 THEN 1 ELSE 0 END)::NUMERIC AS metric_value
         , count(*)::BIGINT                                             AS sub_metric_value
    FROM first_reply_no_ml
    WHERE first_reply != -1
    GROUP BY country, city, corp_client_id, sector, date_trunc('hour', created)
    UNION ALL
    SELECT country
         , city
         , corp_client_id
         , sector
         , date_trunc('hour', created)                                                        AS creation_dt
         , 'first_reply_no_ml__backlog_good_sla'::VARCHAR                                     AS metric_name
         , SUM(CASE WHEN first_reply IS NULL AND backlog_good_sla THEN 1 ELSE 0 END)::NUMERIC AS metric_value
         , count(*)::BIGINT                                                                   AS sub_metric_value
    FROM first_reply_no_ml
    WHERE first_reply != -1
       OR first_reply IS NULL
    GROUP BY country, city, corp_client_id, sector, date_trunc('hour', created)
    UNION ALL
    SELECT country
         , city
         , corp_client_id
         , sector
         , date_trunc('hour', created)                                                            AS creation_dt
         , 'first_reply_no_ml__backlog_bad_sla'::VARCHAR                                          AS metric_name
         , SUM(CASE WHEN first_reply IS NULL AND NOT backlog_good_sla THEN 1 ELSE 0 END)::NUMERIC AS metric_value
         , count(*)::BIGINT                                                                       AS sub_metric_value
    FROM first_reply_no_ml
    WHERE first_reply != -1
       OR first_reply IS NULL
    GROUP BY country, city, corp_client_id, sector, date_trunc('hour', created)
) DISTRIBUTED BY (country, city, corp_client_id, sector, creation_dt);
ANALYZE first_reply_no_ml__total;


-- $first_reply_no_communicate
CREATE TEMPORARY TABLE first_reply_no_communicate
    ON COMMIT DROP
AS (
    SELECT created
         , creation_dttm
         , CASE
               WHEN meta_action IN
                    ('create', 'new_message', 'reopen', 'reopen_after_dismiss', 'answer_on_proactive') AND
                    (LEAD(meta_action) OVER w IN ('tech_close', 'finish_archive'))
                   THEN date_trunc('hour', last_value(created) OVER w)
               WHEN
                           meta_action IN
                           ('create', 'new_message', 'reopen', 'reopen_after_dismiss', 'answer_on_proactive')
                       AND (LEAD(meta_action) OVER w IN
                            ('dismiss', 'comment', 'call_success', 'close', 'comment_ml', 'close_ml',
                             'defer_with_answer', 'close_backlog'))
                   THEN date_trunc('hour', LEAD(created) OVER w)
               ELSE NULL
        END AS close_dttm
         , sla_good
         , ticket_id
         , country
         , city
         , corp_client_id
         , CASE
               WHEN
                           meta_action IN
                           ('create', 'new_message', 'reopen', 'reopen_after_dismiss', 'answer_on_proactive')
                       AND LEAD(meta_action) OVER w IN
                           ('dismiss', 'comment', 'call_success', 'close', 'comment_ml', 'close_ml',
                            'defer_with_answer', 'close_backlog')
                   THEN LEAD(sector) OVER w
               WHEN (LEAD(meta_action) OVER w IS NULL AND (lead_meta_action != 'appear' OR lead_meta_action IS NULL))
                   THEN sector
               WHEN LEAD(meta_action) OVER w IS NULL AND lead_meta_action = 'appear' THEN last_sector
        END AS sector
         , meta_action
         , added_tags
         , CASE
               WHEN meta_action IN
                    ('create', 'new_message', 'reopen', 'reopen_after_dismiss', 'answer_on_proactive') AND
                    (LEAD(meta_action) OVER w IS NULL
                        OR LEAD(meta_action) OVER w = 'close_backlog')
                   THEN NULL
               WHEN
                           meta_action IN
                           ('create', 'new_message', 'reopen', 'reopen_after_dismiss', 'answer_on_proactive')
                       AND LEAD(meta_action) OVER w IN
                           ('dismiss', 'comment', 'call_success', 'close', 'comment_ml', 'close_ml',
                            'defer_with_answer')
                   THEN (LEAD(created) OVER w - created) / INTERVAL '1 second'
               ELSE -1
        END AS first_reply
         , (meta_action IN ('create', 'new_message', 'reopen', 'reopen_after_dismiss', 'answer_on_proactive') AND
            (LEAD(meta_action) OVER w IS NULL) AND
            ((SELECT max_created FROM max_created) - created) / INTERVAL '1 second' <= COALESCE(sla_good, 0))
            AS backlog_good_sla
    FROM (
             SELECT created
                  , event_number
                  , creation_dttm
                  , meta_action
                  , sla_good
                  , ticket_id
                  , country
                  , city
                  , corp_client_id
                  , added_tags
                  , sector
                  , LEAD(meta_action) OVER w  AS lead_meta_action
                  , last_value(sector) OVER w AS last_sector
             FROM chatterbox_history
             WHERE meta_action IN
                   ('dismiss', 'create', 'appear', 'new_message', 'reopen', 'reopen_after_dismiss',
                    'answer_on_proactive',
                    'comment', 'call_success', 'close', 'comment_ml', 'close_ml', 'defer_with_answer',
                    'close_backlog', 'tech_close', 'proactive_message', 'finish_archive')
                 WINDOW w AS (PARTITION BY ticket_id
                     ORDER BY event_number
                     ROWS BETWEEN CURRENT ROW AND UNBOUNDED FOLLOWING
                     )
         ) t
    WHERE meta_action IN
          ('dismiss', 'create', 'new_message', 'reopen', 'reopen_after_dismiss', 'answer_on_proactive', 'comment',
           'call_success', 'close', 'comment_ml',
           'close_ml', 'defer_with_answer', 'close_backlog', 'tech_close', 'proactive_message', 'finish_archive')
        WINDOW w AS (PARTITION BY ticket_id
            ORDER BY event_number)
) DISTRIBUTED BY (ticket_id);
ANALYZE first_reply_no_communicate;


-- $first_reply_no_communicate__total
CREATE TEMPORARY TABLE first_reply_no_communicate__total
    ON COMMIT DROP
AS (
    SELECT country
         , city
         , corp_client_id
         , sector
         , date_trunc('hour', created)                                  AS creation_dt
         , 'first_reply_no_communicate__5m'::VARCHAR                    AS metric_name
         , SUM(CASE WHEN first_reply <= 300 THEN 1 ELSE 0 END)::NUMERIC AS metric_value
         , count(*)::BIGINT                                             AS sub_metric_value
    FROM first_reply_no_communicate
    WHERE first_reply != -1
    GROUP BY country, city, corp_client_id, sector, date_trunc('hour', created)
    UNION ALL
    SELECT country
         , city
         , corp_client_id
         , sector
         , date_trunc('hour', created)                                  AS creation_dt
         , 'first_reply_no_communicate__5m_perc'::VARCHAR               AS metric_name
         , SUM(CASE WHEN first_reply <= 300 THEN 1 ELSE 0 END)::NUMERIC AS metric_value
         , count(*)::BIGINT                                             AS sub_metric_value
    FROM first_reply_no_communicate
    WHERE first_reply != -1
    GROUP BY country, city, corp_client_id, sector, date_trunc('hour', created)
    UNION ALL
    SELECT country
         , city
         , corp_client_id
         , sector
         , date_trunc('hour', created)                                                        AS creation_dt
         , 'first_reply_no_communicate__10m'::VARCHAR                                         AS metric_name
         , SUM(CASE WHEN first_reply > 300 and first_reply <= 600 THEN 1 ELSE 0 END)::NUMERIC AS metric_value
         , count(*)::BIGINT                                                                   AS sub_metric_value
    FROM first_reply_no_communicate
    WHERE first_reply != -1
    GROUP BY country, city, corp_client_id, sector, date_trunc('hour', created)
    UNION ALL
    SELECT country
         , city
         , corp_client_id
         , sector
         , date_trunc('hour', created)                                  AS creation_dt
         , 'first_reply_no_communicate__10m_perc'::VARCHAR              AS metric_name
         , SUM(CASE WHEN first_reply <= 600 THEN 1 ELSE 0 END)::NUMERIC AS metric_value
         , count(*)::BIGINT                                             AS sub_metric_value
    FROM first_reply_no_communicate
    WHERE first_reply != -1
    GROUP BY country, city, corp_client_id, sector, date_trunc('hour', created)
    UNION ALL
    SELECT country
         , city
         , corp_client_id
         , sector
         , date_trunc('hour', created)                                                         AS creation_dt
         , 'first_reply_no_communicate__1h'::VARCHAR                                           AS metric_name
         , SUM(CASE WHEN first_reply > 600 and first_reply <= 3600 THEN 1 ELSE 0 END)::NUMERIC AS metric_value
         , count(*)::BIGINT                                                                    AS sub_metric_value
    FROM first_reply_no_communicate
    WHERE first_reply != -1
    GROUP BY country, city, corp_client_id, sector, date_trunc('hour', created)
    UNION ALL
    SELECT country
         , city
         , corp_client_id
         , sector
         , date_trunc('hour', created)                                   AS creation_dt
         , 'first_reply_no_communicate__1h_perc'::VARCHAR                AS metric_name
         , SUM(CASE WHEN first_reply <= 3600 THEN 1 ELSE 0 END)::NUMERIC AS metric_value
         , count(*)::BIGINT                                              AS sub_metric_value
    FROM first_reply_no_communicate
    WHERE first_reply != -1
    GROUP BY country, city, corp_client_id, sector, date_trunc('hour', created)
    UNION ALL
    SELECT country
         , city
         , corp_client_id
         , sector
         , date_trunc('hour', created)                                                          AS creation_dt
         , 'first_reply_no_communicate__2h'::VARCHAR                                            AS metric_name
         , SUM(CASE WHEN first_reply > 3600 and first_reply <= 7200 THEN 1 ELSE 0 END)::NUMERIC AS metric_value
         , count(*)::BIGINT                                                                     AS sub_metric_value
    FROM first_reply_no_communicate
    WHERE first_reply != -1
    GROUP BY country, city, corp_client_id, sector, date_trunc('hour', created)
    UNION ALL
    SELECT country
         , city
         , corp_client_id
         , sector
         , date_trunc('hour', created)                                   AS creation_dt
         , 'first_reply_no_communicate__2h_perc'::VARCHAR                AS metric_name
         , SUM(CASE WHEN first_reply <= 7200 THEN 1 ELSE 0 END)::NUMERIC AS metric_value
         , count(*)::BIGINT                                              AS sub_metric_value
    FROM first_reply_no_communicate
    WHERE first_reply != -1
    GROUP BY country, city, corp_client_id, sector, date_trunc('hour', created)
    UNION ALL
    SELECT country
         , city
         , corp_client_id
         , sector
         , date_trunc('hour', created)                                  AS creation_dt
         , 'first_reply_no_communicate__more_2h'::VARCHAR               AS metric_name
         , SUM(CASE WHEN first_reply > 7200 THEN 1 ELSE 0 END)::NUMERIC AS metric_value
         , count(*)::BIGINT                                             AS sub_metric_value
    FROM first_reply_no_communicate
    WHERE first_reply != -1
    GROUP BY country, city, corp_client_id, sector, date_trunc('hour', created)
    UNION ALL
    SELECT country
         , city
         , corp_client_id
         , sector
         , date_trunc('hour', created)                                                        AS creation_dt
         , 'first_reply_no_communicate__backlog_good_sla'::VARCHAR                            AS metric_name
         , SUM(CASE WHEN first_reply IS NULL AND backlog_good_sla THEN 1 ELSE 0 END)::NUMERIC AS metric_value
         , count(*)::BIGINT                                                                   AS sub_metric_value
    FROM first_reply_no_communicate
    WHERE first_reply != -1
       OR first_reply is null
    GROUP BY country, city, corp_client_id, sector, date_trunc('hour', created)
    UNION ALL
    SELECT country
         , city
         , corp_client_id
         , sector
         , date_trunc('hour', created)                                                            AS creation_dt
         , 'first_reply_no_communicate__backlog_bad_sla'::VARCHAR                                 AS metric_name
         , SUM(CASE WHEN first_reply IS NULL AND NOT backlog_good_sla THEN 1 ELSE 0 END)::NUMERIC AS metric_value
         , count(*)::BIGINT                                                                       AS sub_metric_value
    FROM first_reply_no_communicate
    WHERE first_reply != -1
       OR first_reply is null
    GROUP BY country, city, corp_client_id, sector, date_trunc('hour', created)
) DISTRIBUTED BY (country, city, corp_client_id, sector, creation_dt);
ANALYZE first_reply_no_communicate__total;


-- $extremely_first_reply
CREATE TEMPORARY TABLE extremely_first_reply
    ON COMMIT DROP
AS (
    SELECT created
         , creation_dttm
         , CASE
               WHEN
                           meta_action IN ('create', 'answer_on_proactive')
                       AND LEAD(meta_action) OVER w IN
                           ('comment', 'communicate', 'call_success', 'close', 'temp_answer', 'comment_ml',
                            'communicate_ml',
                            'close_ml', 'defer_with_answer', 'dismiss')
                   THEN date_trunc('hour', LEAD(created) OVER w)
        END AS close_dttm
         , CASE
               WHEN
                           meta_action IN ('create', 'answer_on_proactive')
                       AND LEAD(meta_action) OVER w IN
                           ('comment', 'communicate', 'call_success', 'close', 'temp_answer', 'comment_ml',
                            'communicate_ml',
                            'close_ml', 'defer_with_answer', 'dismiss')
                   THEN LEAD(sla_good) OVER w
               WHEN (LEAD(meta_action) OVER w IS NULL AND (lead_meta_action != 'appear' OR lead_meta_action IS NULL))
                   THEN sla_good
               WHEN LEAD(meta_action) OVER w IS NULL AND lead_meta_action = 'appear' THEN last_sla_good
        END AS sla_good
         , ticket_id
         , country
         , city
         , corp_client_id
         , CASE
               WHEN
                           meta_action IN ('create', 'answer_on_proactive')
                       AND LEAD(meta_action) OVER w IN
                           ('comment', 'communicate', 'call_success', 'close', 'temp_answer', 'comment_ml',
                            'communicate_ml',
                            'close_ml', 'defer_with_answer', 'dismiss')
                   THEN LEAD(sector) OVER w
               WHEN (LEAD(meta_action) OVER w IS NULL AND (lead_meta_action != 'appear' OR lead_meta_action IS NULL))
                   THEN sector
               WHEN LEAD(meta_action) OVER w IS NULL AND lead_meta_action = 'appear' THEN last_sector
        END AS sector
         , meta_action
         , added_tags
         , CASE
               WHEN meta_action IN ('create', 'answer_on_proactive')
                   AND
                    (LEAD(meta_action) OVER w IS NULL
                        OR LEAD(meta_action) OVER w = 'close_backlog')
                   THEN NULL
               WHEN
                           meta_action IN ('create', 'answer_on_proactive')
                       AND LEAD(meta_action) OVER w IN
                           ('comment', 'communicate', 'call_success', 'close', 'temp_answer', 'comment_ml',
                            'communicate_ml',
                            'close_ml', 'defer_with_answer', 'dismiss')
                   THEN (LEAD(created) OVER w - created) / INTERVAL '1 second'
               ELSE -1
        END AS extremely_first_reply -- (Здесь extremely_first_reply, НЕ first_reply)
         , meta_action IN ('create', 'answer_on_proactive') AND
           (LEAD(meta_action) OVER w IS NULL AND (lead_meta_action IS NULL OR lead_meta_action != 'appear'))
        AND (((SELECT max_created FROM max_created) - created) / INTERVAL '1 second' <= COALESCE(sla_good, 0))
            AS backlog_good_sla

    FROM (
             SELECT created
                  , event_number
                  , creation_dttm
                  , meta_action
                  , sla_good
                  , ticket_id
                  , country
                  , city
                  , corp_client_id
                  , added_tags
                  , sector
                  , LEAD(meta_action) OVER w    AS lead_meta_action
                  , LAST_VALUE(sector) OVER w   AS last_sector
                  , LAST_VALUE(sla_good) OVER w AS last_sla_good
             FROM chatterbox_history
             WHERE meta_action IN
                   ('create', 'appear', 'answer_on_proactive', 'comment', 'communicate', 'call_success', 'close',
                    'temp_answer',
                    'comment_ml', 'communicate_ml', 'close_ml', 'defer_with_answer', 'close_backlog', 'tech_close',
                    'proactive_message', 'finish_archive', 'dismiss')
                 WINDOW w AS (
                     PARTITION BY ticket_id
                     ORDER BY event_number
                     ROWS BETWEEN CURRENT ROW AND UNBOUNDED FOLLOWING
                     )
         ) t
    WHERE meta_action IN
          ('create', 'answer_on_proactive', 'comment', 'communicate', 'call_success', 'close', 'temp_answer',
           'comment_ml', 'communicate_ml', 'close_ml', 'defer_with_answer', 'close_backlog', 'tech_close',
           'proactive_message', 'finish_archive', 'dismiss')
        WINDOW w AS (PARTITION BY ticket_id ORDER BY event_number)
) DISTRIBUTED BY (ticket_id);
ANALYZE extremely_first_reply;


-- $extremely_first_reply__total
CREATE TEMPORARY TABLE extremely_first_reply__total
    ON COMMIT DROP
AS (
    SELECT country
         , city
         , corp_client_id
         , sector
         , date_trunc('hour', created)                                            AS creation_dt
         , 'extremely_first_reply__5m'::VARCHAR                                   AS metric_name
         , SUM(CASE WHEN extremely_first_reply <= 300 THEN 1 ELSE 0 END)::NUMERIC AS metric_value
         , count(*)::BIGINT                                                       AS sub_metric_value
    FROM extremely_first_reply
    WHERE extremely_first_reply != -1
    GROUP BY country, city, corp_client_id, sector, date_trunc('hour', created)
    UNION ALL
    SELECT country
         , city
         , corp_client_id
         , sector
         , date_trunc('hour', created)                                            AS creation_dt
         , 'extremely_first_reply__5m_perc'::VARCHAR                              AS metric_name
         , SUM(CASE WHEN extremely_first_reply <= 300 THEN 1 ELSE 0 END)::NUMERIC AS metric_value
         , count(*)::BIGINT                                                       AS sub_metric_value
    FROM extremely_first_reply
    WHERE extremely_first_reply != -1
    GROUP BY country, city, corp_client_id, sector, date_trunc('hour', created)
    UNION ALL
    SELECT country
         , city
         , corp_client_id
         , sector
         , date_trunc('hour', created)           AS creation_dt
         , 'extremely_first_reply__10m'::VARCHAR AS metric_name
         , SUM(CASE
                   WHEN extremely_first_reply > 300 and extremely_first_reply <= 600 THEN 1
                   ELSE 0 END)::NUMERIC          AS metric_value
         , count(*)::BIGINT                      AS sub_metric_value
    FROM extremely_first_reply
    WHERE extremely_first_reply != -1
    GROUP BY country, city, corp_client_id, sector, date_trunc('hour', created)
    UNION ALL
    SELECT country
         , city
         , corp_client_id
         , sector
         , date_trunc('hour', created)                                            AS creation_dt
         , 'extremely_first_reply__10m_perc'::VARCHAR                             AS metric_name
         , SUM(CASE WHEN extremely_first_reply <= 600 THEN 1 ELSE 0 END)::NUMERIC AS metric_value
         , count(*)::BIGINT                                                       AS sub_metric_value
    FROM extremely_first_reply
    WHERE extremely_first_reply != -1
    GROUP BY country, city, corp_client_id, sector, date_trunc('hour', created)
    UNION ALL
    SELECT country
         , city
         , corp_client_id
         , sector
         , date_trunc('hour', created)          AS creation_dt
         , 'extremely_first_reply__1h'::VARCHAR AS metric_name
         , SUM(CASE
                   WHEN extremely_first_reply > 600 and extremely_first_reply <= 3600 THEN 1
                   ELSE 0 END)::NUMERIC         AS metric_value
         , count(*)::BIGINT                     AS sub_metric_value
    FROM extremely_first_reply
    WHERE extremely_first_reply != -1
    GROUP BY country, city, corp_client_id, sector, date_trunc('hour', created)
    UNION ALL
    SELECT country
         , city
         , corp_client_id
         , sector
         , date_trunc('hour', created)                                             AS creation_dt
         , 'extremely_first_reply__1h_perc'::VARCHAR                               AS metric_name
         , SUM(CASE WHEN extremely_first_reply <= 3600 THEN 1 ELSE 0 END)::NUMERIC AS metric_value
         , count(*)::BIGINT                                                        AS sub_metric_value
    FROM extremely_first_reply
    WHERE extremely_first_reply != -1
    GROUP BY country, city, corp_client_id, sector, date_trunc('hour', created)
    UNION ALL
    SELECT country
         , city
         , corp_client_id
         , sector
         , date_trunc('hour', created)          AS creation_dt
         , 'extremely_first_reply__2h'::VARCHAR AS metric_name
         , SUM(CASE
                   WHEN extremely_first_reply > 3600 and extremely_first_reply <= 7200 THEN 1
                   ELSE 0 END)::NUMERIC         AS metric_value
         , count(*)::BIGINT                     AS sub_metric_value
    FROM extremely_first_reply
    WHERE extremely_first_reply != -1
    GROUP BY country, city, corp_client_id, sector, date_trunc('hour', created)
    UNION ALL
    SELECT country
         , city
         , corp_client_id
         , sector
         , date_trunc('hour', created)                                             AS creation_dt
         , 'extremely_first_reply__2h_perc'::VARCHAR                               AS metric_name
         , SUM(CASE WHEN extremely_first_reply <= 7200 THEN 1 ELSE 0 END)::NUMERIC AS metric_value
         , count(*)::BIGINT                                                        AS sub_metric_value
    FROM extremely_first_reply
    WHERE extremely_first_reply != -1
    GROUP BY country, city, corp_client_id, sector, date_trunc('hour', created)
    UNION ALL
    SELECT country
         , city
         , corp_client_id
         , sector
         , date_trunc('hour', created)                                            AS creation_dt
         , 'extremely_first_reply__more_2h'::VARCHAR                              AS metric_name
         , SUM(CASE WHEN extremely_first_reply > 7200 THEN 1 ELSE 0 END)::NUMERIC AS metric_value
         , count(*)::BIGINT                                                       AS sub_metric_value
    FROM extremely_first_reply
    WHERE extremely_first_reply != -1
    GROUP BY country, city, corp_client_id, sector, date_trunc('hour', created)
    UNION ALL
    SELECT country
         , city
         , corp_client_id
         , sector
         , date_trunc('hour', created)                                                                  AS creation_dt
         , 'extremely_first_reply__backlog_good_sla'::VARCHAR                                           AS metric_name
         , SUM(CASE WHEN extremely_first_reply IS NULL AND backlog_good_sla THEN 1 ELSE 0 END)::NUMERIC AS metric_value
         , count(*)::BIGINT                                                                             AS sub_metric_value
    FROM extremely_first_reply
    WHERE extremely_first_reply != -1
       OR extremely_first_reply IS NULL
    GROUP BY country, city, corp_client_id, sector, date_trunc('hour', created)
    UNION ALL
    SELECT country
         , city
         , corp_client_id
         , sector
         , date_trunc('hour', created)                   AS creation_dt
         , 'first_reply_no_ml__backlog_bad_sla'::VARCHAR AS metric_name
         , SUM(CASE
                   WHEN extremely_first_reply IS NULL AND NOT backlog_good_sla THEN 1
                   ELSE 0 END)::NUMERIC                  AS metric_value
         , count(*)::BIGINT                              AS sub_metric_value
    FROM extremely_first_reply
    WHERE extremely_first_reply != -1
       OR extremely_first_reply IS NULL
    GROUP BY country, city, corp_client_id, sector, date_trunc('hour', created)
) DISTRIBUTED BY (country, city, corp_client_id, sector, creation_dt);
ANALYZE extremely_first_reply__total;


-- $csat_per_ticket
CREATE TEMPORARY TABLE csat_per_ticket
    ON COMMIT DROP
AS (
    SELECT country
         , city
         , corp_client_id
         , sector
         , ticket_id
         , coalesce(csat, lead_csat) AS csat
         , first(sla_good)           AS sla_good
         , max(created)              AS created
    FROM (
             SELECT a.sector
                  , a.created
                  , a.ticket_id
                  , a.country
                  , a.city
                  , a.corp_client_id
                  , a.meta_action
                  , a.csat
                  , first_value(a.csat)
                    OVER (PARTITION BY a.ticket_id, value_partition, sector ORDER BY a.event_number DESC) AS lead_csat
                  , first_value(fr.metric_value)
                    OVER (PARTITION BY a.ticket_id, value_partition ORDER BY a.event_number) < a.sla_good AS sla_good
             FROM (
                      SELECT country
                           , city
                           , corp_client_id
                           , sector
                           , created
                           , ticket_id
                           , meta_action
                           , sla_good
                           , event_number
                           , csat
                           , sum(CASE WHEN csat IS NULL THEN 0 ELSE 1 END)
                             OVER (PARTITION BY ticket_id ORDER BY event_number DESC) AS value_partition
                      FROM chatterbox_history
                  ) a
                      LEFT JOIN taxi_cdm_contactcenter.dm_first_reply AS fr
                                ON a.ticket_id = fr.ticket_id
                                    AND a.meta_action = fr.meta_action_name
                                    AND a.created = fr.utc_event_dttm
                                    AND a.sector = fr.sector_code
         ) tt
    WHERE meta_action IN ('call_success', 'close', 'close_ml', 'proactive_message')
      AND coalesce(csat, lead_csat) IS NOT NULL
    GROUP BY 1, 2, 3, 4, 5, 6
) DISTRIBUTED BY (country, city, corp_client_id, sector, ticket_id, created);
ANALYZE csat_per_ticket;


-- $chatterbox_csat_per_ticket + $chatterbox_csat + $csat
CREATE TEMPORARY TABLE csat
    ON COMMIT DROP
AS (
    SELECT country
         , city
         , corp_client_id
         , sector                    AS sector
         , creation_dt               AS creation_dt
         , 'csat'::VARCHAR           AS metric_name
         , AVG(csat)::NUMERIC        AS metric_value
         , SUM(ticket_count)::BIGINT AS sub_metric_value
    FROM (
             SELECT country
                  , city
                  , corp_client_id
                  , sector
                  , date_trunc('hour', created) AS creation_dt
                  , ticket_id
                  , MAX(csat)                   AS csat
                  , COUNT(ticket_id)::BIGINT    AS ticket_count
             FROM csat_per_ticket
             WHERE csat IS NOT NULL
             GROUP BY 1, 2, 3, 4, 5, 6
         ) S
    GROUP BY 1, 2, 3, 4, 5, 6
) DISTRIBUTED BY (country, city, corp_client_id, sector, creation_dt);
ANALYZE csat;


-- $overall + $inbox_tickets
CREATE TEMPORARY TABLE inbox_tickets
    ON COMMIT DROP
AS (
    SELECT country
         , city
         , corp_client_id
         , sector
         , date_trunc('hour', created) AS creation_dt
         , 'inbox_tickets'::VARCHAR    AS metric_name
         , SUM(inbox_tickets)::NUMERIC AS metric_value
         , NULL::BIGINT                AS sub_metric_value
    FROM (SELECT created
               , country
               , city
               , corp_client_id
               , sector
               , CASE
                     WHEN (meta_action IN ('create', 'new_message', 'appear', 'reopen', 'reopen_after_dismiss',
                                           'answer_on_proactive') AND
                           (NOT (LEAD(meta_action) OVER w = 'forward' AND LEAD(login) OVER w = 'superuser') AND
                            LEAD(meta_action) OVER w NOT IN ('proactive_support', 'proactive_message')
                               OR LEAD(meta_action) OVER w IS NULL)
                         )
                         THEN 1
                     ELSE 0
            END AS inbox_tickets
          FROM chatterbox_history
          WHERE meta_action IN
                ('create', 'new_message', 'appear', 'reopen', 'reopen_after_dismiss', 'forward', 'proactive_support',
                 'answer_on_proactive',
                 'proactive_message')
              WINDOW w AS (PARTITION BY ticket_id ORDER BY event_number)
         ) t
    GROUP BY country, city, corp_client_id,
             sector, date_trunc('hour', created)
) DISTRIBUTED BY (country, city, corp_client_id, sector, creation_dt);
ANALYZE inbox_tickets;


-- $tickets
CREATE TEMPORARY TABLE tickets
    ON COMMIT DROP
AS (
    SELECT country
         , city
         , corp_client_id
         , sector::VARCHAR             AS sector
         , date_trunc('hour', created) AS creation_dt
         , 'tickets'::VARCHAR          AS metric_name
         , COUNT(ticket_id)::NUMERIC   AS metric_value
         , NULL::BIGINT                AS sub_metric_value
    FROM (
             SELECT ticket_id
                  , MIN(country)        AS country
                  , MIN(city)           AS city
                  , MIN(corp_client_id) AS corp_client_id
                  , MIN(created)        AS created
                  , MIN(last_sector)    AS sector
             FROM (
                      SELECT country
                           , city
                           , corp_client_id
                           , sector
                           , ticket_id
                           , created
                           , last_value(sector) OVER w AS last_sector
                      FROM chatterbox_history
                          WINDOW w AS (PARTITION BY ticket_id
                              ORDER BY event_number
                              ROWS BETWEEN CURRENT ROW AND UNBOUNDED FOLLOWING
                              )
                  ) t
             GROUP BY ticket_id
         ) tt
    GROUP BY country, city, corp_client_id, sector, date_trunc('hour', created)
) DISTRIBUTED BY (country, city, corp_client_id, sector, creation_dt);
ANALYZE tickets;


-- $orders
CREATE TEMPORARY TABLE orders
    ON COMMIT DROP
AS (
    SELECT country_name_ru                            AS country
         , agglomeration_name_ru                      AS city
         , corp_client_id
         , date_trunc('hour', utc_order_created_dttm) AS creation_dt
         , 'orders'::VARCHAR                          AS metric_name
         , COUNT(*)::NUMERIC                          AS metric_value
         , NULL::BIGINT                               AS sub_metric_value
    FROM taxi_cdm_marketplace.fct_order
    WHERE tariff_class_code in ('cargo', 'express', 'courier', 'cargocorp', 'ubernight')
    and utc_order_created_dttm between (SELECT start_date FROM start_date) and (SELECT end_date FROM end_date)
    GROUP BY country_name_ru, agglomeration_name_ru, corp_client_id, date_trunc('hour', utc_order_created_dttm)
) DISTRIBUTED BY (country, city, corp_client_id, creation_dt);
ANALYZE orders;


-- $unique_contact_rate_perc
CREATE TEMPORARY TABLE unique_contact_rate_perc
    ON COMMIT DROP
AS (
    SELECT country,
           city,
           corp_client_id,
           sector,
           creation_dt,
           'unique_contact_rate_perc' AS metric_name,
           metric_value,
           NULL::BIGINT               AS sub_metric_value
    FROM tickets
    UNION ALL
    SELECT country,
           city,
           corp_client_id,
           'orders'                   AS sector,
           creation_dt,
           'unique_contact_rate_perc' AS metric_name,
           NULL::BIGINT               AS metric_value,
           metric_value               AS sub_metric_value
    FROM orders
) DISTRIBUTED BY (country, city, corp_client_id, sector, creation_dt);
ANALYZE unique_contact_rate_perc;


-- $contact_rate_perc
CREATE TEMPORARY TABLE contact_rate_perc
    ON COMMIT DROP
AS (
    SELECT country,
           city,
           corp_client_id,
           sector              AS sector,
           creation_dt,
           'contact_rate_perc' AS metric_name,
           metric_value,
           NULL::BIGINT        AS sub_metric_value
    FROM inbox_tickets
    UNION ALL
    SELECT country,
           city,
           corp_client_id,
           'orders'            AS sector,
           creation_dt,
           'contact_rate_perc' AS metric_name,
           NULL::BIGINT        AS metric_value,
           metric_value        AS sub_metric_value
    FROM orders
) DISTRIBUTED BY (country, city, corp_client_id, sector, creation_dt);
ANALYZE contact_rate_perc;


-- $reopens
CREATE TEMPORARY TABLE reopens
    ON COMMIT DROP
AS (
    SELECT country
         , city
         , corp_client_id
         , sector
         , date_trunc('hour', created)                                      AS creation_dt
         , 'reopens'::VARCHAR                                               AS metric_name
         , sum(CASE WHEN first_event = 'reopen' THEN 1 ELSE 0 END)::NUMERIC AS metric_value
         , NULL::BIGINT                                                     AS sub_metric_value
    FROM fct_ticket_session
    GROUP BY country, city, corp_client_id, sector, date_trunc('hour', created)
) DISTRIBUTED BY (country, city, corp_client_id, sector, creation_dt);
ANALYZE reopens;


-- $reopens_without_waiting
CREATE TEMPORARY TABLE reopens_without_waiting
    ON COMMIT DROP
AS (
    SELECT country
         , city
         , corp_client_id
         , sector
         , date_trunc('hour', created)           AS creation_dt
         , 'reopens_without_waiting'::VARCHAR    AS metric_name
         , SUM(reopens_without_waiting)::NUMERIC AS metric_value
         , NULL::BIGINT                          AS sub_metric_value
    FROM (SELECT created
               , country
               , city
               , corp_client_id
               , sector
               , CASE
                     WHEN meta_action IN
                          ('comment', 'communicate', 'call_success', 'close', 'temp_answer', 'comment_ml',
                           'communicate_ml', 'close_ml', 'defer_with_answer', 'tech_close')
                         AND lag(meta_action) over w != 'comment'
                         AND lead(meta_action) over w IN ('reopen', 'reopen_after_dismiss')
                         THEN 1
                     ELSE 0
            END AS reopens_without_waiting
          FROM chatterbox_history
          WHERE meta_action IN
                ('comment', 'communicate', 'call_success', 'close', 'temp_answer', 'comment_ml', 'communicate_ml',
                 'close_ml', 'defer_with_answer', 'tech_close', 'reopen', 'reopen_after_dismiss')
              WINDOW w AS (PARTITION BY ticket_id ORDER BY event_number)
         ) t
    GROUP BY country, city, corp_client_id, sector, date_trunc('hour', created)
) DISTRIBUTED BY ( country, city, corp_client_id, sector, creation_dt);
ANALYZE reopens_without_waiting;


-- $reopens_to_sessions_perc
CREATE TEMPORARY TABLE reopens_to_sessions_perc
    ON COMMIT DROP
AS (
    SELECT country
         , city
         , corp_client_id
         , sector
         , date_trunc('hour', created)                                      AS creation_dt
         , 'reopens_to_sessions_perc'::VARCHAR                              AS metric_name
         , sum(CASE WHEN first_event = 'reopen' THEN 1 ELSE 0 END)::NUMERIC AS metric_value
         , count(*)::BIGINT                                                 AS sub_metric_value
    FROM fct_ticket_session
    GROUP BY country, city, corp_client_id, sector, date_trunc('hour', created)
) DISTRIBUTED BY (country, city, corp_client_id, sector, creation_dt);
ANALYZE reopens_to_sessions_perc;


-- $by_action
CREATE TEMPORARY TABLE by_action
    ON COMMIT DROP
AS (
    SELECT country
         , city
         , corp_client_id
         , sector
         , date_trunc('hour', created)      AS creation_dt
         , 'overall_without_ml'             AS metric_name
         , SUM(overall_without_ml)::NUMERIC AS metric_value
         , NULL::BIGINT                     AS sub_metric_value
    FROM (SELECT country
               , city
               , corp_client_id
               , created
               , sector
               , CASE
                     WHEN (meta_action IN ('create', 'new_message', 'appear', 'reopen') AND
                           (NOT ((LEAD(meta_action) OVER w = 'forward' AND LEAD(login) OVER w = 'superuser') OR
                                 'ar_reply' = ANY (LEAD(added_tags) OVER w))
                               OR LEAD(meta_action) OVER w IS NULL)
                         AND LEAD(meta_action) OVER w NOT IN ('proactive_support', 'proactive_message')
                         )
                         THEN 1
                     ELSE 0
            END AS overall_without_ml
          FROM chatterbox_history
          WHERE meta_action IN
                ('create', 'new_message', 'appear', 'reopen', 'reopen_after_dismiss', 'forward', 'close', 'comment',
                 'communicate')
              WINDOW w AS (PARTITION BY ticket_id ORDER BY event_number)
         ) t
    GROUP BY country, city, corp_client_id, sector, date_trunc('hour', created)
    UNION ALL
    SELECT country
         , city
         , corp_client_id
         , sector
         , date_trunc('hour', created) AS creation_dt
         , 'close'::VARCHAR            AS metric_name
         , COUNT(*)::NUMERIC           AS metric_value
         , NULL::BIGINT                AS sub_metric_value
    FROM chatterbox_history
    WHERE login != 'superuser'
      AND meta_action = 'close'
    GROUP BY country, city, corp_client_id, sector, date_trunc('hour', created)
    UNION ALL
    SELECT country
         , city
         , corp_client_id
         , sector
         , date_trunc('hour', created) AS creation_dt
         , 'dismiss'::VARCHAR          AS metric_name
         , COUNT(*)::NUMERIC           AS metric_value
         , NULL::BIGINT                AS sub_metric_value
    FROM chatterbox_history
    WHERE login != 'superuser'
      AND meta_action = 'dismiss'
    GROUP BY country, city, corp_client_id, sector, date_trunc('hour', created)
    UNION ALL
    SELECT country
         , city
         , corp_client_id
         , sector
         , date_trunc('hour', created) AS creation_dt
         , 'close_dismiss'::VARCHAR    AS metric_name
         , COUNT(*)::NUMERIC           AS metric_value
         , NULL::BIGINT                AS sub_metric_value
    FROM chatterbox_history
    WHERE login != 'superuser'
      AND meta_action IN ('close', 'dismiss')
    GROUP BY country, city, corp_client_id, sector, date_trunc('hour', created)
) DISTRIBUTED BY (country, city, corp_client_id, sector, creation_dt);
ANALYZE by_action;


-- $dates_days
CREATE TEMPORARY TABLE dates_days
    ON COMMIT DROP
AS (
    SELECT i::DATE AS dates
    FROM generate_series((SELECT start_date FROM start_date), (SELECT end_date FROM end_date), INTERVAL '1 day') i
) DISTRIBUTED RANDOMLY;
ANALYZE dates_days;


-- $pre_pre_backlog
CREATE TEMPORARY TABLE pre_pre_backlog
    ON COMMIT DROP
AS (
    SELECT date_trunc('day', b.dates - INTERVAL '1 day')   AS dates
         , a.ticket_id                                     AS ticket_id
         , a.country                                       AS country
         , a.city                                          AS city
         , a.corp_client_id                                AS corp_client_id
         , a.sector                                        AS sector
         , (b.dates - creation_dttm) / INTERVAL '1 second' AS lifetime_no_sla
    FROM first_reply AS a
             CROSS JOIN dates_days AS b
    WHERE b.dates <= COALESCE(a.close_dttm, now())
      AND a.created < b.dates
      AND (b.dates - creation_dttm) / INTERVAL '1 second' < COALESCE(first_reply, 100000000000000000)
      AND (b.dates - creation_dttm) / INTERVAL '1 second' > sla_good
) DISTRIBUTED BY (ticket_id, sector, dates);
ANALYZE pre_pre_backlog;


CREATE TEMPORARY TABLE total_res_1
    ON COMMIT DROP
AS (
    SELECT country
         , city
         , corp_client_id
         , sector
         , creation_dt
         , metric_name
         , metric_value
         , sub_metric_value
    FROM first_reply__total
    UNION ALL
    SELECT country
         , city
         , corp_client_id
         , sector
         , creation_dt
         , metric_name
         , metric_value
         , sub_metric_value
    FROM first_reply_no_ml__total
    UNION ALL
    SELECT country
         , city
         , corp_client_id
         , sector
         , creation_dt
         , metric_name
         , metric_value
         , sub_metric_value
    FROM first_reply_no_communicate__total
    UNION ALL
    SELECT country
         , city
         , corp_client_id
         , sector
         , creation_dt
         , metric_name
         , metric_value
         , sub_metric_value
    FROM extremely_first_reply__total
    UNION ALL
    SELECT country
         , city
         , corp_client_id
         , sector
         , creation_dt
         , metric_name
         , metric_value
         , sub_metric_value
    FROM csat
    UNION ALL
    SELECT country
         , city
         , corp_client_id
         , sector
         , creation_dt
         , metric_name
         , metric_value
         , sub_metric_value
    FROM by_action
    UNION ALL
    SELECT country
         , city
         , corp_client_id
         , sector
         , creation_dt
         , metric_name
         , metric_value
         , sub_metric_value
    FROM inbox_tickets
    UNION ALL
    SELECT country
         , city
         , corp_client_id
         , sector
         , creation_dt
         , metric_name
         , metric_value
         , sub_metric_value
    FROM unique_contact_rate_perc
    UNION ALL
    SELECT country
         , city
         , corp_client_id
         , sector
         , creation_dt
         , metric_name
         , metric_value
         , sub_metric_value
    FROM contact_rate_perc
    UNION ALL
    SELECT country
         , city
         , corp_client_id
         , sector
         , creation_dt
         , metric_name
         , metric_value
         , sub_metric_value
    FROM reopens
    UNION ALL
    SELECT country
         , city
         , corp_client_id
         , sector
         , creation_dt
         , metric_name
         , metric_value
         , sub_metric_value
    FROM reopens_without_waiting
    UNION ALL
    SELECT country
         , city
         , corp_client_id
         , sector
         , creation_dt
         , metric_name
         , metric_value
         , sub_metric_value
    FROM reopens_to_sessions_perc
) DISTRIBUTED BY (country, city, corp_client_id, sector, creation_dt, metric_name);
ANALYZE total_res_1;


truncate table snb_delivery.support_dm_sector_board;
insert into snb_delivery.support_dm_sector_board
(_etl_processed_dttm, country, city, corp_client_id, sector_name, utc_attributed_dttm, metric_name, metric_value,
 sub_metric_value)
SELECT now()                 AS _etl_processed_dttm
     , country               AS country
     , city                  AS city
     , corp_client_id        AS corp_client_id
     , sector                AS sector_name
     , creation_dt           AS utc_attributed_dttm
     , metric_name           AS metric_name
     , metric_value::numeric AS metric_value
     , sub_metric_value      AS sub_metric_value
FROM total_res_1 AS a
WHERE sector in (select sector_code from logistics_sector_meta_information)
    or sector = 'orders';


-- top_tickets ------------------------------------------------------------------


-- $today
CREATE TEMPORARY TABLE today
    ON COMMIT DROP
AS (
    SELECT date_trunc('day', current_datetime) AS today
    FROM current_datetime
) DISTRIBUTED RANDOMLY;
ANALYZE today;


-- $top_csat_5
CREATE TEMPORARY TABLE top_csat_5
    ON COMMIT DROP
AS (
    SELECT country
         , city
         , corp_client_id
         , sector      AS sector_name
         , creation_dt AS utc_created_dttm
         , metric_name
         , metric_value
         , ticket_id
    FROM (
             SELECT country
                  , city
                  , corp_client_id
                  , sector
                  , creation_dt
                  , 'csat_5'::VARCHAR   AS metric_name
                  , csat                AS metric_value
                  , ticket_id
                  , ROW_NUMBER() OVER w AS row_num
             FROM (
                      SELECT country
                           , city
                           , corp_client_id
                           , sector
                           , date_trunc('hour', created) AS creation_dt
                           , ticket_id
                           , MAX(csat)                   AS csat
                      FROM csat_per_ticket
                      WHERE csat IS NOT NULL
                        AND csat <= 3
                      GROUP BY 1, 2, 3, 4, 5, 6
                  ) S
                 WINDOW w AS (PARTITION BY sector, creation_dt
                     ORDER BY csat DESC)
         ) t
    WHERE row_num <= 50::bigint
) DISTRIBUTED BY (ticket_id, sector_name, utc_created_dttm);
ANALYZE top_csat_5;


-- $top_first_reply_backlog_by_created_dttm
CREATE TEMPORARY TABLE top_first_reply_backlog_by_created_dttm
    ON COMMIT DROP
AS (
    SELECT country
         , city
         , corp_client_id
         , sector      AS sector_name
         , creation_dt AS utc_created_dttm
         , metric_name
         , metric_value
         , ticket_id
    FROM (
             SELECT country
                  , city
                  , corp_client_id
                  , sector
                  , creation_dt
                  , 'first_reply_backlog_by_created_dttm'::VARCHAR AS metric_name
                  , first_reply                                    AS metric_value
                  , ticket_id
                  , ROW_NUMBER() OVER w                            AS row_num
             FROM (
                      SELECT country
                           , city
                           , corp_client_id
                           , ticket_id
                           , sector
                           , first_reply
                           , date_trunc('hour', created) AS creation_dt
                      FROM first_reply
                      WHERE first_reply IS NULL
                        AND NOT backlog_good_sla
                  ) S
                 WINDOW w AS (PARTITION BY sector, creation_dt
                     ORDER BY COALESCE(first_reply, 1000000) DESC)
         ) t
    WHERE row_num <= 50::bigint
) DISTRIBUTED BY (ticket_id, sector_name, utc_created_dttm);
ANALYZE top_first_reply_backlog_by_created_dttm;


-- $top_first_reply
CREATE TEMPORARY TABLE top_first_reply
    ON COMMIT DROP
AS (
    SELECT country
         , city
         , corp_client_id
         , sector      AS sector_name
         , creation_dt AS utc_created_dttm
         , metric_name
         , metric_value
         , ticket_id
    FROM (
             SELECT country
                  , city
                  , corp_client_id
                  , sector
                  , creation_dt
                  , 'first_reply'::VARCHAR AS metric_name
                  , first_reply            AS metric_value
                  , ticket_id
                  , ROW_NUMBER() OVER w    AS row_num
             FROM (
                      SELECT distinct country
                                    , city
                                    , corp_client_id
                                    , ticket_id
                                    , sector
                                    , first_reply
                                    , close_dttm::date AS creation_dt
                      FROM first_reply
                      WHERE close_dttm BETWEEN (SELECT start_date FROM start_date) AND (SELECT end_date FROM end_date)
                        AND ((first_reply != -1 AND first_reply > 0) OR first_reply IS NULL)
                  ) tt
                 WINDOW w AS (PARTITION BY sector, creation_dt
                     ORDER BY first_reply DESC)
         ) t
    WHERE row_num <= 50::bigint
) DISTRIBUTED BY (ticket_id, sector_name, utc_created_dttm);
ANALYZE top_first_reply;


-- $top_extremely_first_reply
CREATE TEMPORARY TABLE top_extremely_first_reply
    ON COMMIT DROP
AS (
    SELECT country
         , city
         , corp_client_id
         , sector      AS sector_name
         , creation_dt AS utc_created_dttm
         , metric_name
         , metric_value
         , ticket_id
    FROM (
             SELECT country
                  , city
                  , corp_client_id
                  , sector
                  , creation_dt
                  , 'extremely_first_reply'::VARCHAR AS metric_name
                  , extremely_first_reply            AS metric_value
                  , ticket_id
                  , ROW_NUMBER() OVER w              AS row_num
             FROM (
                      SELECT distinct country
                                    , city
                                    , corp_client_id
                                    , ticket_id
                                    , sector
                                    , extremely_first_reply
                                    , close_dttm::date AS creation_dt
                      FROM extremely_first_reply
                      WHERE close_dttm BETWEEN (SELECT start_date FROM start_date) AND (SELECT end_date FROM end_date)
                        AND (
                              (extremely_first_reply != -1 AND extremely_first_reply > 0) OR
                              extremely_first_reply IS NULL
                          )
                  ) tt
                 WINDOW w AS (PARTITION BY sector, creation_dt
                     ORDER BY extremely_first_reply DESC)
         ) t
    WHERE row_num <= 50::bigint
) DISTRIBUTED BY (ticket_id, sector_name, utc_created_dttm);
ANALYZE top_extremely_first_reply;


-- $top_sector_sla
CREATE TEMPORARY TABLE top_sector_sla
    ON COMMIT DROP
AS (
    SELECT country
         , city
         , corp_client_id
         , sector      AS sector_name
         , creation_dt AS utc_created_dttm
         , metric_name
         , metric_value
         , ticket_id
    FROM (
             SELECT country
                  , city
                  , corp_client_id
                  , sector
                  , creation_dt
                  , 'sector_sla'::VARCHAR AS metric_name
                  , sector_sla            AS metric_value
                  , ticket_id
                  , ROW_NUMBER() OVER w   AS row_num
             FROM (
                      SELECT distinct country
                                    , city
                                    , corp_client_id
                                    , ticket_id
                                    , sector
                                    , sector_sla
                                    , close_dttm::date AS creation_dt
                      FROM sector_sla
                      WHERE close_dttm BETWEEN (SELECT start_date FROM start_date) AND (SELECT end_date FROM end_date)
                        AND (
                          (sector_sla != -1 AND sector_sla > 0) OR sector_sla IS NULL
                          )
                  ) tt
                 WINDOW w AS (PARTITION BY sector, creation_dt
                     ORDER BY sector_sla DESC)
         ) t
    WHERE row_num <= 50::bigint
) DISTRIBUTED BY (ticket_id, sector_name, utc_created_dttm);
ANALYZE top_sector_sla;


-- $top_reaction_time
CREATE TEMPORARY TABLE top_reaction_time
    ON COMMIT DROP
AS (
    SELECT country
         , city
         , corp_client_id
         , sector      AS sector_name
         , creation_dt AS utc_created_dttm
         , metric_name
         , metric_value
         , ticket_id
    FROM (
             SELECT country
                  , city
                  , corp_client_id
                  , sector
                  , creation_dt
                  , 'reaction_time'::VARCHAR AS metric_name
                  , reaction_time            AS metric_value
                  , ticket_id
                  , ROW_NUMBER() OVER w      AS row_num
             FROM (
                      SELECT country
                           , city
                           , corp_client_id
                           , ticket_id
                           , sector
                           , close_dttm::date   AS creation_dt
                           , max(reaction_time) AS reaction_time
                      FROM reaction_time
                      WHERE close_dttm BETWEEN (SELECT start_date FROM start_date) AND (SELECT end_date FROM end_date)
                        AND (
                          (reaction_time != -1 AND reaction_time > 0) OR reaction_time IS NULL
                          )
                      GROUP BY country
                             , city
                             , corp_client_id
                             , ticket_id
                             , sector
                             , close_dttm::date
                  ) tt
                 WINDOW w AS (PARTITION BY sector, creation_dt
                     ORDER BY reaction_time DESC)
         ) t
    WHERE row_num <= 50::bigint
      AND metric_value IS NOT NULL
) DISTRIBUTED BY (ticket_id, sector_name, utc_created_dttm);
ANALYZE top_reaction_time;


-- $sector_old_ticket
CREATE TEMPORARY TABLE sector_old_ticket
    ON COMMIT DROP
AS (
    SELECT country
         , city
         , corp_client_id
         , created
         , ticket_id
         , sector
         , meta_action
         , login
         , CASE
               WHEN meta_action = 'appear'
                   THEN (created - FIRST_VALUE(created) OVER w) / INTERVAL '1 second'
        END AS sector_old
    FROM chatterbox_history
    WHERE meta_action IN ('create', 'appear')
        WINDOW w AS (PARTITION BY ticket_id ORDER BY event_number)
) DISTRIBUTED BY (ticket_id, sector, created);
ANALYZE sector_old_ticket;


-- $top_sector_old
CREATE TEMPORARY TABLE top_sector_old
    ON COMMIT DROP
AS (
    SELECT country
         , city
         , corp_client_id
         , sector      AS sector_name
         , creation_dt AS utc_created_dttm
         , metric_name
         , metric_value
         , ticket_id
    FROM (
             SELECT country
                  , city
                  , corp_client_id
                  , sector
                  , creation_dt
                  , 'sector_old'::VARCHAR AS metric_name
                  , sector_old            AS metric_value
                  , ticket_id
                  , ROW_NUMBER() OVER w   AS row_num
             FROM (
                      SELECT country
                           , city
                           , corp_client_id
                           , ticket_id
                           , sector
                           , date_trunc('day', created) AS creation_dt
                           , max(sector_old)            AS sector_old
                      FROM sector_old_ticket
                      WHERE meta_action = 'appear'
                      GROUP BY country
                             , city
                             , corp_client_id
                             , ticket_id
                             , sector
                             , date_trunc('day', created)
                  ) tt
             WHERE sector_old != 0
                 WINDOW w AS (PARTITION BY sector
                     , creation_dt ORDER BY sector_old DESC)
         ) t
    WHERE row_num <= 50::bigint
      AND creation_dt BETWEEN (SELECT start_date FROM start_date)
        AND (SELECT end_date FROM end_date)
) DISTRIBUTED BY (ticket_id, sector_name, utc_created_dttm);
ANALYZE top_sector_old;


-- $defer_time
CREATE TEMPORARY TABLE defer_time
    ON COMMIT DROP
AS (
    SELECT created
         , last_value(created) OVER w AS ticket_created
         , ticket_id
         , country
         , city
         , corp_client_id
         , sector
         , meta_action
         , CASE
               WHEN meta_action = 'defer'
                   AND lead(meta_action) OVER w IN ('temp_answer', 'communicate_ml', 'close', 'tech_close', 'close_ml',
                                                    'call_success', 'call_fail', 'reopen', 'dismiss', 'communicate',
                                                    'defer_with_answer', 'new_message',
                                                    'finish_archive', 'comment_ml', 'close_backlog',
                                                    'reopen_after_defer', 'tech_reopen', 'extra_message',
                                                    'comment')
                   THEN (lead(created) OVER w - created) / INTERVAL '1 second'
               WHEN meta_action = 'defer' AND lead(meta_action) OVER w IS NULL
                   THEN ((SELECT today FROM today) - created) / INTERVAL '1 second'
               ELSE NULL
        END                           AS defer_time
    FROM chatterbox_history
    WHERE created < (SELECT today FROM today)
      AND meta_action NOT IN ('finish_routing', 'finish_export', 'take', 'switch_task', 'external_response', 'forward',
                              'manual_update_meta', 'skip', 'hidden_comment', 'enqueue_routing', 'ensure_predispatched',
                              'appear', 'assign',
                              'waiting_zendesk', 'forward_task', 'fail_autoreply', 'status_change', 'take_another_task',
                              'macro_processing_fail', 'offer', 'generate_extra_ticket_link', 'external_request',
                              'forward_messages',
                              'predispatch')
        WINDOW w AS (PARTITION BY ticket_id
            ORDER BY event_number
            ROWS BETWEEN UNBOUNDED PRECEDING AND UNBOUNDED FOLLOWING)
) DISTRIBUTED BY (ticket_id, sector, created);
ANALYZE defer_time;


-- $top_defer_time
CREATE TEMPORARY TABLE top_defer_time
    ON COMMIT DROP
AS (
    SELECT country
         , city
         , corp_client_id
         , sector      AS sector_name
         , creation_dt AS utc_created_dttm
         , metric_name
         , metric_value
         , ticket_id
    FROM (
             SELECT country
                  , city
                  , corp_client_id
                  , ticket_id
                  , sector
                  , defer_time            AS metric_value
                  , 'defer_time'::VARCHAR AS metric_name
                  , ticket_created        AS creation_dt
                  , ROW_NUMBER() OVER w   AS row_num
             FROM (
                      SELECT country
                           , city
                           , corp_client_id
                           , ticket_id
                           , sector
                           , date_trunc('day', ticket_created) AS ticket_created
                           , SUM(defer_time)::NUMERIC          AS defer_time
                      FROM defer_time
                      WHERE defer_time IS NOT NULL
                      GROUP BY country
                             , city
                             , corp_client_id
                             , ticket_id
                             , sector
                             , date_trunc('day', ticket_created)
                  ) tt
                 WINDOW w AS (PARTITION BY sector, ticket_created
                     ORDER BY defer_time DESC)
         ) t
    WHERE row_num <= 50::bigint
      AND metric_value IS NOT NULL
      AND creation_dt BETWEEN (SELECT start_date FROM start_date)
        AND (SELECT end_date FROM end_date)
) DISTRIBUTED BY (ticket_id, sector_name, utc_created_dttm);
ANALYZE top_defer_time;


-- $top_sla_reply_null_all
CREATE TEMPORARY TABLE top_sla_reply_null_all
    ON COMMIT DROP
AS (
    SELECT country
         , city
         , corp_client_id
         , sector          AS sector_name
         , creation_dt     AS utc_created_dttm
         , metric_name
         , lifetime_no_sla AS metric_value
         , ticket_id
    FROM (
             SELECT country
                  , city
                  , corp_client_id
                  , sector
                  , 'lifetime_no_sla'::VARCHAR AS metric_name
                  , ticket_id
                  , lifetime_no_sla
                  , dates                      AS creation_dt
                  , ROW_NUMBER() OVER w        AS row_num
             FROM pre_pre_backlog
                 WINDOW w AS (
                     PARTITION BY dates, sector
                     ORDER BY lifetime_no_sla DESC)
         ) t
    WHERE row_num <= 50::bigint
) DISTRIBUTED BY (ticket_id, sector_name, utc_created_dttm);
ANALYZE top_sla_reply_null_all;


CREATE TEMPORARY TABLE total_res
    ON COMMIT DROP
AS (
    SELECT *
    FROM top_first_reply
    UNION ALL
    SELECT *
    FROM top_extremely_first_reply
    UNION ALL
    SELECT *
    FROM top_sector_sla
    UNION ALL
    SELECT *
    FROM top_reaction_time
    UNION ALL
    SELECT *
    FROM top_sector_old
    UNION ALL
    SELECT *
    FROM top_defer_time
    UNION ALL
    SELECT *
    FROM top_sla_reply_null_all
    UNION ALL
    SELECT *
    FROM top_csat_5
    UNION ALL
    SELECT *
    FROM top_first_reply_backlog_by_created_dttm
) DISTRIBUTED BY (ticket_id, sector_name, utc_created_dttm, metric_name);
ANALYZE total_res;


CREATE TEMPORARY TABLE result_table
    ON COMMIT DROP AS (
    SELECT country
         , city
         , corp_client_id
         , sector_name
         , utc_created_dttm AS utc_attributed_dttm
         , metric_name
         , metric_value
         , ticket_id
    FROM total_res
) DISTRIBUTED BY (ticket_id, country, city, corp_client_id, sector_name, utc_attributed_dttm, metric_name);
ANALYZE result_table;


truncate table snb_delivery.support_dm_top_ticket_board;
insert into snb_delivery.support_dm_top_ticket_board
(_etl_processed_dttm, country, city, corp_client_id, sector_name, ticket_id, metric_name, metric_value,
 utc_attributed_dttm)
SELECT now()               AS _etl_processed_dttm
     , country             AS country
     , city                AS city
     , corp_client_id      AS corp_client_id
     , sector_name         AS sector_name
     , ticket_id           AS ticket_id
     , metric_name         AS metric_name
     , metric_value        AS metric_value
     , utc_attributed_dttm AS utc_attributed_dttm
FROM result_table
WHERE sector_name in (select sector_code from logistics_sector_meta_information);