import os
import random
import requests
import json
from datetime import datetime
from requests.adapters import HTTPAdapter
from urllib3.util.retry import Retry

import yt.wrapper as yt
import ticket_parser2_py3.api.v1.exceptions as tp2_exceptions
import ticket_parser2_py3.api.v1 as tp2


class TvmRequestError(Exception):
    """Error during performing request to tvm."""


class TVMClient:
    def __init__(self, tvm_secret: str, src_id: int, dsts: dict):
        self._tvm_client = tp2.TvmClient(
            tp2.TvmApiClientSettings(
                self_client_id=src_id,
                enable_service_ticket_checking=True,
                enable_user_ticket_checking=tp2.BlackboxEnv.Test,
                self_secret=tvm_secret,
                dsts=dsts,
            ),
        )

    def get_ticket(self, dst_service_name):
        try:
            return self._tvm_client.get_service_ticket_for(dst_service_name)
        except tp2_exceptions.TvmException as ex:
            raise TvmRequestError(
                f'cannot perform request to {dst_service_name}: {ex}',
            )


def create_session():
    session = requests.Session()
    tvm = TVMClient(
        tvm_secret=os.environ.get('TVM_SECRET'),
        src_id=2027260,
        dsts={'tvmtaxidispatchsettingsstable': 2016001},
    )
    tvm_ticket = tvm.get_ticket(
        dst_service_name='tvmtaxidispatchsettingsstable',
    )
    session.headers = {'X-Ya-Service-Ticket': tvm_ticket}
    retry = Retry(
        total=5,
        backoff_factor=1,
        status_forcelist=(500, 502, 503, 504),
        allowed_methods=('POST', 'GET'),
    )
    session.mount('https://', HTTPAdapter(max_retries=retry))
    return session


def get_current_settings(session, params):
    response = session.get(
        'http://dispatch-settings.taxi.yandex.net/v1/admin/fetch',
        params=params,
    )
    return response.json()


def extract_version(response):
    for x in response['settings'][0]['parameters']:
        if x['name'] == 'MAX_ROBOT_DISTANCE':
            return x['version']


def change_settings(session, params, radius):
    old_settings = get_current_settings(session, params)

    print('== GOT OLD SETTINGS ==')

    old_version = extract_version(old_settings)

    print(f'== GOT OLD VERSION: {old_version} ==')

    data = {
        'settings': [
            {
                'zone_name': params['zone_name'],
                'tariff_name': params['tariff_name'],
                'type': params['type'],
                'parameters': [
                    {
                        'name': 'MAX_ROBOT_DISTANCE',
                        'old_version': old_version,
                        'value': radius,
                    },
                ],
            },
        ],
    }

    change_response = session.post(
        'http://dispatch-settings.taxi.yandex.net/v1/admin/set',
        data=json.dumps(data),
    )

    change_response.raise_for_status()
    
    change_dttm = datetime.now().strftime('%Y-%m-%d %H:%M:%S')

    return change_dttm


def save_results(radius, change_dttm, new_version):
    yt.config["proxy"]["url"] = "hahn.yt.yandex.net"
    yt.write_table(
        yt.TablePath(YT_TABLE_PATH, append=True),
        [
            {
                'change_dttm': change_dttm,
                'radius': radius,
                'new_version': new_version,
                'etl_process_dttm': datetime.now().strftime(
                    '%Y-%m-%d %H:%M:%S',
                )
            },
        ],
    )


if __name__ == '__main__':
    YT_TABLE_PATH = '//home/taxi-delivery/analytics/production/switchback/cargo_free_radius/test_na_prode'

    CARGO_PARAMS = {
        'zone_name': 'moscow',
        'tariff_name': 'cargo',
        'type': 'common_dispatch',
    }

    RADIUS_OPTIONS = [15000, 14999]

    new_session = create_session()

    print('== SESSION CREATED ==')

    new_radius = random.choice(RADIUS_OPTIONS)

    print(f'== NEW RADIUS: {new_radius} ==')

    change_dttm = change_settings(new_session, CARGO_PARAMS, new_radius)

    print(f'== RADIUS CHANGED AT {change_dttm} ==')
    
    new_settings = get_current_settings(new_session, CARGO_PARAMS)
    new_version = extract_version(new_settings)
    
    print('== GOT NEW VERSION ==')

    save_results(new_radius, change_dttm, new_version)

    print('== RESULTS SAVED ==')
