def required_fields(*fields):
    def decorator(method):
        def func(self, *args, **kwargs):
            for field in fields:
                if not hasattr(self, field) or getattr(self, field) is None:
                    raise ValueError(
                        'There is no required field: "{}"'.format(field),
                    )
            return method(self, *args, **kwargs)

        return func

    return decorator


def cached_method(method):
    """
    caches the arguments of the call and the result in the self._cache
    """

    def func(self, *args, **kwargs):
        key = (method.__name__, repr(args), repr(kwargs))
        result = self._cache.get(key)
        if result is None:
            result = method(self, *args, **kwargs)
            self._cache[key] = result
        return result

    return func


def cached_property(property_method):
    """
    caches the property value in the self._cache
    """

    def func(self):
        result = self._cache.get(property_method.__name__)
        if result is None:
            result = property_method(self)
            self._cache[property_method.__name__] = result
        return result

    return property(func)